/* Spherocylinder fluid builder. This program generates site positions and computational unit cell
 vectors for a fluid with a specified density and composition, in any number of dimensions.

 Command-line input: name of input parameter file (param_file)

 Output: configuration file containing computational unit cell matrix and site positions */

//#define MAIN

#include "bob.h"
#include "graphics.h"

//#undef MAIN

int reverse_compare_double(const void *x, const void *y) {
    const double *dx = (const double *) x;
    const double *dy = (const double *) y;

    return -(*dx > *dy) + (*dx < *dy);
}

double mean_func(double x, double measured_mean, double min, double max) {

    double e_min = exp(min / x);
    double e_max = exp(max / x);
    double num = (min - max) * e_min;
    double denom = e_max - e_min;

    return num / denom - measured_mean + min + x;
}

double calc_trunc_exp_mean(double expectation, double min_length, double max_length) {

    int n = 1;
    int n_max = 1000000, success = 0;
    double mid;
    double lower = min_length;
    double upper = max_length;
    double tol = 1e-5;
    while (n < n_max) {
        mid = 0.5 * (upper + lower);

        double value_mid = mean_func(mid, expectation, min_length, max_length);
        double value_lower = mean_func(lower, expectation, min_length, max_length);
        if (value_mid == 0.0 || 0.5 * (upper - lower) < tol) {
            success = 1;
            break;
        }

        if (SIGN(1,value_mid) == SIGN(1,value_lower) )
            lower = mid;
        else
            upper = mid;
        n++;
    }

    if (!success) {
        printf("failed to find effective mean within min_length and max_length with desired value of sphero_length\n");
        exit(1);
    }
    else {
        printf("drawing from exponential with mean %g\n", mid);
        if (ABS(1.0 - mid/expectation) > 0.2) {
            printf("Probability distribution mean deviates more than 20%% of expected mean, consider using different min ");
            printf("and max lengths\n");
        }
        return mid;
    }
}

int main(int argc, char *argv[])
{
    char param_file[F_MAX], *color_file;
    system_parameters parameters;
    system_properties properties;
    long seed;
    int n_dim, n_periodic, n_walls, n_spheros, n_sites, n_bonds, n_rgb, order_type, sphere_flag,
        max_trials, graph_flag, i, j, i_site, i_bond, *site_1, *site_2;
    double diameter, sphero_length, side_length,
        **h, **h_inv,
        **r, **s, **r_bond, **s_bond, **u_bond, **v_bond, *length, *length2;

    /* Get parameters from command-line input. */
    if (argc != 2) {
        printf("Usage: %s param_file\n", argv[0]);
        error_exit("Wrong number of command-line arguments in main");
    }
    strcpy(param_file, argv[1]);

    /* Read in input parameters. */
    parse_parameters(param_file, &parameters);

    /* Set up shortcuts to data structures. */
    n_dim = parameters.n_dim;
    n_walls = parameters.n_walls;
    if (n_walls >= 0)
        n_periodic = parameters.n_periodic = n_dim - n_walls;
    else{
        n_periodic = parameters.n_periodic;
        n_walls = parameters.n_walls = n_dim-n_periodic;
    }

    n_spheros = parameters.n_spheros;
    diameter = parameters.diameter = 1.0;
    sphero_length = parameters.sphero_length;
    side_length = parameters.side_length;
    n_sites = properties.sites.n_sites = 2 * n_spheros;
    n_bonds = properties.bonds.n_bonds = n_spheros;
    seed = parameters.seed;
    max_trials = parameters.max_trials;
    graph_flag = parameters.graph_flag;
    color_file = parameters.color_file;
    order_type = parameters.order_type;
    sphere_flag = parameters.sphere_flag;

    /* Initialize unit cell matrix. A hypercubic volume is assumed. */
    h = properties.unit_cell.h = (double**) allocate_2d_array(n_dim, n_dim, sizeof(double));
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j)
            h[i][j] = 0.0;
    for (i = 0; i < n_dim; ++i)
        h[i][i] = side_length;

    /* Initialize unit cell structure. */
    init_unit_cell_structure(&parameters, &properties);
    h_inv = properties.unit_cell.h_inv;

    /* Allocate memory for site and bond structures. */
    r = properties.sites.r = (double**) allocate_2d_array(n_sites, n_dim, sizeof(double));
    s = properties.sites.s = (double**) allocate_2d_array(n_sites, n_dim, sizeof(double));
    properties.sites.v = (double**) allocate_2d_array(n_sites, n_dim, sizeof(double));
    r_bond = properties.bonds.r_bond = (double**) allocate_2d_array(n_bonds, n_dim, sizeof(double));
    s_bond = properties.bonds.s_bond = (double**) allocate_2d_array(n_bonds, n_dim, sizeof(double));
    u_bond = properties.bonds.u_bond = (double**) allocate_2d_array(n_bonds, n_dim, sizeof(double));
    v_bond = properties.bonds.v_bond = (double**) allocate_2d_array(n_bonds, n_dim, sizeof(double));
    length = properties.bonds.length = (double*) allocate_1d_array(n_bonds, sizeof(double));
    length2 = properties.bonds.length2 = (double*) allocate_1d_array(n_bonds, sizeof(double));
    site_1 = properties.bonds.bond_site_1 = (int*) allocate_1d_array(n_bonds, sizeof(int));
    site_2 = properties.bonds.bond_site_2 = (int*) allocate_1d_array(n_bonds, sizeof(int));

    /* Initialize GSL random number generator state */
    gsl_rng_env_setup();
    properties.rng.T = gsl_rng_default;
    properties.rng.r = gsl_rng_alloc(properties.rng.T);
    gsl_rng_set(properties.rng.r, parameters.seed);

    /* Initialize structures. */
    int poly_dist_type = parameters.poly_dist_type;
    double min_length = parameters.min_length;
    double max_length = parameters.max_length;
    i_site = 0;
    /* Check for sanity in polydispersity */
    if (poly_dist_type) {
        if (parameters.min_length < 0) {
            fprintf(stderr, "minimum length not defined. please set min_length parameter. simulations with min_length < ~4 might be unstable or give poor results\n");
            exit(1);
        }
        if (parameters.max_length < 0) {
            fprintf(stderr, "maximum length not defined. please set max_length parameter. max length must be less than half minimum box dimension\n");
            exit(1);
        }
        else if (parameters.max_length > 0.5 * side_length) {
            fprintf(stderr, "maximum length greater than half box dimension. exiting\n");
            exit(1);
        }

    }

    /* Calculate distribution mean if using exponential distribution */
    if (poly_dist_type == 1)
        sphero_length = calc_trunc_exp_mean(sphero_length, min_length, max_length);

    for (i_bond = 0; i_bond < n_bonds; ++i_bond) {
        if (poly_dist_type == 0)
            length[i_bond] = sphero_length;
        else if (poly_dist_type == 1) {
            do {

                length[i_bond] = -sphero_length * log(gsl_rng_uniform(properties.rng.r));

            } while (length[i_bond] > max_length || length[i_bond] < min_length );
        }
        else if (poly_dist_type == 2) {
            double deltal = 2.0 * (sphero_length - min_length);

            if ((min_length + deltal) > 0.5 * side_length) {
                fprintf(stderr, "maximum length greater than half box dimension. (max_length = 2*sphero_length - min_length) for uniform polydispersity. exiting\n");
                exit(1);
            }

            length[i_bond] = min_length + deltal * gsl_rng_uniform(properties.rng.r);
        }
        else {
            printf("Selected poly_dist_type (%d) not supported, exiting\n", poly_dist_type);
            exit(1);
        }
        site_1[i_bond] = i_site;
        ++i_site;
        site_2[i_bond] = i_site;
        ++i_site;
    }

    /* Reverse sort array so long bonds are inserted first, increasing success rate */
    /* FIX ME: INCOMPATIBLE WITH NEW 3 STATE DYNAMIC INSTABILITY SYSTEM */
    if (poly_dist_type) {
        qsort(length, n_bonds, sizeof(double), reverse_compare_double);
    }

    /* FIXME: introducing new rates has made this more complicated */
    /* if (parameters.dynamic_instability_flag == 1) { */
    /*     double k_rescue = parameters.k_rescue; */
    /*     double k_catastrophe = parameters.k_catastrophe; */

    /*     double p_growing = k_rescue / k_catastrophe; */
    /*     p_growing = p_growing / (1 + p_growing); */

    /*     unsigned char *poly_state = (unsigned char*) allocate_1d_array(n_bonds, sizeof(unsigned char)); */
    /*     for (i_bond = 0; i_bond < n_bonds; ++i_bond) { */
    /*         poly_state[i_bond] = ran3(&seed) < p_growing; */
    /*     } */

    /*     FILE *f_poly = gfopen(parameters.poly_config, "w"); */
    /*     fwrite(poly_state, sizeof(unsigned char), n_bonds, f_poly); */
    /*     fclose(f_poly); */

    /*     free(poly_state); */
    /* } */

    /* Generate initial configuration by random insertion of spherocylinders in a hypercubic volume with
       periodic boundary conditions. */
    seed = -seed;
    if (order_type == 1)
        parallelepiped_nematic_insertion_sphero_periodic(&parameters, n_dim, n_periodic, h,
                                                         h_inv, n_bonds, diameter,
                                                         max_trials, properties.rng.r,
                                                         r, s, r_bond, s_bond, u_bond,
                                                         length, site_1, site_2,
                                                         sphere_flag);
    else if (order_type == 2)
        parallelepiped_square_insertion_sphero_periodic(&parameters, n_dim, n_periodic, h,
                                                        h_inv, n_bonds, diameter,
                                                        max_trials, properties.rng.r,
                                                        r, s, r_bond, s_bond, u_bond,
                                                        length, site_1, site_2);
    else if (order_type == 3)
        parallelepiped_bundle_insertion_sphero_periodic(&parameters, n_dim, n_periodic, h,
                                                        h_inv, n_bonds, diameter,
                                                        max_trials, properties.rng.r,
                                                        r, s, r_bond, s_bond, u_bond,
                                                        length, site_1, site_2);
    else
        parallelepiped_insertion_sphero_periodic(&parameters, n_dim, n_periodic, h,
                                                 h_inv, n_bonds, diameter,
                                                 max_trials, properties.rng.r,
                                                 r, s, r_bond, s_bond, u_bond,
                                                 length, site_1, site_2);

    if (order_type) {
        for (i_bond = 0; i_bond < n_bonds; ++i_bond)
            u_bond[i_bond][n_dim - 1] = -1.0;

        double polarity = -1.0;
        double n_down = n_bonds;
        while (polarity < parameters.target_polarity) {
            i_bond = (int) (n_bonds * gsl_rng_uniform(properties.rng.r));

            if (u_bond[i_bond][n_dim - 1] < 0) {
                u_bond[i_bond][n_dim - 1] = 1.0;
                n_down--;
                polarity = (n_bonds - 2.0*n_down) / n_bonds;
            }
        }
        printf("final polarity = %g\n", polarity);
        update_bond_site_positions(n_dim, n_periodic, n_bonds, n_sites, h,
                                   h_inv, site_1, site_2, r_bond,
                                   u_bond, length, r, s);

    }

    /* Update bond vectors. */
    update_bond_vectors(n_dim, n_periodic, n_bonds, h, s, r, site_1, site_2,
                        v_bond, u_bond, length, length2);

    /* if (order_type) */
    /*     for (i_bond = 0; i_bond < n_bonds; ++i_bond) { */
    /*         u_bond[i_bond][n_dim - 1] = 1.0; */
    /*         for (i = 0; i < n_dim; i++) { */
    /*             r[site_1[i_bond]][i] = -0.5 * length[i_bond] * u_bond[i_bond][i] + r_bond[i_bond][i]; */
    /*             r[site_2[i_bond]][i] = 0.5 * length[i_bond] * u_bond[i_bond][i] + r_bond[i_bond][i]; */
    /*         } */
    /*     } */

#ifndef NOGRAPH
    Graphics graphics;
    /* Activate graphics if graph_flag == 1. */
    if (graph_flag) {
        graphics.Init(&parameters, parameters.n_dim, properties.unit_cell.h, 0);
        graphics.ResizeWindow(800,800);
        graphics.SetBoundaryType("cube");
        graphics.DrawLoop(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.sites.n_sites,
                          properties.sites.r,
                          parameters.sphere_diameter);
        //graphics.DrawLoop(n_bonds, h, r_bond, u_bond, length);
    }
#endif

    /* Set up pointer to header write function. */
    properties.write_header_func = write_header_sphero;

    /* Write configuration to output file. */
    write_config(&parameters, &properties, "fluid_sphero.config");

    if (poly_dist_type) {
        double mean_l = 0.0;
        for (i_bond = 0; i_bond < n_bonds; ++i_bond) {
            mean_l += length[i_bond];
        }
        mean_l = mean_l / n_bonds;

        printf("Measured mean length: %g\n", mean_l);
    }

    /* Normal termination. */
    exit(0);
}
