#ifndef NOGRAPH
#include <GL/glew.h>
#include "graphics.h"
#include <cmath>
#include <cstdlib>
#include <stdarg.h>
#include "macros.h"
#include <cstdio>
#include <iostream>
#include "xlink_entry.h"
#include "xlink_management.h"
#include "kinetochore.h"
#include "anchor_list.h"
#include "spatial_helpers.h"
#include "triangle_mesh.h"

void cross_product(double *a, double *b, double *c, int n_dim);
double dot_product(int n_dim, double *a, double *b);
void error_exit(const char *error_msg, ...);

static void key_callback2d(GLFWwindow *window, int key, int scancode, int action, int mods) {
    // set the window pointer to the keyboard object
    // You must set this pointer first before key callbacks are called!
    Graphics *graphics =
        static_cast<Graphics*>(glfwGetWindowUserPointer(window));

    if (action == GLFW_PRESS) {
        graphics->key_is_pressed_ = true;
        graphics->key_being_held_ = key;
        switch (key) {
        case GLFW_KEY_ESCAPE:
            graphics->keep_going_ = 1;
            break;
        case GLFW_KEY_UP:
            graphics->yTrans_ += 0.5;
            break;
        case GLFW_KEY_DOWN:
            graphics->yTrans_ -= 0.5;
            break;
        case GLFW_KEY_RIGHT:
            graphics->xTrans_ += 0.5;
            break;
        case GLFW_KEY_LEFT:
            graphics->xTrans_ -= 0.5;
            break;
        case GLFW_KEY_Z:
            if (mods & GLFW_MOD_SHIFT)
                graphics->zAngle_ += 5.0;
            else
                graphics->zAngle_ -= 5.0;
            break;
        case GLFW_KEY_SPACE:
            graphics->xyzScale_ += 1.0 / 10.0;
            break;
        case GLFW_KEY_BACKSPACE:
            graphics->xyzScale_ -= 1.0 / 10.0;
            break;
        case GLFW_KEY_C:
            if (mods & GLFW_MOD_SHIFT)
                graphics->color_switch_ = (graphics->color_switch_ + 1) % 3;
            else {
                graphics->color_switch_ = (graphics->color_switch_ - 1);
                if (graphics->color_switch_ < 0)
                    graphics->color_switch_ = 2;
            }
            break;
        case GLFW_KEY_A:
            if (mods & GLFW_MOD_SHIFT) {
                graphics->alpha_ += 0.1;
                if (graphics->alpha_ > 1.0)
                    graphics->alpha_ = 1.0;
            }
            else {
                graphics->alpha_ -= 0.1;
                if (graphics->alpha_ < 0.0)
                    graphics->alpha_ = 0.0;
            }
            break;
        case GLFW_KEY_T:
            // graphics->draw_trajectory_flag_ =
            //     (graphics->draw_trajectory_flag_ + 1) % 2;
            break;
        default:
            break;
        }
    }

    if (action == GLFW_RELEASE)
        graphics->key_is_pressed_ = false;
}

static void key_callback3d(GLFWwindow *window, int key, int scancode, int action, int mods) {
    // set the window pointer to the keyboard object
    // You must set this pointer first before key callbacks are called!
    Graphics *graphics =
        static_cast<Graphics*>(glfwGetWindowUserPointer(window));

    if (action == GLFW_PRESS) {
        graphics->key_is_pressed_ = true;
        graphics->key_being_held_ = key;
        switch (key) {
        case GLFW_KEY_ESCAPE:
            graphics->keep_going_ = 1;
            break;
        case GLFW_KEY_UP:
            graphics->xAngle_ += 5.0;
            break;
        case GLFW_KEY_DOWN:
            graphics->xAngle_ -= 5.0;
            break;
        case GLFW_KEY_RIGHT:
            graphics->yAngle_ += 5.0;
            break;
        case GLFW_KEY_LEFT:
            graphics->yAngle_ -= 5.0;
            break;
        case GLFW_KEY_Z:
            if (mods & GLFW_MOD_SHIFT)
                graphics->zAngle_ -= 5.0;
            else
                graphics->zAngle_ += 5.0;
            break;
        case GLFW_KEY_C:
            if (mods & GLFW_MOD_SHIFT)
                graphics->cNear_ -= 0.5;
            else
                graphics->cNear_ += 0.5;
            break;
        case GLFW_KEY_V:
            if (mods & GLFW_MOD_SHIFT)
                graphics->cFar_ += 0.5;
            else
                graphics->cFar_ -= 0.5;
            break;
        case GLFW_KEY_J:
            if (mods & GLFW_MOD_SHIFT)
                graphics->cxAngle_ -= 5.0;
            else
                graphics->cxAngle_ += 5.0;
            break;
        case GLFW_KEY_K:
            if (mods & GLFW_MOD_SHIFT)
                graphics->cyAngle_ -= 5.0;
            else
                graphics->cyAngle_ += 5.0;
            break;
        case GLFW_KEY_SPACE:
            graphics->xyzScale_ += 1.0 / 10.0;
            break;
        case GLFW_KEY_BACKSPACE:
            graphics->xyzScale_ -= 1.0 / 10.0;
            break;
        case GLFW_KEY_M:
            graphics->color_switch_ = (graphics->color_switch_ + 1) % 3;
            break;
        case GLFW_KEY_R:
            graphics->xyzScale_ *= 0.5;
            break;
        case GLFW_KEY_T:
            graphics->xyzScale_ *= 2.0;
            break;
        case GLFW_KEY_Y:
            graphics->tomogram_view_ = graphics->tomogram_view_ + 1;
            if (graphics->tomogram_view_ > 4) {
                graphics->tomogram_view_ = 0;
            }
            break;
        default:
            break;
        }
    }

    if (action == GLFW_RELEASE)
        graphics->key_is_pressed_ = false;

    
}

static void mouse_button_callback(GLFWwindow *window, int button, int action, int mods) {

    Graphics *graphics =
        static_cast<Graphics*>(glfwGetWindowUserPointer(window));

    if (button == GLFW_MOUSE_BUTTON_1) {
        switch (action) {
        case GLFW_PRESS:
            graphics->mouse_is_pressed_ = true;
            break;
        case GLFW_RELEASE:
            graphics->mouse_is_pressed_ = false;
            graphics->mousex_ = 0.0;
            graphics->mousey_ = 0.0;
            break;   
        }

    }
        
}

void Graphics::KeyInteraction() {
    // Tell glfw where the graphics class is
    glfwSetWindowUserPointer(window_, static_cast<void*>(this));
    
    // Set the proper key callback, since 2d and 3d should have different key
    // controls. 
    GLfloat *xInc = NULL, *yInc = NULL;
    double mouse_scale = 0, key_scale = 0;
    if (n_dim_ == 2) {
        glfwSetKeyCallback(window_, key_callback2d);
        xInc = &xTrans_;
        yInc = &yTrans_;
        mouse_scale = 0.1;
        key_scale = 0.05;
    }
    else if (n_dim_ == 3) {
        glfwSetKeyCallback(window_, key_callback3d);
        xInc = &yAngle_;
        yInc = &xAngle_;
        mouse_scale = 1.0;
        key_scale = 0.5;
    }

    // Set mouse callback 
    glfwSetMouseButtonCallback(window_, mouse_button_callback);
   
    // Check window for key presses/mouse interaction
    glfwPollEvents();

    // If mouse if being dragged, calculate drag distance and change view accordingly
    if (mouse_is_pressed_) {
        if (mousex_ > 0.0 && mousey_ > 0.0) {
            double old_x = mousex_;
            double old_y = mousey_;
            glfwGetCursorPos(window_, &mousex_, &mousey_);
            *yInc -= mouse_scale * (mousey_ - old_y);
            *xInc += mouse_scale * (mousex_ - old_x);
        }
        else {
            glfwGetCursorPos(window_, &mousex_, &mousey_);
        }
        
    }

    // If Arrow key is being held, continuosly change view
    if (key_is_pressed_) {
        switch (key_being_held_) {
        case GLFW_KEY_UP:
            *yInc += key_scale;
            break;
        case GLFW_KEY_DOWN:
            *yInc -= key_scale;
            break;
        case GLFW_KEY_RIGHT:
            *xInc += key_scale;
            break;
        case GLFW_KEY_LEFT:
            *xInc -= key_scale;
            break;
        }
    }
    
    
    // Exit program if you close graphics window
    if (glfwWindowShouldClose(window_)) {
        glfwDestroyWindow(window_);
        glfwTerminate();
        exit(1);
    }
}

void Graphics::Init( system_parameters * params,int n_dim, double **h, int n_anchors, triangle_mesh *example_kinetochore) {
    params_ = params;
    n_anchors_ = n_anchors;
    n_dim_ = n_dim;
    keep_going_ = 0;
    mouse_is_pressed_ = false;
    mousex_ = 0.0;
    mousey_ = 0.0;
    std::vector<double> empty_vec;
    empty_vec.push_back(0.0);
    empty_vec.push_back(0.0);
    empty_vec.push_back(1.0);
    utilde_.push_back(empty_vec);
    first_call_quaternion_ = false;

    InitColormap();
    InitWindow(n_dim, h);
    InitDiscoRectangle();
    InitSpheroCylinder();
    InitKinetochoreMesh(example_kinetochore);

}

void Graphics::InitWindow(int n_dim, double **h) {
    if (n_dim == 2)
        Init2dWindow(h);
    else if (n_dim == 3)
        Init3dWindow(h);
}

void Graphics::Init2dWindow(double **h) {
    // Defaults for window view and coloring
    xyzScale_ = 0.95;
    xTrans_ = yTrans_ = 0.0; // x,y translation
    color_switch_ = 1; // Default 'nice' spherocylinder coloring
    alpha_ = 1.0; // transparency

    /* Compute size of the window respecting the aspect ratio. */
    if (h[0][0] > h[1][1]) {
        windx_ = 800;
        windy_ = (int) (windx_ * h[1][1] / h[0][0]);
    } else {
        windy_ = 800;
        windx_ = (int) (windy_ * h[0][0] / h[1][1]);
    }

    { // Make dummy window so we can get GL extensions 
        glfwWindowHint(GLFW_VISIBLE, GL_FALSE);
        if (!glfwInit())
            error_exit("Unable to initialize GLFW toolkit\n");
        
        window_ = glfwCreateWindow(windx_, windy_, "2D Sphero", NULL, NULL);
        
        if (!window_) {
            glfwTerminate();
            error_exit("Failed to create display window\n");
        }

        /* Wrangle in GL function pointers for GL version > 2.0 */
        glewInit();
        glfwDestroyWindow(window_);
    }

    /* Rebuild window now that we have proper information about available GL extensions */
    glfwWindowHint(GLFW_SAMPLES, 16);
    window_ = glfwCreateWindow(windx_, windy_, "2D Sphero", NULL, NULL);

    if (!window_) {
        glfwTerminate();
        error_exit("Failed to create display window\n");
    }

    glfwMakeContextCurrent(window_); 
    glfwSwapInterval(0); // swap buffers immediately
    glewInit(); // re-initialize glew for current window context

    // Blending and antialiasing. Makes things pretty.
    glEnable(GL_LINE_SMOOTH);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
    glEnable(GL_POLYGON_SMOOTH);
    glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
    glEnable(GL_MULTISAMPLE);

    /* frame buffer clears should be to black by default in 2d */
    background_color_[0] = background_color_[1] = background_color_[2] = 0.0;
    glClearColor(background_color_[0], background_color_[1], background_color_[2], 0.0);

    /* set up projection transform */
    double xmin = -h[0][0] / 2.;
    double xmax = h[0][0] / 2.;
    double ymin = -h[1][1] / 2.;
    double ymax = h[1][1] / 2.;
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluOrtho2D((GLdouble) (xmin - 0.05 * h[0][0]),
               (GLdouble) (xmax + 0.05 * h[0][0]),
               (GLdouble) (ymin - 0.05 * h[1][1]),
               (GLdouble) (ymax + 0.05 * h[1][1])); //deprecated, but annoying
                                                    //to implement

    /* establish initial viewport */
    glViewport(0, 0, windx_, windy_);
}

void Graphics::Init3dWindow(double **h) {
    xAngle_ = 90;
    yAngle_ = 0;
    zAngle_ = 90; 
    xyzScale_ = 1.05; // zoom in just a hair. 
    //xyzScale_ = 0.75; // FIXME: Zoom used for deconfined spindle simualtions
    zTrans_ = -24; // camera position in z
    color_switch_ = 0;

    // FIXME: Assumes a square box. Better way to calculate this, but we
    // typically use square boxes for now
    double a_perp_max = 0.0;
    for (int i = 0; i < 3; ++i)
        a_perp_max = MAX(h[i][i], a_perp_max);

    cNear_ = -2.0 * a_perp_max; // clipping plane near
    cFar_ = 2.0 * a_perp_max; // clipping plane far
    cxAngle_ = cyAngle_ = 0; // plane angles

    windx_ = windy_ = 800; // window size in pixels

     // Make dummy window so we can get GL extensions 
    glfwWindowHint(GLFW_VISIBLE, GL_FALSE); 
    if (!glfwInit())
        error_exit("Unable to initialize GLFW toolkit\n");
    
    window_ = glfwCreateWindow(windx_, windy_, "3D Sphero", NULL, NULL);
    
    if (!window_) {
        glfwTerminate();
        error_exit("Failed to create display window\n");
    }
    
    /* Wrangle in GL function pointers for GL version > 2.0 */
    glewInit();
    glfwDestroyWindow(window_);

    // Rebuild window now that we have proper information about the GL extensions
    glfwWindowHint(GLFW_SAMPLES, 16); // 16x Multisampling
    window_ = glfwCreateWindow(windx_, windy_, "3D Sphero", NULL, NULL);

    if (!window_) {
        glfwTerminate();
        error_exit("Failed to create display window\n");
    }

    glfwMakeContextCurrent(window_);
    glfwSwapInterval(0); // Swap buffers immediately
    glewInit(); // Get GL extensions for current context

    /* FIXME: this probably does nothing nowadays */
    /* Material properties. */
    GLfloat no_mat[] = { 0.0, 0.0, 0.0, 1.0 };
    GLfloat mat_diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat low_shininess[] = { 5.0 };
    glMaterialfv(GL_FRONT, GL_AMBIENT, no_mat);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
    glMaterialfv(GL_FRONT, GL_SHININESS, low_shininess);
    glMaterialfv(GL_FRONT, GL_EMISSION, no_mat);

    /* Lights. */
    GLfloat ambient[] = { 0.0, 0.0, 0.0, 1.0 };
    GLfloat diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat position[] = { 0.0, 2.0, 3.0, 0.0 };
    GLfloat lmodel_ambient[] = { 0.4, 0.4, 0.4, 1.0 };
    GLfloat local_view[] = { 0.0 };
    glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
    glLightfv(GL_LIGHT0, GL_POSITION, position);
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
    glLightModelfv(GL_LIGHT_MODEL_LOCAL_VIEWER, local_view);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    /* Used in order to change one material property (here the color). */
    glColorMaterial(GL_FRONT, GL_DIFFUSE);
    glEnable(GL_COLOR_MATERIAL);

    /* Blending and antialiasing. */
    glEnable(GL_LINE_SMOOTH);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glHint(GL_LINE_SMOOTH_HINT, GL_DONT_CARE);
    glEnable(GL_MULTISAMPLE);

    zTrans_ = -1.5 * h[2][2];
    glEnable(GL_DEPTH_TEST);    /* enable depth buffering */
    glDepthFunc(GL_LESS);       /* pedantic, GL_LESS is the default */
    glClearDepth(1.0);          /* pedantic, 1.0 is the default */

    /* frame buffer clears should be to white */
    glClearColor(1.0, 1.0, 1.0, 1.0); /* FIXME: Guess that we're drawing
                                         spherical confined system */
    /* set up projection transform */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(60.0, 1.0, 1.2 * h[2][2] / 2.0,
                   10.0 * 1.2 * h[2][2] / 2.0); // deprecated

    /* establish initial viewport */
    glViewport(0, 0, windx_, windy_);
}

void Graphics::InitColormap() { // Pretty straight forward. Builds
                                // colormap. Automatic for all instances of
                                // Graphics class after Init() call
    /* Loop over colormap indices. */
    for (int i_color = 0; i_color < n_rgb_; ++i_color) {
        GLfloat red, green, blue;
        if (i_color < 64) {
            red = 1.0;
            green = i_color / 64.0;
            blue = 0.0;
        } else if (i_color >= 64 && i_color < 128) {
            red = (128.0 - i_color) / 64.0;
            green = 1.0;
            blue = 0.0;
        } else if (i_color >= 128 && i_color < 192) {
            red = 0.0;
            green = 1.0;
            blue = (i_color - 128.0) / 64.0;
        } else if (i_color >= 192 && i_color < 256) {
            red = 0.0;
            green = (256.0 - i_color) / 64.0;
            blue = 1.0;
        } else if (i_color >= 256 && i_color < 320) {
            red = (i_color - 256.0) / 64.0;
            green = 0.0;
            blue = 1.0;
        } else {
            red = 1.0;
            green = 0.0;
            blue = (384.0 - i_color) / 64.0;
        }

        colormap_[i_color*4 + 0] = red;
        colormap_[i_color*4 + 1] = green;
        colormap_[i_color*4 + 2] = blue;
        colormap_[i_color*4 + 3] = 1.0;
    }
}

// Draw just connected (doubly bound) xlinks
void Graphics::DrawCrosslinks(int n_bonds, double **r, double **u, double *length,
                              int n_types, xlink_list **xlinks) {
    glUseProgram(0); // Use old fixed pipeline
    glMatrixMode(GL_MODELVIEW); // Make sure we're using the model transform
    
    /* Turn off the light. */
    glDisable(GL_LIGHTING);

    /* Select line width. */
    glLineWidth(3.0);

    /* Select line color. */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
        for (int i_type = 0; i_type < n_types; ++i_type) {
            
            for (xlink_list::iterator xlink = xlinks[i_type][i_bond].begin();
                 xlink < xlinks[i_type][i_bond].end();
                 xlink++) {
                /* Get the pointer to the actual data */
                if(xlink->IsActive()) {
                    GLfloat *color = xlink->GetColor();
                    glColor4fv(color);
                    GLfloat v[3] = { 0.0, 0.0, 0.0 };
                    /* Guaranteed to be active */
                    int bond_1 = xlink->head_parent_[0];

                    glBegin(GL_LINES);
                    for (int j = 0; j < n_dim_; ++j)
                        v[j] = r[bond_1][j] + u[bond_1][j] *
                            (xlink->cross_position_[0] - 0.5 * length[bond_1]);

                    glVertex3fv(v);

                    for (int j = 0; j < n_dim_; ++j) {
                        v[j] += xlink->r_cross_[j];
                    }

                    glVertex3fv(v);
                    glEnd();
                }
            }
        }
    }
    if (n_dim_ == 3)
        glEnable(GL_LIGHTING);

}

// Draw all the stages of xlinks
void Graphics::DrawCrosslinks(int n_bonds, double **r, double **u, double *length,
                              int n_types, xlink_list *xlinks_0, xlink_list **xlinks_1,
                              xlink_list **xlinks_2) {
    // Do not draw crosslinks if the tomogram view is greater than 1
    if (tomogram_view_ > 1) {
        return;
    }

    DrawCrosslinks(n_bonds, r, u, length, n_types, xlinks_2);
    
    glMatrixMode(GL_MODELVIEW);

    /* Don't draw back facing triangles (as they would be occluded anyway */
    glEnable(GL_CULL_FACE);

    spherocylinder_.PrepDraw();
    
    /* A sphere is just a zero length spherocylinder! */
    glUniform1f(spherocylinder_.uniforms_.half_l, 0.0);

    for (int i_type = 0; i_type < n_types; ++i_type) {
        for (xlink_list::iterator xl_iter = xlinks_0[i_type].begin();
             xl_iter < xlinks_0[i_type].end();
             ++xl_iter) {
            if (xl_iter->IsActive()) {
                glUniform1f(spherocylinder_.uniforms_.diameter,
                            xl_iter->GetDiameter());

                /* Get site coordinates */
                GLfloat v0 = xl_iter->r_cross_[0];
                GLfloat v1 = xl_iter->r_cross_[1];
                GLfloat v2 = xl_iter->r_cross_[2];
                glColor4fv(xl_iter->GetColor());

                spherocylinder_.Draw(v0, v1, v2);
            }
        }
        for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
            for (xlink_list::iterator xl_iter = xlinks_1[i_type][i_bond].begin();
                 xl_iter < xlinks_1[i_type][i_bond].end();
                 ++xl_iter) {
                if (xl_iter->IsActive()) {
                    glUniform1f(spherocylinder_.uniforms_.diameter, 1.5);
                    
                    /* Get site coordinates */
                    GLfloat v0 = r[i_bond][0] +
                        (xl_iter->cross_position_[0] - 0.5 * length[i_bond]) * u[i_bond][0];
                    GLfloat v1 = r[i_bond][1] +
                        (xl_iter->cross_position_[0] - 0.5 * length[i_bond]) * u[i_bond][1];
                    GLfloat v2 = r[i_bond][2] +
                        (xl_iter->cross_position_[0] - 0.5 * length[i_bond]) * u[i_bond][2];
                    glColor4fv(xl_iter->GetColor());

                    spherocylinder_.Draw(v0, v1, v2);
                }
            }
        }
    }
    // Reset the glUniform1f for other drawings!!!!
    glUniform1f(spherocylinder_.uniforms_.diameter, 1.0);
    glDisableVertexAttribArray(spherocylinder_.attributes_.position);
}

void Graphics::DrawKinetochoreMesh(Kinetochore *kc) {
    glDisable(GL_CULL_FACE);
    // First, get the triangle mesh of this kinetochore
    triangle_mesh *tri = kc->tri_;
    // Set the color
    glColor4fv(kc->GetColor());

    //std::cout << "color(" << kc->GetColor()[0] << ", " << kc->GetColor()[1] << ", " << kc->GetColor()[2] <<
    //    ", " << kc->GetColor()[3] << ")\n";

    glBegin(GL_TRIANGLES);
    for (int itri = 0; itri < tri->numTriang; ++itri) {
        // Each triangle maps to 3 verticies
        int ind0 = tri->indVert[0][itri];
        int ind1 = tri->indVert[1][itri];
        int ind2 = tri->indVert[2][itri];
        //std::cout << "tri: " << itri << ", ind(" << ind0 << ", " << ind1 << ", " << ind2 << ")\n";
        //std::cout << "  v0(" 
        //    << (float)tri->verts[0][ind0] << ", "
        //    << (float)tri->verts[1][ind0] << ", "
        //    << (float)tri->verts[2][ind0] << ")\n";
        //std::cout << "  v1(" 
        //    << (float)tri->verts[0][ind1] << ", "
        //    << (float)tri->verts[1][ind1] << ", "
        //    << (float)tri->verts[2][ind1] << ")\n";
        //std::cout << "  v2(" 
        //    << (float)tri->verts[0][ind2] << ", "
        //    << (float)tri->verts[1][ind2] << ", "
        //    << (float)tri->verts[2][ind2] << ")\n";
        glVertex3f((float)tri->verts[0][ind0], (float)tri->verts[1][ind0], (float)tri->verts[2][ind0]);
        glVertex3f((float)tri->verts[0][ind1], (float)tri->verts[1][ind1], (float)tri->verts[2][ind1]);
        glVertex3f((float)tri->verts[0][ind2], (float)tri->verts[1][ind2], (float)tri->verts[2][ind2]);
    }
    glEnd();
    glEnable(GL_CULL_FACE);
}

void Graphics::DrawKinetochores(int n_bonds,
                                double **r,
                                double **u,
                                double *length,
                                int n_chromosomes,
                                ChromosomeManagement *chromosomes) {
    double clength = 1.0;

    // Draw each individual kinetochore as a small disc
    for (int ikc = 0; ikc < 2*n_chromosomes; ++ikc) {
        Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);

        if (tomogram_view_ <= 1) {
            DrawKinetochoreMesh(kc_iter);
        } else {
            int ic = tomogram_view_ - 2;
            int ikc0 = 2*ic;
            int ikc1 = 2*ic + 1;
            if (ikc0 == ikc || ikc1 == ikc) {
                DrawKinetochoreMesh(kc_iter);
            }
        }
    }

    // As a separate step, draw the anchor locations
    glMatrixMode(GL_MODELVIEW);

    glEnable(GL_CULL_FACE);

    spherocylinder_.PrepDraw();

    // A sphere is just a zero length, spherocylinder, also set the size correctly
    glUniform1f(spherocylinder_.uniforms_.half_l, 0.0);

    for (int ikc = 0; ikc < 2*n_chromosomes; ++ikc) {
        Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);

        // Now, draw each of it's binding sites as a smaller sphere
        for (auto p = chromosomes->anchors_[ikc].begin();
                 p != chromosomes->anchors_[ikc].end();
                 ++p) {
            glUniform1f(spherocylinder_.uniforms_.diameter,
                        1.0);

            // Location
            GLfloat w0 = p->pos[0];
            GLfloat w1 = p->pos[1];
            GLfloat w2 = p->pos[2];

            // Color the same as the kc
            glColor4fv(kc_iter->GetColorAF());

            // Draw it
            if (tomogram_view_ <= 1) {
                spherocylinder_.Draw(w0, w1, w2);
            } else {
                int ic = tomogram_view_ - 2;
                int ikc0 = 2*ic;
                int ikc1 = 2*ic + 1;
                if (ikc0 == ikc || ikc1 == ikc) {
                    spherocylinder_.Draw(w0, w1, w2);
                }
            }
        }
    }

    // We also want to know somethign about the chromatid material to exclude and draw
    glUniform1f(spherocylinder_.uniforms_.half_l, 0.5*chromosomes->chromatid_length_);
    glUniform1f(spherocylinder_.uniforms_.diameter, chromosomes->chromatid_diameter_);
    for (int ikc = 0; ikc < 2*n_chromosomes; ++ikc) {
        Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);

        // Determine phi amount of this, based on the v vector of the kinetochore, which we are aligned to
        double phi = acos(kc_iter->v_[2]);

        /* Determine theta rotation, amount to rotate about z. */
        double length_xy = sqrt(SQR(kc_iter->v_[0]) + SQR(kc_iter->v_[1]));
        double theta = 0.0;
        if (length_xy > 0.0) {
            theta = acos(kc_iter->v_[0] / length_xy);
            if (kc_iter->v_[1] < 0.0)
                theta = (2.0 * M_PI) - theta;
        }

        // Location of the center of the chromatid
        double flipsister = *(kc_iter->second_sister_) ? -1.0 : 1.0;
        GLfloat w0 = kc_iter->r_[0] - flipsister * chromosomes->chromatid_kc_offset_ * kc_iter->u_[0];
        GLfloat w1 = kc_iter->r_[1] - flipsister * chromosomes->chromatid_kc_offset_ * kc_iter->u_[1];
        GLfloat w2 = kc_iter->r_[2] - flipsister * chromosomes->chromatid_kc_offset_ * kc_iter->u_[2];

        // Color the same as the kc
        glColor4fv(kc_iter->GetColor());
        //glColor4fv(chromosomes->color_chromatid_);

        // Draw the chromatid
        if (tomogram_view_ <= 1) {
            spherocylinder_.Draw(w0, w1, w2, theta, phi);
        } else {
            int ic = tomogram_view_ - 2;
            int ikc0 = 2*ic;
            int ikc1 = 2*ic + 1;
            if (ikc0 == ikc || ikc1 == ikc) {
                spherocylinder_.Draw(w0, w1, w2, theta, phi);
            }
        }
    }

    // Reset the glUniform1f for other drawings!!!!
    glUniform1f(spherocylinder_.uniforms_.diameter, 1.0);
    glDisableVertexAttribArray(spherocylinder_.attributes_.position);

    // Now, draw the linked ones, because graphics
    glUseProgram(0); // Use old fixed pipeline
    glMatrixMode(GL_MODELVIEW); // Make sure we're using the model transform
    
    /* Turn off the light. */
    glDisable(GL_LIGHTING);

    /* Select line width. */
    if (tomogram_view_ == 0 || tomogram_view_ == 1) {
        glLineWidth(5.0);
    } else {
        glLineWidth(20.0);
    }
    for (int ikc = 0; ikc < 2*n_chromosomes; ++ikc) {
        Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);

        // Color
        glColor4fv(kc_iter->GetColorAF());

        // Each site
        for (int isite = 0; isite < chromosomes->naf_; ++isite) {
            if (kc_iter->attach_[isite] == -1) {
                continue;
            }
            
            // Check to see if we are just drawing chromosome of one type or another
            if (tomogram_view_ > 1) {
                int ic = tomogram_view_ - 2;
                int ikc0 = 2*ic;
                int ikc1 = 2*ic+1;
                if ((ikc != ikc0) && (ikc != ikc1)) {
                    continue;
                }
            }

            GLfloat v[3] = {0.0, 0.0, 0.0};
            for (int i = 0; i < chromosomes->ndim_; ++i) {
                v[i] = chromosomes->anchors_[ikc][isite].pos[i];
            }

            glBegin(GL_LINES);

            glVertex3fv(v);

            for (int i = 0; i < chromosomes->ndim_; ++i) {
                v[i] += kc_iter->r_cross_[isite][i];
            }

            glVertex3fv(v);
            glEnd();
        }
    }
    glEnable(GL_LIGHTING);
}

void Graphics::DrawKinetochores_old(int n_bonds,
                                double **r,
                                double **u,
                                double *length,
                                int n_chromosomes,
                                ChromosomeManagement *chromosomes) {
    // Draw the spheres first
    glMatrixMode(GL_MODELVIEW);

    glEnable(GL_CULL_FACE);

    spherocylinder_.PrepDraw();

    // A sphere is just a zero length, spherocylinder, also set the size correctly
    glUniform1f(spherocylinder_.uniforms_.half_l, 0.0);

    // Loop over kinetochores
    for (int ikc = 0; ikc < 2*n_chromosomes; ++ikc) {
        //Kinetochore *kc_iter = &(kcs[ikc]);
        Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);

        glUniform1f(spherocylinder_.uniforms_.diameter,
                    kc_iter->GetDrawDiameter());

        // Location
        GLfloat v0 = kc_iter->r_[0];
        GLfloat v1 = kc_iter->r_[1];
        GLfloat v2 = kc_iter->r_[2];

        // Color
        glColor4fv(kc_iter->GetColor());

        spherocylinder_.Draw(v0, v1, v2);

        // Now, draw each of it's binding sites as a smaller sphere
        for (auto p = chromosomes->anchors_[ikc].begin();
                 p != chromosomes->anchors_[ikc].end();
                 ++p) {
            glUniform1f(spherocylinder_.uniforms_.diameter,
                        1.0);

            // Location
            GLfloat w0 = p->pos[0];
            GLfloat w1 = p->pos[1];
            GLfloat w2 = p->pos[2];

            // Color the same as the kc
            glColor4fv(kc_iter->GetColor());

            // Draw it
            spherocylinder_.Draw(w0, w1, w2);
        }
    }
    // Reset the glUniform1f for other drawings!!!!
    glUniform1f(spherocylinder_.uniforms_.diameter, 1.0);
    glDisableVertexAttribArray(spherocylinder_.attributes_.position);

    // Now, draw the linked ones, because graphics
    glUseProgram(0); // Use old fixed pipeline
    glMatrixMode(GL_MODELVIEW); // Make sure we're using the model transform
    
    /* Turn off the light. */
    glDisable(GL_LIGHTING);

    /* Select line width. */
    glLineWidth(3.0);
    for (int ikc = 0; ikc < 2*n_chromosomes; ++ikc) {
        Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);

        // Color
        glColor4fv(kc_iter->GetColor());

        // Each site
        for (int isite = 0; isite < chromosomes->naf_; ++isite) {
            if (kc_iter->attach_[isite] == -1) {
                continue;
            }
            GLfloat v[3] = {0.0, 0.0, 0.0};
            for (int i = 0; i < chromosomes->ndim_; ++i) {
                v[i] = chromosomes->anchors_[ikc][isite].pos[i];
            }

            glBegin(GL_LINES);

            glVertex3fv(v);

            for (int i = 0; i < chromosomes->ndim_; ++i) {
                v[i] += kc_iter->r_cross_[isite][i];
            }

            glVertex3fv(v);
            glEnd();
        }
    }
    glEnable(GL_LIGHTING);
}

void Graphics::DrawCrosslinks(int n_bonds, double **r, double **u, double *length,
                              int n_types, xlink_list *xlinks_0, xlink_list **xlinks_2) {
    DrawCrosslinks(n_bonds, r, u, length, n_types, xlinks_2);
    
    glMatrixMode(GL_MODELVIEW);

    /* Don't draw back facing triangles (as they would be occluded anyway */
    glEnable(GL_CULL_FACE);

    spherocylinder_.PrepDraw();
    
    /* A sphere is just a zero length spherocylinder! */
    glUniform1f(spherocylinder_.uniforms_.half_l, 0.0);

    for (int i_type = 0; i_type < n_types; ++i_type) {
        for (xlink_list::iterator xl_iter = xlinks_0[i_type].begin();
             xl_iter < xlinks_0[i_type].end();
             ++xl_iter) {
            if (xl_iter->IsActive()) {
                glUniform1f(spherocylinder_.uniforms_.diameter,
                            xl_iter->GetDiameter());

                /* Get site coordinates */
                GLfloat v0 = xl_iter->r_cross_[0];
                GLfloat v1 = xl_iter->r_cross_[1];
                GLfloat v2 = xl_iter->r_cross_[2];
                glColor4fv(xl_iter->GetColor());

                spherocylinder_.Draw(v0, v1, v2);
            }
        }
    }
    glDisableVertexAttribArray(spherocylinder_.attributes_.position);
}


void Graphics::DrawDiscorectangles(int n_spheros, double **r, double **u, double *length) {
    glMatrixMode(GL_MODELVIEW); // Make sure we're using the model transform

    discorectangle_.PrepDraw();

    // Set default bond color  
    GLfloat color[4] = {0.0, 0.0, 1.0, 1.0};
    for (int i_bond = 0; i_bond < n_spheros; ++i_bond) {
        double theta = atan2(u[i_bond][1], u[i_bond][0]); // rotation angle
        
        if (color_switch_ == 1) { // basic theta dependent colorwheel
            double theta_color = theta + 1.5 * M_PI;
            if (theta_color > 2.0 * M_PI)
                theta_color -= 2.0 * M_PI;
            
            /* Color based only on orientation */
            int color_index = (int) (((theta_color) / (2.0 * M_PI)) * n_rgb_);
            if (color_index < 0)
                color_index = 0;
            else if (color_index >= n_rgb_)
                color_index = n_rgb_ - 1;
            
            glColor4fv(&colormap_[color_index * 4]);
        }
        else if (color_switch_ == 2) { // FIXME: doesn't work currently. just set the default color
            /* Custom colors externally defined... */
            // glColor4f(bond_colors_[i_bond][0],
            //           bond_colors_[i_bond][1],
            //           bond_colors_[i_bond][2],
            //           bond_colors_[i_bond][3]);
            glColor4fv(color);
        }
        else {
            /* Flat color */
            glColor4fv(color);
        }

        glPushMatrix(); // Duplicate current modelview
        {
            glTranslatef(r[i_bond][0], r[i_bond][1], 0.0); // Translate rod
            glRotatef((GLfloat) ((theta / M_PI) * 180.0 - 90.0), 0.0, 0.0, 1.0); // rotate rod

            // Tell shader about our spherocylinder parameters
            double half_length = 0.5 * length[i_bond];
            glUniform1f(discorectangle_.uniforms_.half_l, half_length);
            glUniform1f(discorectangle_.uniforms_.diameter, 1.0);
            glDrawElements(GL_TRIANGLES,
                           discorectangle_.n_triangles_*3,
                           GL_UNSIGNED_SHORT,
                           (void*) 0); // draw.
        }
        glPopMatrix(); // restore default modelview
    }

    glDisableVertexAttribArray(discorectangle_.attributes_.position); // Tell GL to forget about our vertex array
}

void Graphics::Draw2d(int n_bonds, double **h,
                      double **r, double **u, double *length) {
    KeyInteraction();
    UpdateWindow(); // Recalculate window parameters (in case of resize)
    DrawDiscorectangles(n_bonds, r, u, length); 
    DrawBoundary(h);
    glfwSwapBuffers(window_); // Print frame to screen
}

void Graphics::Draw2d(int n_bonds, double **h,
                      double **r, double **u, double *length,
                      int n_types, xlink_list **xlinks) {
    KeyInteraction();
    UpdateWindow(); // Recalculate window parameters (in case of resize)
    DrawDiscorectangles(n_bonds, r, u, length);
    DrawCrosslinks(n_bonds, r, u, length, n_types, xlinks);
    DrawBoundary(h);
    glfwSwapBuffers(window_); // Print frame to screen
}


void Graphics::DrawSpheros(int n_spheros, double **r, double **u, double *length, ChromosomeManagement *chromosomes) {
    //static int r_factor=0; //Code to scan colors to find the best fit
    //static int b_factor=0; 
    //static int g_factor=-10; 
    glMatrixMode(GL_MODELVIEW); // Use modelview matrix

    /* Don't draw back facing triangles (as they would be occluded anyway */
    glEnable(GL_CULL_FACE);

    spherocylinder_.PrepDraw();

    GLfloat color[4] = {0.0, 0.0, 1.0, 1.0}; // default bond color
    for (int i_bond = 0; i_bond < n_spheros; ++i_bond) {
        /* Determine phi rotation angle, amount to rotate about y. */
        double phi = acos(u[i_bond][2]);

        /* Determine theta rotation, amount to rotate about z. */
        double length_xy = sqrt(SQR(u[i_bond][0]) + SQR(u[i_bond][1]));
        double theta = 0.0;
        if (length_xy > 0.0) {
            theta = acos(u[i_bond][0] / length_xy);
            if (u[i_bond][1] < 0.0)
                theta = (2.0 * M_PI) - theta;
        }

        if (i_bond < n_spheros/2 && n_anchors_ == 2){
            //FIXME Hack for when spbs have the same number of spheros on each
            //color[0] = color_vector_[0] = 0.5586+double(r_factor)/100;
            //color[1] = color_vector_[1] = 0.5156+double(g_factor)/100; 
            //color[2] = color_vector_[2] = 0.5977+double(b_factor)/100; 
            color[0] = 0.618600;
            color[1] = 0.435600;
            color[2] = 0.757700; //Purple spherocylinders
        }
        else{
            color[0] = 0;
            color[1] = 0.6680; 
            color[2] = 0.4258; //Green spherocylinders
        }

        // If we have chromosomes that can attach, loop through and figure out which of the
        // spherocylinders have attachments and color them BRIGHT
        int is_attached = 0;
        if (chromosomes) {
            // Loop through kinetochore attachments and see if we have a attachment to this bond
            for (int ikc = 0; ikc < chromosomes->nkcs_; ++ikc) {
                for (int isite = 0; isite < chromosomes->naf_; ++isite) {
                    if (chromosomes->kinetochores_[ikc].attach_[isite] == i_bond) {
                        int ic = tomogram_view_ - 2;
                        int ikc0 = 2*ic;
                        int ikc1 = 2*ic+1;
                        if (ikc0 == ikc || ikc1 == ikc) {
                            is_attached = 1;
                        }
                        // Neon color!
                        if (i_bond < n_spheros/2 && n_anchors_ == 2){
                            //// Neon purple
                            //color[0] = 0.6;
                            //color[1] = 0.0;
                            //color[2] = 1.0;
                            ////Pink?
                            //color[0] = 1.0;
                            //color[1] = 0.08;
                            //color[2] = 0.58;
                            color[0] = 1.0;
                            color[1] = 0.0;
                            color[2] = 1.0;
                        } else {
                            //// Neon Green
                            //color[0] = 0.5;
                            //color[1] = 1.0;
                            //color[2] = 0.0;
                            color[0] = 1.0;
                            color[1] = 1.0;
                            color[2] = 0.0;
                        }
                    }
                }
            }
        }

        /* Select cylinder color. */
        if (color_switch_ == 1) {
            /* Convert from HSL to RGB coloring scheme, unique orientation coloring scheme */
            double L = 0.3 * u[i_bond][2] + 0.5;
            double C = (1 - ABS(2*L - 1));
            double H_prime = 3.0 * theta / M_PI;
            double X = C * (1.0 - ABS(fmod(H_prime, 2.0) - 1));

            if (H_prime < 0.0) {
                color[0] = 0.0;
                color[1] = 0.0;
                color[2] = 0.0;
            }
            else if (H_prime < 1.0) {
                color[0] = C;
                color[1] = X;
                color[2] = 0;
            }
            else if (H_prime < 2.0) {
                color[0] = X;
                color[1] = C;
                color[2] = 0;
            }
            else if (H_prime < 3.0) {
                color[0] = 0;
                color[1] = C;
                color[2] = X;
            }
            else if (H_prime < 4.0) {
                color[0] = 0;
                color[1] = X;
                color[2] = C;
            }
            else if (H_prime < 5.0) {
                color[0] = X;
                color[1] = 0;
                color[2] = C;
            }
            else if (H_prime < 6.0) {
                color[0] = C;
                color[1] = 0;
                color[2] = X;
            }
            color[3] = alpha_;

            double m = L - 0.5 * C;
            color[0] = color[0] + m;
            color[1] = color[1] + m;
            color[2] = color[2] + m;

            glColor4fv(color);
        }
        else if (color_switch_ == 2) {
            double cos_phi = cos(phi);
            int color_index = (int) (ABS(cos_phi) * (n_rgb_ - shift_));
            if (color_index < 0)
                glColor4fv(&colormap_[0]);
            else if (color_index >= n_rgb_ - shift_)
                glColor4fv(&colormap_[4*(n_rgb_ - shift_ - 1)]);
            else
                glColor4fv(&colormap_[4*color_index]);
        }
        else
            glColor4fv(color);

        /* Get position of spherocylinder center. */
        GLfloat v0 = r[i_bond][0];
        GLfloat v1 = r[i_bond][1];
        GLfloat v2 = r[i_bond][2];

        /* Let the shader know the length of our bond */
        GLfloat half_length = 0.5 * length[i_bond];
        glUniform1f(spherocylinder_.uniforms_.half_l, half_length);
        glUniform1f(spherocylinder_.uniforms_.diameter, 1.0);

        if (tomogram_view_ <= 1) {
            spherocylinder_.Draw(v0, v1, v2, theta, phi);
        } else {
            // For the chromosome views, if attached to this chromosome, then draw, otherwise, don't!
            if (is_attached) {
                spherocylinder_.Draw(v0, v1, v2, theta, phi);
            }
        }
    }
    glDisableVertexAttribArray(spherocylinder_.attributes_.position);
    //More code to scan through colors
    //b_factor++;
    //if (b_factor > 20){
        //b_factor = 0;
        //r_factor++;
        //if (r_factor > 20){
            //r_factor = -0;
            //g_factor++;
            //if (g_factor > 10)
                //g_factor = -10;
        //}
    //}
}

void Graphics::DrawAxes() {
    glDisable(GL_LIGHTING);
    glPushMatrix();
    //glTranslatef (-2.4, -1.5, -5);
    glLineWidth (5.0);
    glBegin (GL_LINES);
    // Do some hopeful math with the generation of this length based on the viewport
    double f = 1.0/tan(60.0/2.0*M_PI/180.0);
    double y = -zTrans_ / f;
    double yview = f*y/-zTrans_;
    double yscale = y/xyzScale_;

    std::cout << "f: " << f << std::endl;
    std::cout << "y: " << y << std::endl;
    std::cout << "yview: " << yview << std::endl;
    std::cout << "yscale: " << yscale << std::endl;
    double lengths = yscale-1;
        glColor3f (1,0,0); glVertex3f(0.0, 0.0, 0.0); glVertex3f(lengths, 0.0, 0.0);
        glColor3f (0,1,0); glVertex3f(0.0, 0.0, 0.0); glVertex3f(0.0, lengths, 0.0);
        glColor3f (0,0,1); glVertex3f(0.0, 0.0, 0.0); glVertex3f(0.0, 0.0, lengths);
        //glColor3f (1,1,0); glVertex3f(0.0, 0.0, 0.0); glVertex3f(ntomo_[0]*1000.0, ntomo_[1]*1000.0, ntomo_[2]*1000.0);
    glEnd();
    glPopMatrix();
}

void Graphics::DrawSpheres(int n_spheres, double **r, double sphere_diameter) { 
    glMatrixMode(GL_MODELVIEW);

    /* Don't draw back facing triangles (as they would be occluded anyway */
    glEnable(GL_CULL_FACE);

    spherocylinder_.PrepDraw();
    
    /* A sphere is just a zero length spherocylinder! */
    glUniform1f(spherocylinder_.uniforms_.half_l, 0.0);
    glUniform1f(spherocylinder_.uniforms_.diameter,
                sphere_diameter);

    GLfloat color[4] = { 0.25, 0.0, 0.5, 1.0 };
    glColor4fv(color);

    for (int i_sphere = 0; i_sphere < n_spheres; ++i_sphere) {
        /* Get site coordinates */
        GLfloat v0 = r[i_sphere][0];
        GLfloat v1 = r[i_sphere][1];
        GLfloat v2 = r[i_sphere][2];

        spherocylinder_.Draw(v0, v1, v2);
    }
    glDisableVertexAttribArray(spherocylinder_.attributes_.position);
}

void Graphics::DrawAnchorsOld(int n_anchors, double **r_anchor, double *anchor_diameter, double **h,
        float **anchor_color) {
    qobj_ = gluNewQuadric();
    gluQuadricCallback(qobj_, GLU_ERROR, NULL);
    gluQuadricDrawStyle(qobj_, GLU_FILL);
    gluQuadricNormals(qobj_, GLU_SMOOTH);

    //glColor4f(1, 0.64453125, 0.0, 1.0);
    double clength = 2.0;

    for (int ianchor = 0; ianchor < n_anchors; ++ianchor) {
    //for (int ianchor = 0; ianchor < 1; ++ianchor) {
        //double r_box = 0.0;
        //for (int i = 0; i < 3; ++i) {
        //    r_box += SQR(r_anchor[ianchor][i]);
        //}
        //r_box = sqrt(r_box);
        double r_box = 0.5 * h[0][0];
        // Azimuthal angle
        double phi = atan2(r_anchor[ianchor][1], r_anchor[ianchor][0]);
        // Polar angle
        double theta = acos(r_anchor[ianchor][2] / r_box);

        // Set the position of one side of the cylinder
        double v0 = r_anchor[ianchor][0] - 0.5 * clength * r_anchor[ianchor][0] / r_box;
        double v1 = r_anchor[ianchor][1] - 0.5 * clength * r_anchor[ianchor][1] / r_box;
        double v2 = r_anchor[ianchor][2] - 0.5 * clength * r_anchor[ianchor][2] / r_box;

        // Set the color
        glColor4f(anchor_color[ianchor][0],
                  anchor_color[ianchor][1],
                  anchor_color[ianchor][2],
                  anchor_color[ianchor][3]);

        glPushMatrix();
        glTranslatef(v0, v1, v2);
        if (phi != 0.0)
            glRotatef((GLfloat) ((phi / M_PI) * 180.0), 0.0, 0.0, 1.0);
        if (theta != 0.0)
            glRotatef((GLfloat) ((theta / M_PI) * 180.0), 0.0, 1.0, 0.0);
        gluQuadricOrientation(qobj_, GLU_OUTSIDE);
        if (tomogram_view_ <= 1) {
            gluCylinder(qobj_,
                        0.5 * anchor_diameter[ianchor],
                        0.5 * anchor_diameter[ianchor],
                        clength, 16, 1);
            gluQuadricOrientation(qobj_, GLU_INSIDE);
            gluDisk(qobj_,
                    0.0,
                    0.5 * anchor_diameter[ianchor],
                    16, 16);
        }
        glPopMatrix();

        v0 = r_anchor[ianchor][0] + 0.5 * clength * r_anchor[ianchor][0] / r_box;
        v1 = r_anchor[ianchor][1] + 0.5 * clength * r_anchor[ianchor][1] / r_box;
        v2 = r_anchor[ianchor][2] + 0.5 * clength * r_anchor[ianchor][2] / r_box;

        glPushMatrix();
        glTranslatef(v0, v1, v2);
        if (phi != 0.0)
            glRotatef((GLfloat) ((phi / M_PI) * 180.0), 0.0, 0.0, 1.0);
        if (theta != 0.0)
            glRotatef((GLfloat) ((theta / M_PI) * 180.0), 0.0, 1.0, 0.0);
        if (tomogram_view_ <= 1) {
            gluQuadricOrientation(qobj_, GLU_OUTSIDE);
            gluDisk(qobj_,
                    0.0,
                    0.5 * anchor_diameter[ianchor],
                    16, 16);
        }
        glPopMatrix();
    }

    // Remove the quadric
    gluDeleteQuadric(qobj_);
}

// Fully 3d centrosomes must be drawn correctly
void Graphics::DrawAnchorsNew(int n_anchors, double **r_anchor, double **u_anchor, double **v_anchor, double **w_anchor, double *anchor_diameter, double **h, float **anchor_color) {
    qobj_ = gluNewQuadric();
    gluQuadricCallback(qobj_, GLU_ERROR, NULL);
    gluQuadricDrawStyle(qobj_, GLU_FILL);
    gluQuadricNormals(qobj_, GLU_SMOOTH);

    double clength = 2.0;
    for (int ianchor = 0; ianchor < n_anchors; ++ianchor) {
        // Direction of the anchor
        double phi = atan2(u_anchor[ianchor][1], u_anchor[ianchor][0]);
        double theta = acos(u_anchor[ianchor][2]);
        // The anchor points in the uhat direction
        double v0 = r_anchor[ianchor][0] - 0.5 * clength * u_anchor[ianchor][0];
        double v1 = r_anchor[ianchor][1] - 0.5 * clength * u_anchor[ianchor][1];
        double v2 = r_anchor[ianchor][2] - 0.5 * clength * u_anchor[ianchor][2];

        // Set the color
        glColor4f(anchor_color[ianchor][0],
                  anchor_color[ianchor][1],
                  anchor_color[ianchor][2],
                  anchor_color[ianchor][3]);

        glPushMatrix();
        glTranslatef(v0, v1, v2);
        if (phi != 0.0)
            glRotatef((GLfloat) ((phi / M_PI) * 180.0), 0.0, 0.0, 1.0);
        if (theta != 0.0)
            glRotatef((GLfloat) ((theta / M_PI) * 180.0), 0.0, 1.0, 0.0);
        gluQuadricOrientation(qobj_, GLU_OUTSIDE);
        if (tomogram_view_ <= 1) {
            gluCylinder(qobj_,
                        0.5 * anchor_diameter[ianchor],
                        0.5 * anchor_diameter[ianchor],
                        clength, 16, 1);
            gluQuadricOrientation(qobj_, GLU_INSIDE);
            gluDisk(qobj_,
                    0.0,
                    0.5 * anchor_diameter[ianchor],
                    16, 16);
        }
        glPopMatrix();

        v0 = r_anchor[ianchor][0] + 0.5 * clength * u_anchor[ianchor][0];
        v1 = r_anchor[ianchor][1] + 0.5 * clength * u_anchor[ianchor][1];
        v2 = r_anchor[ianchor][2] + 0.5 * clength * u_anchor[ianchor][2];

        glPushMatrix();
        glTranslatef(v0, v1, v2);
        if (phi != 0.0)
            glRotatef((GLfloat) ((phi / M_PI) * 180.0), 0.0, 0.0, 1.0);
        if (theta != 0.0)
            glRotatef((GLfloat) ((theta / M_PI) * 180.0), 0.0, 1.0, 0.0);
        if (tomogram_view_ <= 1) {
            gluQuadricOrientation(qobj_, GLU_OUTSIDE);
            gluDisk(qobj_,
                    0.0,
                    0.5 * anchor_diameter[ianchor],
                    16, 16);
        }
        glPopMatrix();
    }
    // Remove the quadric
    gluDeleteQuadric(qobj_);
}

void Graphics::DrawAnchors(int n_anchors, double **r_anchor, double *anchor_diameter, double **h) {
    glColor4f(1, 0.64453125, 0.0, 1.0);
    double clength = 2.0;

    for (int ianchor = 0; ianchor < n_anchors; ++ianchor) {
        //double r_box = 0.0;
        //for (int i = 0; i < 3; ++i) {
        //    r_box += SQR(r_anchor[ianchor][i]);
        //}
        //r_box = sqrt(r_box);
        double r_box = 0.5 * h[0][0];
        // Azimuthal angle
        double phi = atan2(r_anchor[ianchor][1], r_anchor[ianchor][0]);
        // Polar angle
        double theta = acos(r_anchor[ianchor][2] / r_box);

        // Middle of the anchor
        double v0 = r_anchor[ianchor][0];
        double v1 = r_anchor[ianchor][1];
        double v2 = r_anchor[ianchor][2];

        GLfloat r0 = 0.5 * anchor_diameter[ianchor];

        // set up local coordinates
        glPushMatrix();
        glTranslatef(v0, v1, v2);
        if (phi != 0.0)
            glRotatef((GLfloat) ((phi / M_PI) * 180.0), 0.0, 0.0, 1.0);
        if (theta != 0.0)
            glRotatef((GLfloat) ((theta / M_PI) * 180.0), 0.0, 1.0, 0.0);

        glBegin(GL_QUAD_STRIP);
        for (GLint i = 0; i < 16; ++i) {
           GLfloat angle = i * 2.f * (float)M_PI/16.0; 
           glNormal3f(-(float) cos(angle), -(float) sin(angle), 0.f);
           glVertex3f(r0 * (float) cos(angle), r0 * (float) sin(angle), -clength * 0.5f);
           glVertex3f(r0 * (float) cos(angle), r0 * (float) sin(angle), clength * 0.5f);
        }
        glEnd();

        // finished, pop
        glPopMatrix();
    }
}

void Graphics::Draw3d(int n_bonds, double **h,
                      double **r, double **u, double *length,
                      int n_sites, double **r_site,
                      double sphere_diameter) {
    KeyInteraction();
    UpdateWindow();
    DrawSpheros(n_bonds, r, u, length);
    if (n_sites > n_bonds * 2) {
        DrawSpheres(n_sites - 2*n_bonds, &r_site[n_bonds*2], sphere_diameter);
    }
    DrawBoundary(h);
    glfwSwapBuffers(window_);
}

void Graphics::Draw3d(int n_bonds, double **h,
                      double **r, double **u, double *length) {
    KeyInteraction();
    UpdateWindow();
    DrawSpheros(n_bonds, r, u, length);
    DrawBoundary(h);
    glfwSwapBuffers(window_);
}

void Graphics::Draw3d(int n_bonds, double **h,
                      double **r, double **u, double *length,
                      int n_types, xlink_list **xlinks) {
    KeyInteraction();
    UpdateWindow();
    DrawSpheros(n_bonds, r, u, length);
    DrawCrosslinks(n_bonds, r, u, length, n_types, xlinks);
    DrawBoundary(h);
    glfwSwapBuffers(window_);
}

void Graphics::Draw3d(int n_bonds, double **h,
                      double **r, double **u, double *length,
                      int n_types, xlink_list *xlinks_0, xlink_list **xlinks_1,
                      xlink_list **xlinks_2) {
    KeyInteraction();
    UpdateWindow();
    DrawSpheros(n_bonds, r, u, length);
    DrawCrosslinks(n_bonds, r, u, length, n_types, xlinks_0, xlinks_1, xlinks_2);
    DrawBoundary(h);
    glfwSwapBuffers(window_);
}

void Graphics::Draw3d(int n_bonds, double **h,
                      double **r, double **u, double *length,
                      int n_types, xlink_list *xlinks_0, xlink_list **xlinks_2) {
    KeyInteraction();
    UpdateWindow();
    DrawSpheros(n_bonds, r, u, length);
    DrawCrosslinks(n_bonds, r, u, length, n_types, xlinks_0, xlinks_2);
    DrawBoundary(h);
    glfwSwapBuffers(window_);
}

// Stage 0 1 and 2 xlinks with anchors and chromosomes
void Graphics::Draw3d(int n_bonds,
                      double **h,
                      double **r_bond,
                      double **u_bond,
                      double *rlength,
                      int n_types,
                      xlink_list *xlinks0,
                      xlink_list **xlinks1,
                      xlink_list **xlinks2,
                      int n_anchors,
                      int anchor_type,
                      float **color_anchor,
                      double **r_anchor,
                      double **u_anchor,
                      double **v_anchor,
                      double **w_anchor,
                      double *adiam,
                      int n_chromosomes,
                      ChromosomeManagement *chromosomes) {
    KeyInteraction();
    if (tomogram_view_ == 1) {
        //GenerateTomogramViewUtildeAvg(n_bonds, u_bond, n_anchors);
        GenerateTomogramViewSPBs(r_anchor, n_anchors);
    } else if (tomogram_view_ > 1) {
        // Chromosome view
        GenerateChromosomeView(n_chromosomes, chromosomes);
    }
    UpdateWindow();
    // HAVE TO DRAW ANCHORS FIRST BECAUSE OTHERWISE DIAMETERS AND STUFF
    // GETS ALL SCREWED UP AND ALL CAPS BLARG!~!!!!!
    if (n_anchors != 0 && anchor_type == 0) {
        DrawAnchorsOld(n_anchors, r_anchor, adiam, h, color_anchor);
    } else if (anchor_type == 1) {
        DrawAnchorsNew(n_anchors, r_anchor, u_anchor, v_anchor, w_anchor, adiam, h, color_anchor);
    }
    DrawKinetochores(n_bonds, r_bond, u_bond, rlength, n_chromosomes, chromosomes);
    DrawSpheros(n_bonds, r_bond, u_bond, rlength, chromosomes);
    DrawCrosslinks(n_bonds, r_bond, u_bond, rlength, n_types, xlinks0, xlinks1, xlinks2);
    DrawBoundary(h);
    //DrawAxes();
    glfwSwapBuffers(window_);
}

// Generate the tomographic view for SPBs utilde vectors
void Graphics::GenerateTomogramViewUtildeAvg(int nbonds,
                                        double **ubond,
                                        int nanchors) {
    //reset the values of the utilde vectors
    utilde_.clear();
    for (int ispb = 0; ispb < nanchors; ++ispb) {
        utilde_.push_back(std::vector<double>(3, 0.0));
    }
    int bonds_per_spb = nbonds / nanchors;
    for (int ibond = 0; ibond < nbonds; ++ibond) {
        int ispb = ibond/bonds_per_spb;
        for (int i = 0; i < n_dim_; ++i) {
            utilde_[ispb][i] += ubond[ibond][i]/bonds_per_spb; 
        }
    }
    //for (int ispb = 0; ispb < nanchors; ++ispb) {
    //    std::cout << "utilde[" << ispb << "] (" << utilde_[ispb][0] << ", "
    //                                            << utilde_[ispb][1] << ", "
    //                                            << utilde_[ispb][2] << ")\n";
    //}
    cross_product(utilde_[0].data(), utilde_[1].data(), ntomo_, n_dim_);
    double norm_factor = sqrt(dot_product(n_dim_, ntomo_, ntomo_));
    for (int i = 0; i < n_dim_; ++i) {
        ntomo_[i] = ntomo_[i] / norm_factor;
    }
    //std::cout << "ntomo (" << ntomo_[0] << ", "
    //                       << ntomo_[1] << ", "
    //                       << ntomo_[2] << ")\n";
}

// Generate the tomographic view for SPBs vectors
void Graphics::GenerateTomogramViewSPBs(double **ranchor,
                                        int nanchors) {
    //reset the values of the utilde vectors
    utilde_.clear();
    for (int ispb = 0; ispb < nanchors; ++ispb) {
        std::vector<double> uspb(3, 0.0);
        double normu = 0.0;
        uspb[0] = ranchor[ispb][0];
        uspb[1] = ranchor[ispb][1];
        uspb[2] = ranchor[ispb][2];
        normu = sqrt(SQR(uspb[0]) + SQR(uspb[1]) + SQR(uspb[2]));
        uspb[0] /= normu;
        uspb[1] /= normu;
        uspb[2] /= normu;
        utilde_.push_back(uspb);
    }
    // Calculate the spb rhatvector
    for (int i = 0; i < n_dim_; ++i) {
        rhatspb_[i] = ranchor[1][i] - ranchor[0][i];
    }
    //for (int ispb = 0; ispb < nanchors; ++ispb) {
    //    std::cout << "utilde[" << ispb << "] (" << utilde_[ispb][0] << ", "
    //                                            << utilde_[ispb][1] << ", "
    //                                            << utilde_[ispb][2] << ")\n";
    //}
    cross_product(utilde_[0].data(), utilde_[1].data(), ntomo_, n_dim_);
    double norm_factor = sqrt(dot_product(n_dim_, ntomo_, ntomo_));
    for (int i = 0; i < n_dim_; ++i) {
        ntomo_[i] = ntomo_[i] / norm_factor;
    }
    norm_factor = sqrt(dot_product(n_dim_, rhatspb_, rhatspb_));
    for (int i = 0; i < n_dim_; ++i) {
        rhatspb_[i] = rhatspb_[i] / norm_factor;
    }
    //std::cout << "ntomo (" << ntomo_[0] << ", "
    //                       << ntomo_[1] << ", "
    //                       << ntomo_[2] << ")\n";
    //std::cout << "rhat (" << rhatspb_[0] << ", "
    //                      << rhatspb_[1] << ", "
    //                      << rhatspb_[2] << ")\n";
    //exit(0);
}

// Generate the Chromosome view according to tomorogram_view parameter
void Graphics::GenerateChromosomeView(int n_chromosomes, ChromosomeManagement *chromosomes) {
    // Grab the chromosome of interest
    int ic = tomogram_view_ - 2;

    // Find the initial r vector r = r_A - r_B
    double rmag2 = 0.0;
    double r[3] = {0.0};
    for (int i = 0; i < 3; ++i) {
        r[i] = chromosomes->kinetochores_[2*ic].r_[i] - 
               chromosomes->kinetochores_[2*ic+1].r_[i];
        rmag2 += SQR(r[i]);
    }
    double rmag = sqrt(rmag2);
    double rhat[3] = {0.0};
    for (int i = 0; i < 3; ++i) {
        rhat[i] = r[i] / rmag;
    }

    // Get the orientation vector along the chromatid of KC1
    double ukc[3] = {0.0};
    for (int i = 0; i < 3; ++i) {
        ukc[i] = chromosomes->kinetochores_[2*ic].v_[i];
    }
    cross_product(rhat, ukc, ntomo_, n_dim_);
    double norm_factor = sqrt(dot_product(n_dim_, ntomo_, ntomo_));
    for (int i = 0; i < n_dim_; ++i) {
        ntomo_[i] = ntomo_[i] / norm_factor;
    }

    // Overload rhatspb to use the rkc vector
    for (int i = 0; i < 3; ++i) {
        rhatspb_[i] = rhat[i];
        rchromosome_[i] = (chromosomes->kinetochores_[2*ic].r_[i] + chromosomes->kinetochores_[2*ic+1].r_[i])/2.0;
    }
}

void Graphics::SetBoundaryType(std::string boundary_type) {
    if (boundary_type.compare("sphere") == 0) {
        boundary_type_ = boundary_type;
    }
    else if (boundary_type.compare("cube") == 0) {
        boundary_type_ = boundary_type;
    }
    else {
        std::cerr << "Error setting graphics boundary type of '" << boundary_type << "'\n";
        std::cerr << "Defaulting to no boundary\n";
    }
}

void Graphics::DrawBoundary(double **h) {
    glUseProgram(0);

    GLfloat color[4] = {0.8, 0.8, 0.8, 1.0};
    //GLfloat color[4] = {0.0, 0.0, 0.0, 1.0}; //black
    glColor4fv(color);
    if (!params_->graph_boundary_flag)
        return;
    if (boundary_type_.compare("sphere") == 0) {
        glDisable(GL_LIGHTING);
        glDisable(GL_CULL_FACE);

        // Turn on wireframe mode
        DrawWireSphere(0.5*h[0][0], 16, 16);

    }
    else if (boundary_type_.compare("cube") == 0) {
        DrawBox(h);
    }
    
}

void Graphics::DrawWireSphere(double r, int lats, int longs) {
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    glLineWidth(0.5);

    for(int i = 0; i <= lats; i++) {
        double lat0 = M_PI * (-0.5 + (double) (i - 1) / lats);
        double z0  = r*sin(lat0);
        double zr0 =  cos(lat0);
        double lat1 = M_PI * (-0.5 + (double) i / lats);
        double z1 = r*sin(lat1);
        double zr1 = cos(lat1);
        glBegin(GL_QUAD_STRIP);
        for(int j = 0; j <= longs; j++) {
            double lng = 2 * M_PI * (double) (j - 1) / longs;
            double x = r * cos(lng);
            double y = r * sin(lng);
            glNormal3f(x * zr0, y * zr0, z0);
            glVertex3f(x * zr0, y * zr0, z0);
            glNormal3f(x * zr1, y * zr1, z1);
            glVertex3f(x * zr1, y * zr1, z1);
        }
        glEnd();
    }
    // Turn off wireframe mode
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    glEnable(GL_LIGHTING);
    glEnable(GL_CULL_FACE);
}

void Graphics::DrawLoop(int n_bonds, double **h,
                        double **r, double **u, double *length,
                        int n_sites, double **r_site,
                        double sphere_diameter) {
    keep_going_ = 0;
    do { Draw(n_bonds, h, r, u, length, n_sites, r_site, sphere_diameter); }
    while (!keep_going_);
    return;
}

void Graphics::DrawLoop(int n_bonds, double **h,
                        double **r, double **u, double *length,
                        int n_types, xlink_list **xlinks) {
    keep_going_ = 0;
    do { Draw(n_bonds, h, r, u, length, n_types, xlinks); }
    while (!keep_going_);
    return;
}

void Graphics::DrawLoop(int n_bonds, double **h,
                        double **r, double **u, double *length) {
    keep_going_ = 0;
    do { Draw(n_bonds, h, r, u, length); }
    while (!keep_going_);
    return;
}

void Graphics::DrawLoop(int n_bonds, double **h,
                        double **r, double **u, double *length,
                        int n_types, xlink_list *xlinks_0, xlink_list **xlinks_2) {
    keep_going_ = 0;
    do { Draw(n_bonds, h, r, u, length, n_types, xlinks_0, xlinks_2); }
    while (!keep_going_);
    return;
}

void Graphics::DrawLoop(int n_bonds, double **h,
                        double **r, double **u, double *length,
                        int n_types, xlink_list *xlinks_0,
                        xlink_list **xlinks_1,
                        xlink_list **xlinks_2) {
    keep_going_ = 0;
    do { Draw(n_bonds, h, r, u, length, n_types, xlinks_0, xlinks_1, xlinks_2); }
    while (!keep_going_);
    return;
}

// Full spindle drawing type
void Graphics::DrawLoop(int n_bonds,
                        double **h,
                        double **r,
                        double **u,
                        double *length,
                        int n_types,
                        xlink_list *xlinks_0,
                        xlink_list **xlinks_1,
                        xlink_list **xlinks_2,
                        int n_anchors,
                        int anchor_type,
                        float **color_anchor,
                        double **r_anchor,
                        double **u_anchor,
                        double **v_anchor,
                        double **w_anchor,
                        double *adiam,
                        int n_chromosomes,
                        ChromosomeManagement *chromosomes) {
    keep_going_ = 0;
    do { Draw(n_bonds, h, r, u, length, n_types, xlinks_0, xlinks_1, xlinks_2,
            n_anchors, anchor_type, color_anchor, r_anchor, u_anchor, v_anchor, w_anchor, adiam, n_chromosomes, chromosomes); }
    while (!keep_going_);
    return;
}

void Graphics::Draw(int n_bonds, double **h,
                    double **r, double **u, double *length,
                    int n_sites, double **r_site,
                    double sphere_diameter) {
    if (n_dim_ == 2)
        Draw2d(n_bonds, h, r, u, length);        
    else if (n_dim_ == 3) 
        Draw3d(n_bonds, h, r, u, length, n_sites, r_site, sphere_diameter);
    
}

void Graphics::Draw(int n_bonds, double **h,
                    double **r, double **u, double *length) {
    if (n_dim_ == 2)
        Draw2d(n_bonds, h, r, u, length);
    else if (n_dim_ == 3)
        Draw3d(n_bonds, h, r, u, length);
}

void Graphics::Draw(int n_bonds, double **h,
                    double **r, double **u, double *length,
                    int n_types, xlink_list **xlinks) {
    if (n_dim_ == 2)
        Draw2d(n_bonds, h, r, u, length, n_types, xlinks);
    else if (n_dim_ == 3)
        Draw3d(n_bonds, h, r, u, length, n_types, xlinks);
}


void Graphics::Draw(int n_bonds, double **h,
                    double **r, double **u, double *length,
                    int n_types, xlink_list *xlinks_0, xlink_list **xlinks_2) {
    if (n_dim_ == 2)
        Draw2d(n_bonds, h, r, u, length, n_types, xlinks_2); //FIXME: need 2d version
    else if (n_dim_ == 3)
        Draw3d(n_bonds, h, r, u, length, n_types, xlinks_0, xlinks_2);
}

void Graphics::Draw(int n_bonds, double **h,
                    double **r, double **u, double *length,
                    int n_types, xlink_list *xlinks_0,
                    xlink_list **xlinks_1, xlink_list **xlinks_2) {
    if (n_dim_ == 2)
        Draw2d(n_bonds, h, r, u, length, n_types, xlinks_2); //FIXME: need 2d version
    else if (n_dim_ == 3)
        Draw3d(n_bonds, h, r, u, length, n_types, xlinks_0, xlinks_1, xlinks_2);
}

// Graphics draw command for 3d spindle stuff ONLY!!!!
void Graphics::Draw(int n_bonds,
                    double **h,
                    double **r_bond,
                    double **u_bond,
                    double *rlength,
                    int n_types,
                    xlink_list *xlinks0,
                    xlink_list **xlinks1,
                    xlink_list **xlinks2,
                    int n_anchors,
                    int anchor_type,
                    float **color_anchor,
                    double **r_anchor,
                    double **u_anchor,
                    double **v_anchor,
                    double **w_anchor,
                    double *adiam,
                    int n_chromosomes,
                    ChromosomeManagement *chromosomes) {
    if (n_dim_ == 3)
        Draw3d(n_bonds,
               h,
               r_bond,
               u_bond,
               rlength,
               n_types,
               xlinks0,
               xlinks1,
               xlinks2,
               n_anchors,
               anchor_type,
               color_anchor,
               r_anchor,
               u_anchor,
               v_anchor,
               w_anchor,
               adiam,
               n_chromosomes,
               chromosomes);
}


void Graphics::DrawSquare(double **h) {
    glUseProgram(0);

    /* Draw the bounding box. */
    GLfloat box_color[4] = {1.0, 0.0, 0.5, 1.0};
    GLfloat v0, v1;
    GLfloat w = 2;
    glLineWidth(w);
    glColor4fv(box_color);
    glBegin(GL_LINE_LOOP);
    v0 = -h[0][0] / 2 - h[0][1] / 2;
    v1 = -h[1][1] / 2 - h[1][0] / 2;
    glVertex2f(v0, v1);
    v0 = -h[0][0] / 2 + h[0][1] / 2;
    v1 = h[1][1] / 2 - h[1][0] / 2;
    glVertex2f(v0, v1);
    v0 = h[0][0] / 2 + h[0][1] / 2;
    v1 = h[1][1] / 2 + h[1][0] / 2;
    glVertex2f(v0, v1);
    v0 = h[0][0] / 2 - h[0][1] / 2;
    v1 = -h[1][1] / 2 + h[1][0] / 2;
    glVertex2f(v0, v1);
    v0 = -h[0][0] / 2 - h[0][1] / 2;
    v1 = -h[1][1] / 2 - h[1][0] / 2;
    glVertex2f(v0, v1);
    glEnd();
}

void Graphics::DrawCube(double **h) {
    int w;
    GLfloat v0, v1, v2;

    /* Turn off the light. */
    glDisable(GL_LIGHTING);

    /* Line thickness. */
    w = 3;
    glLineWidth((GLfloat) w);

    /* Draw the bounding box. */
    /* front face */
    glBegin(GL_LINE_LOOP);
    v0 = 0.5 * (-h[0][0] - h[0][1] - h[0][2]);
    v1 = 0.5 * (-h[1][0] - h[1][1] - h[1][2]);
    v2 = 0.5 * (-h[2][0] - h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] - h[0][1] - h[0][2]);
    v1 = 0.5 * (h[1][0] - h[1][1] - h[1][2]);
    v2 = 0.5 * (h[2][0] - h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] + h[0][1] - h[0][2]);
    v1 = 0.5 * (h[1][0] + h[1][1] - h[1][2]);
    v2 = 0.5 * (h[2][0] + h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (-h[0][0] + h[0][1] - h[0][2]);
    v1 = 0.5 * (-h[1][0] + h[1][1] - h[1][2]);
    v2 = 0.5 * (-h[2][0] + h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (-h[0][0] - h[0][1] - h[0][2]);
    v1 = 0.5 * (-h[1][0] - h[1][1] - h[1][2]);
    v2 = 0.5 * (-h[2][0] - h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (-h[0][0] - h[0][1] + h[0][2]);
    v1 = 0.5 * (-h[1][0] - h[1][1] + h[1][2]);
    v2 = 0.5 * (-h[2][0] - h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] - h[0][1] + h[0][2]);
    v1 = 0.5 * (h[1][0] - h[1][1] + h[1][2]);
    v2 = 0.5 * (h[2][0] - h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] - h[0][1] - h[0][2]);
    v1 = 0.5 * (h[1][0] - h[1][1] - h[1][2]);
    v2 = 0.5 * (h[2][0] - h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    glEnd();
    glBegin(GL_LINE_LOOP);
    v0 = 0.5 * (-h[0][0] + h[0][1] - h[0][2]);
    v1 = 0.5 * (-h[1][0] + h[1][1] - h[1][2]);
    v2 = 0.5 * (-h[2][0] + h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (-h[0][0] + h[0][1] + h[0][2]);
    v1 = 0.5 * (-h[1][0] + h[1][1] + h[1][2]);
    v2 = 0.5 * (-h[2][0] + h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] + h[0][1] + h[0][2]);
    v1 = 0.5 * (h[1][0] + h[1][1] + h[1][2]);
    v2 = 0.5 * (h[2][0] + h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] + h[0][1] - h[0][2]);
    v1 = 0.5 * (h[1][0] + h[1][1] - h[1][2]);
    v2 = 0.5 * (h[2][0] + h[2][1] - h[2][2]);
    glVertex3f(v0, v1, v2);
    glEnd();
    glBegin(GL_LINE_LOOP);
    v0 = 0.5 * (-h[0][0] - h[0][1] + h[0][2]);
    v1 = 0.5 * (-h[1][0] - h[1][1] + h[1][2]);
    v2 = 0.5 * (-h[2][0] - h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (-h[0][0] + h[0][1] + h[0][2]);
    v1 = 0.5 * (-h[1][0] + h[1][1] + h[1][2]);
    v2 = 0.5 * (-h[2][0] + h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    glEnd();
    glBegin(GL_LINE_LOOP);
    v0 = 0.5 * (h[0][0] - h[0][1] + h[0][2]);
    v1 = 0.5 * (h[1][0] - h[1][1] + h[1][2]);
    v2 = 0.5 * (h[2][0] - h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    v0 = 0.5 * (h[0][0] + h[0][1] + h[0][2]);
    v1 = 0.5 * (h[1][0] + h[1][1] + h[1][2]);
    v2 = 0.5 * (h[2][0] + h[2][1] + h[2][2]);
    glVertex3f(v0, v1, v2);
    glEnd();

    /* Turn on light. */
    glEnable(GL_LIGHTING);
}

void Graphics::DrawBox(double **h) {
    if (n_dim_ == 2)
        DrawSquare(h);
    else if (n_dim_ == 3)
        DrawCube(h);
}

void Graphics::ResizeWindow(int width, int height) {
    glfwSetWindowSize(window_, width, height);
}

void Graphics::UpdateWindow() {
    /* Recalculate the view */
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    if (n_dim_ == 2) {
        glLoadIdentity();
        glTranslatef(xTrans_, yTrans_, 0.0);
    }
    else if (n_dim_ == 3) {
        if (tomogram_view_ == 0) {
            glTranslatef(0.0, 0.0, zTrans_);
            glRotatef(xAngle_, 1.0, 0.0, 0.0);
            glRotatef(yAngle_, 0.0, 1.0, 0.0);
            glRotatef(zAngle_, 0.0, 0.0, 1.0);
        } else if (tomogram_view_ >= 1) {
            // SPB centric planar view OR chromosome centric planar view
            glTranslatef(0.0, 0.0, zTrans_);

            // Version 5 uses 2 quaternions to do the rotation and alignment
            double q1[4] = {0.0};
            double q2[4] = {0.0};
            double q3[4] = {0.0};
            double zaxis[3] = {0.0, 0.0, 1.0};
            double nyaxis[3] = {0.0, -1.0, 0.0};
            double xaxis[3] = {1.0, 0.0, 0.0};
            double theta;
            double eaxis[3] = {0.0};

            // Generate first quaternion between tomo and z axis (aligns spindle into the plane of view)
            quaternion_between_vectors(q1, ntomo_, zaxis);
            normalize_quaternion(q1);

            // Get the axis angle to rotate the spb0 point into this new frame
            axisangle_from_quaternion(&theta, eaxis, q1);

            double vrot[3];
            rodrigues_axisangle(vrot, rhatspb_, theta, eaxis);
            quaternion_between_vectors(q2, vrot, xaxis);
            normalize_quaternion(q2);

            // Multiply the quaternions
            quaternion_multiply(q3, q2, q1);
            normalize_quaternion(q3);
            if (!first_call_quaternion_ && !std::isnan(q3[0])) {
                first_call_quaternion_ = true;
                quaternion_old_[0] = q3[0];
                quaternion_old_[1] = q3[1];
                quaternion_old_[2] = q3[2];
                quaternion_old_[3] = q3[3];

                //print_quaternion(q1);
                //print_quaternion(q2);
                //print_quaternion(q3);
            }
            // Determine if the rotation amount between them is okay
            //print_quaternion(q3);
            double qr[4] = {0.0};
            rotate_towards(quaternion_old_, q3, 3.14f*0.01, qr);
            normalize_quaternion(qr);
            quaternion_old_[0] = qr[0];
            quaternion_old_[1] = qr[1];
            quaternion_old_[2] = qr[2];
            quaternion_old_[3] = qr[3];

            //print_quaternion(qr);

            // Get final, composite rotation
            axisangle_from_quaternion(&theta, eaxis, qr);

            glRotatef(theta/M_PI*180.0, eaxis[0], eaxis[1], eaxis[2]);

            // This has all been done with the SPB/chromosome, but what about focusing and bringing it to the center?
            if (tomogram_view_ > 1) {
                // Get the information about the chormosome position...
                // Also, zoom in on that sucker
                xyzScale_ = 3.0;
                glTranslatef(-rchromosome_[0]*xyzScale_, -rchromosome_[1]*xyzScale_, -rchromosome_[2]*xyzScale_);
            }
        }
    }
    glScalef(xyzScale_, xyzScale_, xyzScale_);

    /* Set square aspect ratio from current window size */
    glfwGetFramebufferSize(window_, &width_fb_, &height_fb_);
    
    int minsize = MIN(width_fb_, height_fb_);
    glViewport((width_fb_ - minsize)/2, (height_fb_ - minsize)/2, minsize, minsize);

    /* Clear background */
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

/* Make a gl buffer */
GLuint GraphicsPrimitive::MakeBuffer(GLenum target,
                                     const void *buffer_data,
                                     GLsizei buffer_size) {
    GLuint buffer;
    glGenBuffers(1, &buffer);
    glBindBuffer(target, buffer);
    glBufferData(target, buffer_size, buffer_data, GL_STATIC_DRAW);
    return buffer;
}

void GraphicsPrimitive::MakeVertexBuffer() {
    vertex_buffer_ = MakeBuffer(GL_ARRAY_BUFFER, vertex_buffer_data_, sizeof(GLfloat)*n_coords_);
}

void GraphicsPrimitive::MakeElementBuffer() {
    element_buffer_ = MakeBuffer(GL_ELEMENT_ARRAY_BUFFER, element_buffer_data_,
                                 sizeof(GLushort) * n_triangles_ * 3);
}


/* Displays what part of our vertex shader failed to compile. This should never be used
   since the shader is hardcoded. */
void GraphicsPrimitive::ShowInfoLog(GLuint object,
                                    PFNGLGETSHADERIVPROC glGet__iv,
                                    PFNGLGETSHADERINFOLOGPROC glGet__InfoLog) {
    GLint log_length;
    char *log;

    glGet__iv(object, GL_INFO_LOG_LENGTH, &log_length);
    log = new char[log_length];
    glGet__InfoLog(object, log_length, NULL, log);
    fprintf(stderr, "%s", log);
    delete log;
}

void Graphics::InitDiscoRectangle() {
    discorectangle_.n_coords_ = (9 + 1)*4;
    discorectangle_.n_triangles_ = (8*2 + 2);
    discorectangle_.vertex_buffer_data_ = new GLfloat[discorectangle_.n_coords_];
    discorectangle_.element_buffer_data_ = new GLushort[discorectangle_.n_triangles_*3];
    int i;
    GLfloat dphi = M_PI / 8.0;
    GLfloat phi;
    int index = 0;
    discorectangle_.vertex_buffer_data_[index+0] = 0.0;
    discorectangle_.vertex_buffer_data_[index+1] = 1e-6;
    index += 2;

    phi = 0.0;
    for (i = 0; i < 9; ++i) {
        discorectangle_.vertex_buffer_data_[index+0] = 0.5 * cos(phi);
        discorectangle_.vertex_buffer_data_[index+1] = 0.5 * sin(phi) + 1e-6;
        index += 2;
        phi += dphi;
    }

    int k = 0;
    for (i = 1; i < 9; ++i) {
        discorectangle_.element_buffer_data_[k ] = 0;
        discorectangle_.element_buffer_data_[k+1] = i;
        discorectangle_.element_buffer_data_[k+2] = i+1;
        k+=3;
    }

    discorectangle_.vertex_buffer_data_[index+0] = 0.0;
    discorectangle_.vertex_buffer_data_[index+1] = -1e-6;

    int center_vert = index / 2;
    index += 2;
    phi = M_PI;
    for (i = 0; i < 9; ++i) {
        discorectangle_.vertex_buffer_data_[index+0] = 0.5 * cos(phi);
        discorectangle_.vertex_buffer_data_[index+1] = 0.5 * sin(phi) - 1e-6;
        index += 2;
        phi += dphi;
    }

    for (i = 1; i < 9; ++i) {
        discorectangle_.element_buffer_data_[k] = center_vert;
        discorectangle_.element_buffer_data_[k+1] = center_vert + i;
        discorectangle_.element_buffer_data_[k+2] = center_vert + i + 1;
        k+=3;
    }

    discorectangle_.element_buffer_data_[k] = 1;
    discorectangle_.element_buffer_data_[k+1] = center_vert + 1;
    discorectangle_.element_buffer_data_[k+2] = center_vert + 9;
    k+=3;
    discorectangle_.element_buffer_data_[k] = 9;
    discorectangle_.element_buffer_data_[k+1] = center_vert + 1;
    discorectangle_.element_buffer_data_[k+2] = 1;
    k+=3;

    discorectangle_.MakeVertexBuffer();
    discorectangle_.MakeElementBuffer();

    /* Vertex shader source code */
    const char *vs =
        "#version 120\n"
        "attribute vec2 position;"
        "uniform float half_l;"
        "uniform float diameter;"
        "void main()"
        "{"
        "    vec4 stretched_pos;"
        "    if (position.y >= 0.0)"
        "        stretched_pos = vec4(diameter * position,0.0,1.0) + vec4(0.0,half_l,0.0,0.0);"
        "    else"
        "        stretched_pos = vec4(diameter * position,0.0,1.0) - vec4(0.0,half_l,0.0,0.0);"
        "    gl_FrontColor = gl_Color;"
        "    gl_Position = gl_ModelViewProjectionMatrix * (stretched_pos);"
        "} "
        "";

    GLint shader_ok;
    discorectangle_.vertex_shader_ = glCreateShader(GL_VERTEX_SHADER);
    glShaderSource(discorectangle_.vertex_shader_, 1, &vs, NULL);
    glCompileShader(discorectangle_.vertex_shader_);
    glGetShaderiv(discorectangle_.vertex_shader_, GL_COMPILE_STATUS, &shader_ok);
    if (!shader_ok) {
        fprintf(stderr, "Failed to compile vertex shader:\n");
        discorectangle_.ShowInfoLog(discorectangle_.vertex_shader_, glGetShaderiv, glGetShaderInfoLog);
        glDeleteShader(discorectangle_.vertex_shader_);
        exit(1);
    }
    if (discorectangle_.vertex_shader_ == 0)
        exit(1);

    /* Fragment shader source code */
    const char *fs =
        "#version 120\n"
        "void main()"
        "{"
        "gl_FragColor = gl_Color;"
        "}";

    discorectangle_.fragment_shader_ = glCreateShader(GL_FRAGMENT_SHADER);
    glShaderSource(discorectangle_.fragment_shader_, 1, &fs, NULL);
    glCompileShader(discorectangle_.fragment_shader_);
    glGetShaderiv(discorectangle_.fragment_shader_, GL_COMPILE_STATUS, &shader_ok);
    if (!shader_ok) {
        fprintf(stderr, "Failed to compile fragment shader:\n");
        discorectangle_.ShowInfoLog(discorectangle_.fragment_shader_, glGetShaderiv, glGetShaderInfoLog);
        glDeleteShader(discorectangle_.fragment_shader_);
    }
    if (discorectangle_.fragment_shader_ == 0)
        exit(1);

    discorectangle_.MakeProgram();

    /* Get integer locations of relevant shader parameters */
    discorectangle_.uniforms_.half_l
        = glGetUniformLocation(discorectangle_.program_, "half_l");
    discorectangle_.uniforms_.diameter
        = glGetUniformLocation(discorectangle_.program_, "diameter");
    discorectangle_.attributes_.position
        = glGetAttribLocation(discorectangle_.program_, "position");
}

void Graphics::InitSpheroCylinder() {
    /* Create appropriate sized buffer objects. */
    spherocylinder_.n_coords_ = (16*16 + 2)*3;
    spherocylinder_.n_triangles_ = (32*16);
    spherocylinder_.vertex_buffer_data_ = new GLfloat[spherocylinder_.n_coords_];
    spherocylinder_.element_buffer_data_ = new GLushort[spherocylinder_.n_triangles_ * 3];

    int i, j;
    double theta, phi;
    double dtheta = M_PI / 16.0;
    double dphi = 2.0 * M_PI / 16.0;
    double x,y,z;
    int index = 3;

    spherocylinder_.vertex_buffer_data_[0] = 0.0;
    spherocylinder_.vertex_buffer_data_[1] = 0.0;
    spherocylinder_.vertex_buffer_data_[2] = 0.5;

    theta = dtheta; phi = 0.0;

    for (i = 0; i < 8; ++i) {
        phi = 0.0;
        z = 0.5*cos(theta) + 1e-6;
        for (j = 0; j < 16; ++j) {
            x = 0.5*cos(phi) * sin(theta);
            y = 0.5*sin(phi) * sin(theta);

            spherocylinder_.vertex_buffer_data_[index+0] = x;
            spherocylinder_.vertex_buffer_data_[index+1] = y;
            spherocylinder_.vertex_buffer_data_[index+2] = z;
            phi += dphi;
            index += 3;
        }
        theta += dtheta;
    }

    theta -= dtheta;
    for (i = 0; i < 8; ++i) {
        phi = 0.0;
        z =  0.5 * cos(theta) - 1e-6;
        for (j = 0; j < 16; ++j) {
            x = 0.5*cos(phi) * sin(theta);
            y = 0.5*sin(phi) * sin(theta);

            spherocylinder_.vertex_buffer_data_[index+0] = x;
            spherocylinder_.vertex_buffer_data_[index+1] = y;
            spherocylinder_.vertex_buffer_data_[index+2] = z;

            phi += dphi;
            index += 3;
        }
        theta += dtheta;
    }

    spherocylinder_.vertex_buffer_data_[index] = 0.0;
    spherocylinder_.vertex_buffer_data_[index+1] = 0.0;
    spherocylinder_.vertex_buffer_data_[index+2] = -0.5;
    index += 3;

    int offset = 0;
    int k = 0;
    for (i = 0; i < 15; ++i) {
        spherocylinder_.element_buffer_data_[k  ] = 0;
        spherocylinder_.element_buffer_data_[k+1] = 1 + i;
        spherocylinder_.element_buffer_data_[k+2] = 2 + i;
        k += 3;
    }
    spherocylinder_.element_buffer_data_[k  ] = 0;
    spherocylinder_.element_buffer_data_[k+1] = 16;
    spherocylinder_.element_buffer_data_[k+2] = 1;
    k += 3;

    for (j = 0; j < 15; ++j) {
        for (i = 1; i < 16; ++i) {
            spherocylinder_.element_buffer_data_[k  ] = i + j*16;
            spherocylinder_.element_buffer_data_[k+1] = i+16 + j*16;
            spherocylinder_.element_buffer_data_[k+2] = i+17 + j*16;
            k += 3;
            spherocylinder_.element_buffer_data_[k  ] = i+1 + j*16;
            spherocylinder_.element_buffer_data_[k+1] = i + j*16;
            spherocylinder_.element_buffer_data_[k+2] = i+17 + j*16;
            k += 3;
        }
        i = 16;
        spherocylinder_.element_buffer_data_[k  ] = i + j*16;
        spherocylinder_.element_buffer_data_[k+1] = i+16 + j*16;
        spherocylinder_.element_buffer_data_[k+2] = i+1 + j*16;
        k += 3;
        spherocylinder_.element_buffer_data_[k  ] = i-15 + j*16;
        spherocylinder_.element_buffer_data_[k+1] = i + j*16;
        spherocylinder_.element_buffer_data_[k+2] = i+1 + j*16;
        k+=3;
    }

    offset = spherocylinder_.n_coords_ / 3 - 17;
    for (i = offset; i < offset+15; ++i) {
        spherocylinder_.element_buffer_data_[k  ] = spherocylinder_.n_coords_ / 3 -1;
        spherocylinder_.element_buffer_data_[k+1] = 1 + i;
        spherocylinder_.element_buffer_data_[k+2] = 0 + i;

        k += 3;
    }
    spherocylinder_.element_buffer_data_[k  ] = spherocylinder_.n_coords_ / 3 - 1;
    spherocylinder_.element_buffer_data_[k+1] = offset;
    spherocylinder_.element_buffer_data_[k+2] = spherocylinder_.n_coords_ / 3 - 2;

    spherocylinder_.MakeVertexBuffer();

    /* It seems that just using the vertex buffer for normal information (since this starts out as a sphere) makes
       a much prettier spherocylinder */
    spherocylinder_.MakeElementBuffer();

    /* Vertex shader source code */
    const char *vs =
        "#version 120\n"
        "attribute vec4 position;"
        "uniform float half_l;"
        "uniform float diameter;"
        "varying vec4 diffuse,ambientGlobal,ambient;"
        "varying vec3 normal,lightDir,halfVector;"
        "varying float dist;"
        "void main()"
        "{"
        "    vec4 ecPos;"
        "    vec3 aux;"
        "    vec4 stretched_pos;"
        "    normal = normalize(gl_NormalMatrix * position.xyz);"
        "    if (position.z >= 0.0)"
        "        stretched_pos = diameter*position + vec4(0.0,0.0,half_l,0.0);"
        "    else"
        "        stretched_pos = diameter*position - vec4(0.0,0.0,half_l,0.0);"
        "    stretched_pos.w = position.w;"
        "    /* these are the new lines of code to compute the light's direction */"
        "    ecPos = gl_ModelViewMatrix * (stretched_pos);"
        "    aux = vec3(gl_LightSource[0].position-ecPos);"
        "    lightDir = normalize(aux);"
        "    dist = length(aux);"
        "    halfVector = normalize(gl_LightSource[0].halfVector.xyz);"
        ""
        "    /* Compute the diffuse, ambient and globalAmbient terms */"
        "    diffuse = gl_FrontMaterial.diffuse * gl_LightSource[0].diffuse;"
        ""
        "    /* The ambient terms have been separated since one of them */"
        "    /* suffers attenuation */"
        "    ambient = gl_FrontMaterial.ambient * gl_LightSource[0].ambient;"
        "    ambientGlobal = gl_LightModel.ambient * gl_FrontMaterial.ambient;"
        ""
        "    gl_Position = gl_ModelViewProjectionMatrix * (stretched_pos);"
        "} "
        "";

    /* Build vertex shader from source specified in vs string */
    GLint shader_ok;
    spherocylinder_.vertex_shader_ = glCreateShader(GL_VERTEX_SHADER);
    glShaderSource(spherocylinder_.vertex_shader_, 1, &vs, NULL);
    glCompileShader(spherocylinder_.vertex_shader_);
    glGetShaderiv(spherocylinder_.vertex_shader_, GL_COMPILE_STATUS, &shader_ok);
    if (!shader_ok) {
        fprintf(stderr, "Failed to compile vertex shader:\n");
        spherocylinder_.ShowInfoLog(spherocylinder_.vertex_shader_, glGetShaderiv, glGetShaderInfoLog);
        glDeleteShader(spherocylinder_.vertex_shader_);
        exit(1);
    }
    if (spherocylinder_.vertex_shader_ == 0)
        exit(1);

    /* Fragment shader source code */
    const char *fs =
        "#version 120\n"
        "varying vec4 diffuse,ambientGlobal, ambient;"
        "varying vec3 normal,lightDir,halfVector;"
        "varying float dist;"
        "void main()"
        "{"
        "vec3 n,halfV,viewV,ldir;"
        "float NdotL,NdotHV;"
        "vec4 color = ambientGlobal;"
        "float att;"
        "/* a fragment shader can't write a varying variable, hence we need"
        "a new variable to store the normalized interpolated normal */"
        "n = normalize(normal);"
        ""
        "        /* compute the dot product between normal and normalized lightdir */"
        "NdotL = max(dot(n,normalize(lightDir)),0.0);"
	""
        "if (NdotL > 0.0) {"
        ""
        "att = 1.0 / (gl_LightSource[0].constantAttenuation +"
        "gl_LightSource[0].linearAttenuation * dist +"
        "gl_LightSource[0].quadraticAttenuation * dist * dist);"
        "color += att * (diffuse * NdotL + ambient);"
        "}"
        ""
        "gl_FragColor = color;"
        "}";

    /* Build Fragment Shader from source specified in fs string */
    spherocylinder_.fragment_shader_ = glCreateShader(GL_FRAGMENT_SHADER);
    glShaderSource(spherocylinder_.fragment_shader_, 1, &fs, NULL);
    glCompileShader(spherocylinder_.fragment_shader_);
    glGetShaderiv(spherocylinder_.fragment_shader_, GL_COMPILE_STATUS, &shader_ok);
    if (!shader_ok) {
        fprintf(stderr, "Failed to compile fragment shader:\n");
        spherocylinder_.ShowInfoLog(spherocylinder_.fragment_shader_, glGetShaderiv, glGetShaderInfoLog);
        glDeleteShader(spherocylinder_.fragment_shader_);
        exit(1);
    }

    spherocylinder_.MakeProgram();

    /* Get integer locations of relevant shader parameters */
    spherocylinder_.uniforms_.half_l
        = glGetUniformLocation(spherocylinder_.program_, "half_l");
    spherocylinder_.uniforms_.diameter 
        = glGetUniformLocation(spherocylinder_.program_, "diameter");
    spherocylinder_.attributes_.position
        = glGetAttribLocation(spherocylinder_.program_, "position");
}

// Create a kinetochore mesh program
void Graphics::InitKinetochoreMesh(triangle_mesh *example_mesh) {

    return;

    // Return if we don't have an example mesh to draw from
    if (!example_mesh) {
        return;
    }

    /* Create appropriate sized buffer objects. */
    kinetochoremesh_.n_coords_ = (example_mesh->numVerts)*3;
    kinetochoremesh_.n_triangles_ = (example_mesh->numTriang);
    kinetochoremesh_.vertex_buffer_data_ = new GLfloat[kinetochoremesh_.n_coords_];
    kinetochoremesh_.element_buffer_data_ = new GLushort[kinetochoremesh_.n_triangles_ * 3];

    // Basically, this is going to be hardcoded for hexagons (for now)
    if (example_mesh->numVerts == 6) {
        
    } else {
        std::cerr << "Kinetochore mesh not possible yet\n";
        std::cerr << "  numVerts  = " << example_mesh->numVerts << std::endl;
        std::cerr << "  numTriang = " << example_mesh->numTriang << std::endl;
        exit(1);
    }
}

/* Initialize our custom shader program for spherocylinders */
void GraphicsPrimitive::MakeProgram() {
    program_ = glCreateProgram();
    glAttachShader(program_, vertex_shader_);
    glAttachShader(program_, fragment_shader_);
    glLinkProgram(program_);

    GLint program_ok;
    glGetProgramiv(program_, GL_LINK_STATUS, &program_ok);
    if (!program_ok) {
        fprintf(stderr, "Failed to link shader program:\n");
        ShowInfoLog(program_, glGetProgramiv, glGetProgramInfoLog);
        glDeleteProgram(program_);
        exit(1);
    }
}

void GraphicsPrimitive::PrepDraw() {
    /* Use vertex/fragment custom shader for spherocylinder */
    glUseProgram(program_);
    /* Get location of half_l parameter, this shouldn't need to be done everytime
       but for whatever reason it wasn't grabbing the correct location at initialization */
    uniforms_.half_l
        = glGetUniformLocation(program_, "half_l");
    uniforms_.diameter
        = glGetUniformLocation(program_, "diameter");
    /* Prep to send vertex data to shader */
    glEnableVertexAttribArray(attributes_.position);
    glBindBuffer(GL_ARRAY_BUFFER, vertex_buffer_);
    glVertexAttribPointer(attributes_.position, // attribute
                          3,                 // number of elements per vertex, here (x,y,z)
                          GL_FLOAT,          // the type of each element
                          GL_FALSE,          // take our values as-is
                          0,                 // no extra data between each position
                          (void*)0                  // offset of first element
                          );
    
    /* Use the element buffer (numerical pointer to each set of vertices that make a triangle) */
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_buffer_);
}

void GraphicsPrimitive::Draw(GLfloat v0, GLfloat v1, GLfloat v2) {
    /* Make copy of modelview matrix to work on */
    glPushMatrix();

    /* Move sphere origin to new position */
    glTranslatef(v0, v1, v2);

    glDrawElements(GL_TRIANGLES,
                   n_triangles_*3,
                   GL_UNSIGNED_SHORT,
                   (void*) 0);

    /* Reset modelview matrix */
    glPopMatrix();
}


void GraphicsPrimitive::Draw(GLfloat v0, GLfloat v1, GLfloat v2,
                             GLfloat theta, GLfloat phi) {
    /* Make copy of modelview matrix to work on */
    glPushMatrix();

    /* Move sphere origin to new position */
    glTranslatef(v0, v1, v2);

    if (theta != 0.0)
        glRotatef((GLfloat) ((theta / M_PI) * 180.0), 0.0, 0.0, 1.0);
    if (phi != 0.0)
        glRotatef((GLfloat) ((phi / M_PI) * 180.0), 0.0, 1.0, 0.0);
    
    glDrawElements(GL_TRIANGLES,
                   n_triangles_*3,
                   GL_UNSIGNED_SHORT,
                   (void*) 0);
    
    /* Reset modelview matrix */
    glPopMatrix();
}

#endif

