/* Initialize variables and allocate memory for multithreaded 
   spherocylinder brownian dynamics simulation.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)
   
   Output: memory for a variety of arrays is allocated,
   and a number of variables are initialized. */

#include "bob.h"
#include "xlink_management.h"


void init_crosslink_sphero_bd_mp(system_parameters *parameters,
                                 system_properties *properties,
                                 system_potential *potential) {

    /* Set up pointers to header read and write functions. */
    properties->read_header_func = read_header_sphero;
    properties->write_header_func = write_header_sphero;

    /* Read initial configuration from input file. */

    properties->time = 0.0;

    /* Turn off periodic boundary conditions for each wall in the system. 
     * If neither n_periodic or n_walls are initialized to have all 
     * dimensions be periodic.*/
    // Note: CJE FIXME Eric needs to understand this so that we don't wind up with
    // accidental periodic boundary conditions for restarts with tactoids
    if (parameters->n_periodic <=0 && parameters->n_walls <=0){
        parameters->n_periodic = parameters->n_dim;
        parameters->n_walls = 0;
    }
    else if (parameters->n_walls <= 0)
        parameters->n_walls = parameters->n_dim - parameters->n_periodic;
    else
        parameters->n_periodic = parameters->n_dim - parameters->n_walls;

    if ( !parameters->xlink_test_flag ){
        read_config(parameters, properties, parameters->config_file);
        init_unit_cell_structure(parameters, properties);
        init_site_structure_sphero(parameters, properties);
        init_bond_structure_sphero(parameters, properties);
    }
    else{
        char * xtf = "xlink_analysis.yaml";
        configurator_generic(parameters, properties, xtf);
    }
    
    /* Initialize control structure. */
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

    init_diffusion_sphero(parameters, properties);

    /* Initialize crosslink structure if crosslinks are turned on. */
    if (parameters->crosslink_flag)
        properties->crosslinks.Init(parameters, properties,
                                    parameters->crosslink_file,
                                    NULL);

    /* Initialize dynamic instability if DI is turned on*/
    if (parameters->dynamic_instability_flag == 1) {
        init_dynamic_instability(parameters, properties);
    }

    /* Load OMP settings */
    set_omp_settings(parameters);
    print_omp_settings();

    /* Initialize GSL random number generator state */
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    #ifdef ENABLE_OPENMP
    #pragma omp parallel 
    #pragma omp master
    #endif
    {
        int n_threads = properties->mp_local.n_threads = properties->rng_mt.n_threads = 1;
            
        #ifdef ENABLE_OPENMP
        n_threads = properties->mp_local.n_threads = properties->rng_mt.n_threads = omp_get_num_threads();
        #endif

        properties->rng_mt.rng = (rng_properties*) 
            allocate_1d_array(properties->rng_mt.n_threads, sizeof(rng_properties));

        gsl_rng_env_setup();
        for (int i_thr = 0; i_thr < properties->rng_mt.n_threads; ++i_thr) {
            properties->rng_mt.rng[i_thr].T = gsl_rng_default;
            properties->rng_mt.rng[i_thr].r = gsl_rng_alloc(properties->rng_mt.rng[i_thr].T);
            gsl_rng_set(properties->rng_mt.rng[i_thr].r, (parameters->seed+1) * i_thr);
        }

        properties->mp_local.f_local = (double***)
            allocate_3d_array(n_threads, properties->bonds.n_bonds, parameters->n_dim, sizeof(double));

        properties->mp_local.t_local = (double***)
            allocate_3d_array(n_threads, properties->bonds.n_bonds, 3, sizeof(double));

        properties->mp_local.virial_local = (double***)
            allocate_3d_array(n_threads, parameters->n_dim, parameters->n_dim, sizeof(double));
        
        properties->mp_local.calc_local = (int**)
            allocate_2d_array(n_threads, properties->bonds.n_bonds, sizeof(int));
    }
    

    /* Initialize local order measurement array */
    properties->bonds.local_order = (double *) allocate_1d_array(properties->bonds.n_bonds, sizeof(double));

    /* Allocate memory for arrays in thermodynamics structure. */
    properties->thermo.virial = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.stress = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.press_tensor = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));

    /* Set r_cutoff to WCA interaction range */
    parameters->r_cutoff_lj = parameters->r_cutoff = pow(2.0, 1.0 / 6.0);

    /* Calculate r_cutoff for crosslink partition function  */

    properties->bonds.length_max = 0.0;
    /* Find maximum length of bond in system */
    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        properties->bonds.length_max =
            MAX(properties->bonds.length_max, properties->bonds.length[i_bond]);
    }

    if (parameters->crosslink_flag) {
        parameters->r_cutoff 
            = MAX(parameters->r_cutoff, *(properties->crosslinks.r_cutoff_1_2_));
        printf("   r_cutoff = %g\n", parameters->r_cutoff);
        if (parameters->r_cutoff != parameters->r_cutoff) {
            fprintf(stderr, "\nUnable to calculate r_cutoff. Consider lowering the spring constant of the crosslinks "
                    "or increasing their concentration\n");
            exit(1);
        }
    }

    {
        double half_min_lin = 0.5 * properties->unit_cell.a_perp[0];
        for (int i = 1; i < parameters->n_periodic; ++i)
            half_min_lin = MIN(half_min_lin, 0.5 * properties->unit_cell.a_perp[i]);
        
        if (parameters->r_cutoff + properties->bonds.length_max > half_min_lin) {
            fprintf(stderr,
                    "interaction range plus maximum length exceeds half the "
                    "minimum linear dimension of the unit cell\n");
            fprintf(stderr, "r_cutoff + length_max = %g, half_min_lin = %g\n\n",
                    parameters->r_cutoff + properties->bonds.length_max, half_min_lin);
            exit(1);
        }
    }
    
    /* Init potential structure */
    init_potential_crosslink_sphero_bd_mp(parameters, properties, potential);

    /* Initialize cell and/or neighbor list structures. */
    properties->n_cell_lists = 0;
    parameters->nl_cell_list = 0;
    if (parameters->neighb_switch == 1 && parameters->nl_update_flag == 1) {
        ++properties->n_cell_lists;
    } else if (parameters->neighb_switch == 2) {
        fprintf(stderr, "Cell list only search not supported. Set neighb_switch option to 1\n");
        exit(1);
    }

    if (properties->n_cell_lists > 0)
        properties->cells = new cell_list[properties->n_cell_lists];
    else
        properties->cells = NULL;
    
    if (parameters->neighb_switch == 1) {
        /* Initialize neighbor lists. */
        properties->neighbors.neighbs = new nl_list[properties->bonds.n_bonds];

        periodic_boundary_conditions(parameters->n_periodic, properties->bonds.n_bonds,
                                     properties->unit_cell.h, properties->unit_cell.h_inv,
                                     properties->bonds.r_bond, properties->bonds.s_bond);

        if (parameters->nl_update_flag == 0) {       /* All-pairs search. */
            error_exit("all pairs neighbor list updated not supported for mp, use nl_update_flag = 1\n");
        }
        else if (parameters->nl_update_flag == 1) {                  /* Cell search. */
            properties->cells[0].init(parameters->n_dim, 
                                      parameters->n_periodic,
                                      properties->bonds.n_bonds,
                                      parameters->r_cutoff + 
                                      properties->bonds.length_max +
                                      parameters->skin,
                                      properties->bonds.s_bond,
                                      properties->unit_cell.a_perp);
            
            /* Initialize neighbor lists. */
            update_neighbor_lists_sphero_cells_mp(
                                      parameters->n_dim,
                                      parameters->n_periodic,
                                      properties->unit_cell.h, 
                                      parameters->skin,
                                      parameters->r_cutoff,
                                      properties->bonds.n_bonds, 
                                      properties->bonds.r_bond,
                                      properties->bonds.s_bond,
                                      properties->bonds.u_bond,
                                      properties->bonds.length,
                                      &(properties->cells[0]), 
                                      properties->neighbors.neighbs,
                                      parameters->nl_twoway_flag);
        }
        else {
            fprintf(stderr, "Invalid nl_update_flag set. Please set to 0 or 1.");
        }
    } 

    /* Initialize control structure. */
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

    return;
}
