/* Allocate memory for and initialize variables in potential energy evaluation structure
   for molecular simulation program.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)

   Output: memory for arrays in the potential energy evaluation structure is allocated,
   and a number of variables in this structure are initialized. */

#include "bob.h"

void init_potential_crosslink_sphero_bd_mp(system_parameters *parameters,
                                           system_properties *properties,
                                           system_potential *potential)
{
    int n_dim, n_periodic, neighb_switch, n_bonds, n_comp;

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    n_periodic = parameters->n_periodic;
    neighb_switch = parameters->neighb_switch;
    n_bonds = properties->bonds.n_bonds;



    /* Set number of distinct contributions to potential energy expression. */

    // Potential for interacting spherocylinders
    n_comp = 1;

    // Potential for crosslinks if crosslink_flag is enabled
    if (parameters->crosslink_flag)
        n_comp++;

    // Enable wall potential if walls exist (i.e. at least one non-periodic/free dimension)
    if ( (parameters->n_periodic != parameters->n_dim) ||
            (parameters->n_walls != 0) )
        n_comp++;

    potential->n_comp = n_comp;

    /* Allocate memory for arrays in potential energy evaluation structure. */
    potential->pot_func = (double (**) (system_parameters *,
                                        system_properties *,
                                        double**,
                                        double**,
                                        double**,
                                        int*)) allocate_1d_array(n_comp, sizeof(void *));
    potential->f_comp = (double ***) allocate_3d_array(n_comp, n_bonds, n_dim, sizeof(double));
    potential->u_comp = (double *) allocate_1d_array(n_comp, sizeof(double));
    potential->virial_comp = (double ***) allocate_3d_array(n_comp, n_dim, n_dim, sizeof(double));
    potential->t_comp = (double ***) allocate_3d_array(n_comp, n_bonds, 3, sizeof(double));
    potential->calc_matrix = (int*) allocate_1d_array(n_bonds, sizeof(int));

    int i_comp = 0;
    /* Set up array of pointers to potential energy/force evaluation functions. */

    // rod-rod interaction potential
    if (neighb_switch == 0) {
        i_comp++;
        fprintf(stderr, "only neighbor lists implemented. set neighb_switch to 1\n");
        exit(1);
    } else if (neighb_switch == 1) {
        potential->pot_func[i_comp] = brownian_sphero_neighbor_lists_mp;
        fprintf(stdout, "Potential: rod-rod interactions (neighbor lists)\n");
        i_comp++;
    } else {
        fprintf(stderr, "only all pairs neighbor lists implemented\n");
        exit(1);
    }

    // Crosslink potential
    if (parameters->crosslink_flag) {
        potential->pot_func[i_comp] = crosslink_interaction_bd_mp;
        fprintf(stdout, "Potential: crosslinks interactions\n");
        i_comp++;
    }

    // rod-tip-wall potential
    if (n_periodic!=n_dim || parameters->n_walls != 0){
        switch(parameters->wall_potential_flag  ) {
            case 0: break;
            case 1:
                    if (parameters->boundary_type == 1)
                        potential->pot_func[i_comp] = wca_cubic_wall_sphero_potential_bd;
                    else
                        potential->pot_func[i_comp] = wca_sphero_wall_potential_bd;
                    fprintf(stdout, "Potential: rod-tip-hard wall\n");
                    i_comp++;
                    break;
            case 2: potential->pot_func[i_comp] = soft_sphero_wall_potential_bd;
                    fprintf(stdout, "Potential: rod-tip-soft wall\n");
                    i_comp++;
                    break;
            case 3: potential->pot_func[i_comp] = linear_sphero_wall_potential_bd;
                    fprintf(stdout, "Potential: rod-tip-linear wall\n");
                    i_comp++;
                    break;
            case 4: potential->pot_func[i_comp] = quadratic_sphero_wall_potential_bd;
                    fprintf(stdout, "Potential: rod-tip-quadratic wall\n");
                    i_comp++;
                    break;
            case 5: potential->pot_func[i_comp] = NM_sphero_wall_potential_bd;
                    fprintf(stdout, "Potential: rod-tip-NM wall\n");
                    i_comp++;
                    break;
        }
    }


    return;

}
