/* Allocate memory for and initialize variables in potential energy evaluation structure
   for molecular simulation program.
 
   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)
 
   Output: memory for arrays in the potential energy evaluation structure is allocated, 
   and a number of variables in this structure are initialized. */

#include "bob.h"

void init_potential_spb_dynamics(system_parameters *parameters,
                                 system_properties *properties,
                                 system_potential *potential) {
    int n_dim, neighb_switch, n_bonds, n_comp, n_sites;

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    n_bonds = properties->bonds.n_bonds;
    n_sites = properties->sites.n_sites;
    neighb_switch = parameters->neighb_switch;

    /* Set number of distinct contributions to potential energy expression. */
    n_comp = 0;
    /* Add in the number of potentials dependent on flags */
    if(parameters->pair_interaction_flag == 1) {
        /* rod-wall and a brownian_sphero */
        n_comp++;
        /* and then crosslinks */
        if (properties->crosslinks.attachment_model_ == 1 ||
            properties->crosslinks.attachment_model_ == 2)
            n_comp++;
    }

    
    /* spbs have either 1 or 2 potential functions */
    if (!parameters->rigid_tether_flag) {
        if (parameters->use_spb_flag == 1) {
            n_comp += 2;
        }
        else {
            n_comp++;
        }
    }

    if (parameters->n_spheres > 0)
        n_comp++;

    if (parameters->wall_potential_flag != 0)
        n_comp++;
    potential->n_comp = n_comp;

    /* Allocate memory for arrays in potential energy evaluation structure. */
    potential->pot_func = (double (**) (system_parameters *,
                                        system_properties *,
                                        double**,
                                        double**,
                                        double**,
                                        int*)) allocate_1d_array(n_comp, sizeof(void *));
    /* FIXME */
    potential->f_comp = (double ***) allocate_3d_array(n_comp, n_sites-n_bonds, n_dim, sizeof(double));
    potential->u_comp = (double *) allocate_1d_array(n_comp, sizeof(double));
    potential->virial_comp = (double ***) allocate_3d_array(n_comp, n_dim, n_dim, sizeof(double));
    potential->t_comp = (double ***) allocate_3d_array(n_comp, n_bonds, n_dim, sizeof(double));
    potential->calc_matrix = (int*) allocate_1d_array(n_bonds, sizeof(int));

    int i_comp = 0;
    /* Set up array of pointers to potential energy/force evaluation functions. */

    /* Spherocylinders and Crosslink interactions */
    if(parameters->pair_interaction_flag == 1) {
        /* brownian_sphero interaction (one of two) */
        /* crosslinks force neighborlists and crosslink_interaction */
        if (properties->crosslinks.attachment_model_ == 1 ||
            properties->crosslinks.attachment_model_ == 2) {
            potential->pot_func[i_comp] = brownian_sphero_neighbor_lists;
            i_comp++;
            potential->pot_func[i_comp] = crosslink_interaction_bd;
            i_comp++;
            fprintf(stdout, "Potential: rod-rod-neighbor\n");
            fprintf(stdout, "Potential: crosslink-rod\n");
        }
        else {
            if (neighb_switch == 0) {
                potential->pot_func[i_comp] = brownian_sphero_all_pairs;
                fprintf(stdout, "Potential: rod-rod-allpairs\n");
            }
            else if (neighb_switch == 1) {
                potential->pot_func[i_comp] = brownian_sphero_neighbor_lists;
                fprintf(stdout, "Potential: rod-rod-neighbor\n");
            }
            else {
                fprintf(stderr, "neighb_switch option must be 0 (all pairs search) or 1 (neighbor list)\n");
            }
            i_comp++;
        }
    }

    if (!parameters->rigid_tether_flag) {
        if (parameters->use_spb_flag == 1) {
            potential->pot_func[i_comp] = anchor_potential_bd;
            i_comp++;
            potential->pot_func[i_comp] = anchor_wca_potential_bd;
            i_comp++;
            fprintf(stdout, "Potential: anchor-rod-base\n");
            fprintf(stdout, "Potential: anchor-anchor\n");
        }
        else {
            potential->pot_func[i_comp] = wall_attach_potential_bd;
            i_comp++;
            fprintf(stdout, "Potential: wall-rod-base\n");
        }
    }

    /* KC - Wall potential */
    if (parameters->n_spheres > 0) {
        potential->pot_func[i_comp] = sphere_wall_potential_bd;
        i_comp++;
    }

    switch(parameters->wall_potential_flag) {
        case 0: break;
        case 1: potential->pot_func[i_comp] = wca_sphero_wall_potential_bd;
                fprintf(stdout, "Potential: rod-tip-hard wall\n");
                i_comp++;
                break;
        case 2: potential->pot_func[i_comp] = soft_sphero_wall_potential_bd;
                fprintf(stdout, "Potential: rod-tip-soft wall\n");
                i_comp++;
                break;
        case 3: potential->pot_func[i_comp] = linear_sphero_wall_potential_bd;
                fprintf(stdout, "Potential: rod-tip-linear wall\n");
                i_comp++;
                break;
        case 4: potential->pot_func[i_comp] = quadratic_sphero_wall_potential_bd;
                fprintf(stdout, "Potential: rod-tip-quadratic wall\n");
                i_comp++;
                break;
        case 5: potential->pot_func[i_comp] = NM_sphero_wall_potential_bd;
                fprintf(stdout, "Potential: rod-tip-NM wall\n");
                i_comp++;
                break;
    }
    return;
}
