/* Allocate memory for and initialize variables in potential energy evaluation structure
   for molecular simulation program.
 
   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)
 
   Output: memory for arrays in the potential energy evaluation structure is allocated, 
   and a number of variables in this structure are initialized. */

#include "bob.h"

#include <iostream>

void init_potential_spindle_bd_mp(system_parameters *parameters,
                                  system_properties *properties,
                                  system_potential *potential) {
    int n_dim, neighb_switch, n_bonds, n_comp, n_sites;

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    n_bonds = properties->bonds.n_bonds;
    n_sites = properties->sites.n_sites;
    neighb_switch = parameters->neighb_switch;

    /* Set number of distinct contributions to potential energy expression. */
    n_comp = 0;
    /* Add in the number of potentials dependent on flags */
    if(parameters->pair_interaction_flag == 1) {
        /* rod-wall and a brownian_sphero */
        n_comp++;
        /* and then crosslinks */
        if (properties->crosslinks.attachment_model_ == 1 ||
            properties->crosslinks.attachment_model_ == 2)
            n_comp++;
    }

    
    /* spbs have either 1 or 2 potential functions */
    if (!parameters->rigid_tether_flag || parameters->n_anchors != 0) {
        if (parameters->use_spb_flag == 1) {
            n_comp += 2;

            // Well, maybe 1 more if there is the angular spring
            if (parameters->spb_angular_spring_flag != 0) {
                n_comp++;
            }
        }
        else {
            n_comp++;
        }
    }

    if (parameters->n_spheres > 0)
        n_comp++;

    if (parameters->wall_potential_flag != 0)
        n_comp++;

    // Check the number of kinetochore potentials
    if (properties->chromosomes.nkcs_ > 0) {
        // Chromatin spring complex
        n_comp++;
        // KC wall potential
        n_comp++;
        if (properties->chromosomes.use_soft_chromatin_potential_ == 1) {
            // MT repulsion
            n_comp++;
        }
        // KC-MT spring attachment
        n_comp++;
        if (properties->chromosomes.use_kc_potential_ == 1) {
            // KC-MT tip steric repulsion
            n_comp++;
        }
    }
    potential->n_comp = n_comp;

    /* Allocate memory for arrays in potential energy evaluation structure. */
    potential->pot_func = (double (**) (system_parameters *,
                                        system_properties *,
                                        double**,
                                        double**,
                                        double**,
                                        int*)) allocate_1d_array(n_comp, sizeof(void *));
    /* FIXME */
    potential->f_comp = (double ***) allocate_3d_array(n_comp, n_sites-n_bonds, n_dim, sizeof(double));
    potential->u_comp = (double *) allocate_1d_array(n_comp, sizeof(double));
    potential->virial_comp = (double ***) allocate_3d_array(n_comp, n_dim, n_dim, sizeof(double));
    potential->t_comp = (double ***) allocate_3d_array(n_comp, n_bonds, n_dim, sizeof(double));
    potential->calc_matrix = (int*) allocate_1d_array(n_bonds, sizeof(int));

    int i_comp = 0;
    /* Set up array of pointers to potential energy/force evaluation functions. */

    /* Spherocylinders and Crosslink interactions */
    if(parameters->pair_interaction_flag == 1) {
        /* brownian_sphero interaction (one of two) */
        /* crosslinks force neighborlists and crosslink_interaction */
        if (properties->crosslinks.attachment_model_ == 1 ||
            properties->crosslinks.attachment_model_ == 2) {
            potential->pot_func[i_comp] = brownian_sphero_neighbor_lists_mp;
            std::cout << "Potential: rod-rod-neighbor " << i_comp << std::endl;
            i_comp++;
            potential->pot_func[i_comp] = crosslink_interaction_bd_mp;
            std::cout << "Potential: crosslink-rod MP " << i_comp << std::endl;
            i_comp++;
        }
        else {
            if (neighb_switch == 0) {
                potential->pot_func[i_comp] = brownian_sphero_all_pairs;
                fprintf(stdout, "Potential: rod-rod-allpairs\n");
                fprintf(stderr, "NOT SUPPORTED!\n");
            }
            else if (neighb_switch == 1) {
                potential->pot_func[i_comp] = brownian_sphero_neighbor_lists_mp;
                fprintf(stdout, "Potential: rod-rod-neighbor MP\n");
            }
            else {
                fprintf(stderr, "neighb_switch option must be 0 (all pairs search) or 1 (neighbor list)\n");
            }
            i_comp++;
        }
    }

    if (!parameters->rigid_tether_flag || parameters->n_anchors != 0 ) {
        if (parameters->use_spb_flag == 1) {
            potential->pot_func[i_comp] = anchor_potential_bd;
            std::cout << "Potential: anchor-rod-base " << i_comp << std::endl;
            i_comp++;
            // Check for the specifics of the anchor potential we are using
            if (properties->anchors.interaction_type_int == 0) {
                potential->pot_func[i_comp] = anchor_wca_potential_bd;
                std::cout << "Potential: anchor-anchor WCA " << i_comp << std::endl;
                i_comp++;
            } else if (properties->anchors.interaction_type_int == 1) {
                potential->pot_func[i_comp] = anchor_lj_potential_bd;
                std::cout << "Potential: anchor-anchor LJ12-6 "  << i_comp << std::endl;
                i_comp++;
            } else {
                std::cerr << "ERROR, somehow using an incorrect SPB interaction potential in init_potential_spindle_bd_mp\n";
                exit(1);
            }

            if (parameters->spb_angular_spring_flag != 0) {
                potential->pot_func[i_comp] = spb_spring_potential_angular;
                std::cout << "Potential: anchor-rod angular spring " << i_comp << std::endl;
                i_comp++;
            }
        }
        else {
            potential->pot_func[i_comp] = wall_attach_potential_bd;
            i_comp++;
            fprintf(stdout, "Potential: wall-rod-base\n");
        }
    }

    /* KC - Wall potential */
    // Deprecated, use chromosomes
    if (parameters->n_spheres > 0) {
        std::cout << "Deprecated, please switch to using real chromosomes\n";
        exit(1);
        potential->pot_func[i_comp] = sphere_wall_potential_bd;
        i_comp++;
    }

    // Chromosomes
    if (properties->chromosomes.nkcs_ > 0) {
        potential->pot_func[i_comp] = chromosome_chromatin_potential;
        std::cout << "Potential: chromatin spring complex " << i_comp << std::endl;
        i_comp++;
        potential->pot_func[i_comp] = kc_wall_wca_potential;
        std::cout << "Potential: kinetochore-wall WCA " << i_comp << std::endl;
        i_comp++;
        if (properties->chromosomes.use_soft_chromatin_potential_ == 1) {
            potential->pot_func[i_comp] = chromosome_mt_soft_gaussian_potential_allpairs;
            std::cout << "Potential: chromosome-mt soft gaussian (all pairs) " << i_comp << std::endl;
            i_comp++;
        }
        // Choose which of the attachment factor forces to use!
        if (properties->chromosomes.af_potential_type_ == "harmonic") {
            potential->pot_func[i_comp] = af_mt_harmonic_potential;
            std::cout << "Potential: Attachment factor-microtubule harmonic " << i_comp << std::endl;
            i_comp++;
        } else if (properties->chromosomes.af_potential_type_ == "quartic") {
            potential->pot_func[i_comp] = af_mt_quartic_potential;
            std::cout << "Potential: Attachment factor-microtubule quartic " << i_comp << std::endl;
            i_comp++;
        } else {
            std::cerr << "Using an improper kind of AF potential: " << properties->chromosomes.af_potential_type_ << std::endl;
            exit(1);
        }
        if (properties->chromosomes.use_kc_potential_ == 1) {
            // IF we are using the kinetochore mesh, use that potential, otherwise, use the point-like potential as it
            // is much simpler for budding yeasts
            if (properties->chromosomes.kc_triangulation_type_ == "rectangle") {
                potential->pot_func[i_comp] = kinetochoremesh_mt_wca_potential_allpairs;
                std::cout << "Potential: Kinetochore mesh-MT WCA (allpairs)" << i_comp << std::endl;
                i_comp++;
                //potential->pot_func[i_comp] = kinetochoremesh_mt_wca_potential_neighbors;
                //std::cout << "Potential: Kinetochore mesh-MT WCA (neighbors) " << i_comp << std::endl;
                //i_comp++;
            } else if (properties->chromosomes.kc_triangulation_type_ == "point") {
                potential->pot_func[i_comp] = kinetochorepoint_mt_wca_potential_allpairs;
                std::cout << "Potential: Kinetochore point-MT WCA (allpairs) " << i_comp << std::endl;
                i_comp++;
            } else {
                std::cerr << "ERROR: Please use a correct kinetochore meshing function: " << properties->chromosomes.kc_triangulation_type_ << std::endl;
                exit(1);
            }
        }
    }

    switch(parameters->wall_potential_flag) {
        case 0: break;
        case 1: potential->pot_func[i_comp] = wca_sphero_wall_potential_bd;
                std::cout << "Potential: rod-tip-hard wall " << i_comp << std::endl;
                i_comp++;
                break;
        case 2: potential->pot_func[i_comp] = soft_sphero_wall_potential_bd;
                std::cout << "Potential: rod-tip soft wall " << i_comp << std::endl;
                i_comp++;
                break;
        case 3: potential->pot_func[i_comp] = linear_sphero_wall_potential_bd;
                std::cout << "Potential: rod-tip linear wall " << i_comp << std::endl;
                i_comp++;
                break;
        case 4: potential->pot_func[i_comp] = quadratic_sphero_wall_potential_bd;
                std::cout << "Potential: rod-tip quadratic wall " << i_comp << std::endl;
                i_comp++;
                break;
        case 5: potential->pot_func[i_comp] = NM_sphero_wall_potential_bd;
                std::cout << "Potential: rod-tip NM wall " << i_comp << std::endl;
                i_comp++;
                break;
    }
    return;
}
