/* Initialize variables and allocate memory for kinetochore capture brownian
   dynamics simulation. 

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)

   Output: memory for a variety of arrays is allocated,
   and a number of variables are initialized */
#include "bob.h"

void init_spb_dynamics(system_parameters *parameters,
                       system_properties *properties,
                       system_potential *potential,
                       char *config_file) {
    /* Set up pointers to header read and write functions. */
    properties->read_header_func = read_header_spb_dynamics;
    properties->write_header_func = write_header_spb_dynamics;

    /* Set up the gsl random generator, as it will be used for
       random assignment*/
    /* Initialize GSL random number generator state */
    gsl_rng_env_setup();
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    configurator_spb_dynamics(parameters, properties, parameters->config_file);

    parameters->r_cutoff = parameters->r_cutoff_lj = pow(2, 1.0/6.0);

    /* No periodic boundary conditions. */
    parameters->n_periodic = 0;

    /* Initialize unit cell structure. */
    init_unit_cell_structure(parameters, properties);

    /* Initialize diffusion information */
    init_diffusion_sphero(parameters, properties);

    /* Initialize crosslink structure. */
    properties->crosslinks.Init(parameters, properties,
                                parameters->crosslink_file,
                                NULL);


    if (parameters->dynamic_instability_flag) {
        fprintf(stdout, "Dynamic instability is enabled\n");
        init_dynamic_instability(parameters, properties);
    }

    properties->bonds.length_max = 0.0;
    /* Find maximum length of bond in system */
    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        properties->bonds.length_max = MAX(properties->bonds.length_max,
                                           properties->bonds.length[i_bond]);
    }

    parameters->neighb_switch = 1;

    
    if (properties->crosslinks.attachment_model_) {

        fprintf(stdout, "Crosslinks are enabled\n");

        // parameters->r_cutoff = MAX(sqrt(properties->crosslinks.r_cutoff2_),
        //                            parameters->r_cutoff);
    }
    

    if (parameters->neighb_switch == 1) {
        /* Initialize neighbor lists. */
        properties->neighbors.neighbs = new nl_list[properties->bonds.n_bonds];

        update_neighbor_lists_sphero_all_pairs(parameters->n_dim, parameters->n_periodic,
                                               properties->unit_cell.h,
                                               parameters->skin, 
                                               parameters->r_cutoff,
                                               properties->bonds.n_bonds, 
                                               properties->bonds.r_bond, 
                                               properties->bonds.s_bond, 
                                               properties->bonds.u_bond, 
                                               properties->bonds.length,
                                               properties->neighbors.neighbs, 0);
    }

    update_bond_site_positions(parameters->n_dim, parameters->n_periodic,
                               properties->bonds.n_bonds, properties->sites.n_sites,
                               properties->unit_cell.h, properties->unit_cell.h_inv,
                               properties->bonds.bond_site_1, properties->bonds.bond_site_2,
                               properties->bonds.r_bond, properties->bonds.u_bond,
                               properties->bonds.length, 
                               properties->sites.r, properties->sites.s);

    /* Allocate memory for arrays in thermodynamics structure. */
    properties->thermo.virial = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.stress = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.press_tensor = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));

    /* Init potential structure */
    init_potential_spb_dynamics(parameters, properties, potential);

    /* Initialize control structure. */
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

    /* Set time to 0 */
    properties->time = 0.0;

    return;
}
