// Initialize multithreaded spindle brownian dynamics simulation

#include "bob.h"
#include "xlink_management.h"

#include <iostream>

void init_spindle_bd_mp(system_parameters *parameters,
                        system_properties *properties,
                        system_potential *potential) {
    // Set up pointers to the header read and write functions
    properties->read_header_func  = read_header_spb_dynamics;
    properties->write_header_func = write_header_spb_dynamics;

    // Set up the master GSL random number generator
    gsl_rng_env_setup();
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    // Run the configurator
    configurator_spb_dynamics(parameters, properties, parameters->config_file);

    // Set up the hard cutoff
    // XXX FIXME change up how we set up potentials
    parameters->r_cutoff = parameters->r_cutoff_lj = pow(2, 1.0/6.0);

    // We cannot have periodic BCs
    parameters->n_periodic = 0;

    // Initialize unit cell structure
    init_unit_cell_structure(parameters, properties);

    // Initialize diffusion information
    init_diffusion_sphero(parameters, properties);

    // Initialize crosslink structure
    properties->crosslinks.Init(parameters,
                                properties,
                                parameters->crosslink_file,
                                NULL);

    // Initialize chromosome structure
    properties->chromosomes.Init(parameters,
                                 properties,
                                 parameters->chromosome_config,
                                 parameters->chromosome_output);

    // Initialize dynamic instability
    if (parameters->dynamic_instability_flag == 1) {
        fprintf(stdout, "Dynamic instability is enabled\n");
        init_dynamic_instability(parameters, properties);
    }

    // Set up max bond length
    properties->bonds.length_max = 0.0;
    for (int ibond = 0; ibond < properties->bonds.n_bonds; ++ibond) {
        properties->bonds.length_max = MAX(properties->bonds.length_max,
                                           properties->bonds.length[ibond]);
    }

    // Load OMP settings
    set_omp_settings(parameters);
    print_omp_settings();

    #ifdef ENABLE_OPENMP
    #pragma omp parallel 
    #pragma omp master
    #endif
    {
        int n_threads = properties->mp_local.n_threads = properties->rng_mt.n_threads = 1;
            
        #ifdef ENABLE_OPENMP
        n_threads = properties->mp_local.n_threads = properties->rng_mt.n_threads = omp_get_num_threads();
        #endif

        properties->rng_mt.rng = (rng_properties*) 
            allocate_1d_array(properties->rng_mt.n_threads, sizeof(rng_properties));

        gsl_rng_env_setup();
        for (int i_thr = 0; i_thr < properties->rng_mt.n_threads; ++i_thr) {
            properties->rng_mt.rng[i_thr].T = gsl_rng_default;
            properties->rng_mt.rng[i_thr].r = gsl_rng_alloc(properties->rng_mt.rng[i_thr].T);
            gsl_rng_set(properties->rng_mt.rng[i_thr].r, (parameters->seed+1) * i_thr);
        }

        properties->mp_local.f_local = (double***)
            allocate_3d_array(n_threads, properties->bonds.n_bonds, parameters->n_dim, sizeof(double));

        properties->mp_local.t_local = (double***)
            allocate_3d_array(n_threads, properties->bonds.n_bonds, 3, sizeof(double));

        properties->mp_local.virial_local = (double***)
            allocate_3d_array(n_threads, parameters->n_dim, parameters->n_dim, sizeof(double));

        // Specialized xlink virial broken up by species
        properties->mp_local.virial_local_xlink = (double****)
            allocate_4d_array(n_threads, properties->crosslinks.n_types_, parameters->n_dim, parameters->n_dim, sizeof(double));
        
        properties->mp_local.calc_local = (int**)
            allocate_2d_array(n_threads, properties->bonds.n_bonds, sizeof(int));

        //// Create the local copy of all of the rng generator states for bonds
        //properties->mp_local.rng_local_bonds = (rng_properties*)
        //    allocate_1d_array(properties->bonds.n_bonds, sizeof(rng_properties));

        //for (int ibond = 0; ibond < properties->bonds.n_bonds; ++ibond) {
        //    //auto mseed = gsl_rng_get(properties->rng.r);
        //    auto mseed = (parameters->seed+1+n_threads+ibond);
        //    properties->mp_local.rng_local_bonds[ibond].T = gsl_rng_default;
        //    properties->mp_local.rng_local_bonds[ibond].r = gsl_rng_alloc(properties->mp_local.rng_local_bonds[ibond].T);
        //    gsl_rng_set(properties->mp_local.rng_local_bonds[ibond].r, mseed);
        //}
    }

    // Create the bonds random number generators
    for (int ibond = 0; ibond < properties->bonds.n_bonds; ++ibond) {
        auto mseed = gsl_rng_get(properties->rng.r); 
        properties->bonds.rng_local[ibond].T = gsl_rng_default;
        properties->bonds.rng_local[ibond].r = gsl_rng_alloc(properties->bonds.rng_local[ibond].T);
        gsl_rng_set(properties->bonds.rng_local[ibond].r, mseed+ibond);
    }

    //std::cout << "Size check\n";
    //std::cout << "   sizeof(int): " << sizeof(int) << std::endl;
    //std::cout << "   sizeof(rng_properties): " << sizeof(rng_properties) << std::endl;
    //std::cout << "   sizeof(properties->mp_local.rng_local_bonds): " << sizeof(properties->mp_local.rng_local_bonds) << std::endl;
    //int base_size = 0;
    //for (int ibond = 0; ibond < properties->bonds.n_bonds; ++ibond) {
    //    base_size += sizeof(properties->mp_local.rng_local_bonds[ibond]);
    //    std::cout << "      [" << ibond << "] size: " << sizeof(properties->mp_local.rng_local_bonds[ibond]) << std::endl;
    //}
    //std::cout << "   sizeof(rngs): " << base_size << std::endl;

    // Calculate the maximum rcutoff in the system from the lj interaction vs the xlinks
    for (int itype = 0; itype < properties->crosslinks.n_types_; ++itype) {
        parameters->r_cutoff = MAX(parameters->r_cutoff,
                                   properties->crosslinks.r_cutoff_1_2_[itype]);
    }
    fprintf(stdout, "   r_cutoff = %g\n", parameters->r_cutoff);

    if (parameters->r_cutoff != parameters->r_cutoff) {
        fprintf(stderr, "\nUnable to calculate r_cutoff. Consider lowering the spring constant of the crosslinks"
                " or increasing their concentration\n");
        exit(1);
    }

    /* Allocate memory for arrays in thermodynamics structure. */
    properties->thermo.virial = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.stress = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.press_tensor = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.virial_xlink = (double ***)
        allocate_3d_array(properties->crosslinks.n_types_, parameters->n_dim, parameters->n_dim,
                sizeof(double));

    // Set up the type of neighbor structure we're using
    // Initialize them too
    properties->n_cell_lists = 0;
    parameters->nl_cell_list = 0;
    if (parameters->neighb_switch == 1 && parameters->nl_update_flag == 1) {
        ++properties->n_cell_lists;
    } else if (parameters->neighb_switch == 2) {
        fprintf(stderr, "Cell list only search not supported. Set neighb_switch option to 1\n");
        exit(1);
    }

    if (properties->n_cell_lists > 0)
        properties->cells = new cell_list[properties->n_cell_lists];
    else
        properties->cells = NULL;

    if (parameters->neighb_switch == 1) {
        // Initialize the neighbor list
        properties->neighbors.neighbs = new nl_list[properties->bonds.n_bonds];

        if (parameters->nl_update_flag == 0) {
            // All pairs search
            fprintf(stdout, "All Pairs MP neighbor list\n");
            update_neighbor_lists_sphero_all_pairs_mp(parameters->n_dim, parameters->n_periodic,
                                                      properties->unit_cell.h,
                                                      parameters->skin, parameters->r_cutoff,
                                                      properties->bonds.n_bonds,
                                                      properties->bonds.r_bond,
                                                      properties->bonds.s_bond,
                                                      properties->bonds.u_bond,
                                                      properties->bonds.length,
                                                      properties->neighbors.neighbs,
                                                      parameters->nl_twoway_flag);
        } else if (parameters->nl_update_flag == 1) {
            // Cell search (if it works correctly)
            fprintf(stdout, "Cell MP neighbor list (might not work correctly)\n");
            properties->cells[0].init(parameters->n_dim, parameters->n_periodic,
                                      properties->bonds.n_bonds,
                                      parameters->r_cutoff + properties->bonds.length_max + parameters->skin,
                                      properties->bonds.s_bond,
                                      properties->unit_cell.a_perp);
            
            /* Initialize neighbor lists. */
            update_neighbor_lists_sphero_cells_mp(parameters->n_dim, parameters->n_periodic,
                                                  properties->unit_cell.h, 
                                                  parameters->skin, parameters->r_cutoff,
                                                  properties->bonds.n_bonds, properties->bonds.r_bond,
                                                  properties->bonds.s_bond, properties->bonds.u_bond,
                                                  properties->bonds.length,
                                                  &(properties->cells[0]), 
                                                  properties->neighbors.neighbs,
                                                  parameters->nl_twoway_flag);
        } else {
            fprintf(stderr, "Invalid nl_update_flag set.  Please set to 0 or 1.");
        }
    }

    // Update the trajectory information
    update_bond_site_positions_mp(parameters->n_dim, parameters->n_periodic,
                                  properties->bonds.n_bonds, properties->sites.n_sites,
                                  properties->unit_cell.h, properties->unit_cell.h_inv,
                                  properties->bonds.bond_site_1, properties->bonds.bond_site_2,
                                  properties->bonds.r_bond, properties->bonds.u_bond,
                                  properties->bonds.length, 
                                  properties->sites.r, properties->sites.s);


    // Create the potentials
    init_potential_spindle_bd_mp(parameters, properties, potential);

    // Initialize control structure
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

    // Set the time to zero
    properties->time = 0.0;

    // For runtime, set the gsl error handler to 0, so that we can use our own exception handling.
    gsl_set_error_handler_off();

    return;
}
