// Initialize the free diffusion similar to the way that spindle_bd_mp does it

#include "bob.h"

#include <iostream>

void init_xlink_analysis(system_parameters *parameters,
                         system_properties *properties,
                         system_potential *potential) {
    // Set up pointers to the header read and write functions
    properties->read_header_func = read_header_spb_dynamics;
    properties->write_header_func = write_header_spb_dynamics;

    // Set up master RNG
    gsl_rng_env_setup();
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    // Run the configurator
    configurator_generic(parameters, properties, parameters->config_file);

    // Initialize unit cell structure
    init_unit_cell_structure(parameters, properties);

    // Initialize diffusion information
    init_diffusion_sphero(parameters, properties);

    // Initialize crosslink structure
    properties->crosslinks.Init(parameters,
                                properties,
                                parameters->crosslink_file,
                                NULL);

    // Set up max bond length
    properties->bonds.length_max = 0.0;
    for (int ibond = 0; ibond < properties->bonds.n_bonds; ++ibond) {
        properties->bonds.length_max = MAX(properties->bonds.length_max,
                                           properties->bonds.length[ibond]);
    }

    // Calculate the maximum rcutoff in the system from the lj interaction vs the xlinks
    for (int itype = 0; itype < properties->crosslinks.n_types_; ++itype) {
        parameters->r_cutoff = MAX(parameters->r_cutoff,
                                   properties->crosslinks.r_cutoff_1_2_[itype]);
    }
    fprintf(stdout, "   r_cutoff = %g\n", parameters->r_cutoff);

    // Set up the all pairs neighbor list
    properties->neighbors.neighbs = new nl_list[properties->bonds.n_bonds];
    update_neighbor_lists_sphero_all_pairs_mp(parameters->n_dim, parameters->n_periodic,
                                              properties->unit_cell.h,
                                              parameters->skin, parameters->r_cutoff,
                                              properties->bonds.n_bonds,
                                              properties->bonds.r_bond,
                                              properties->bonds.s_bond,
                                              properties->bonds.u_bond,
                                              properties->bonds.length,
                                              properties->neighbors.neighbs,
                                              parameters->nl_twoway_flag);

    // Allocate memory for arrays in thermodynamics structure.
    properties->thermo.virial = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.stress = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));
    properties->thermo.press_tensor = (double **)
        allocate_2d_array(parameters->n_dim, parameters->n_dim, sizeof(double));

    // Update the trajectory information
    update_bond_site_positions_mp(parameters->n_dim, parameters->n_periodic,
                                  properties->bonds.n_bonds, properties->sites.n_sites,
                                  properties->unit_cell.h, properties->unit_cell.h_inv,
                                  properties->bonds.bond_site_1, properties->bonds.bond_site_2,
                                  properties->bonds.r_bond, properties->bonds.u_bond,
                                  properties->bonds.length, 
                                  properties->sites.r, properties->sites.s);

    // Set the multithreaded random number generators
    for (int ibond = 0; ibond < properties->bonds.n_bonds; ++ibond) {
        auto mseed = gsl_rng_get(properties->rng.r);
        properties->bonds.rng_local[ibond].T = gsl_rng_default;
        properties->bonds.rng_local[ibond].r = gsl_rng_alloc(properties->bonds.rng_local[ibond].T);
        gsl_rng_set(properties->bonds.rng_local[ibond].r, mseed);
    }

    // NO POTENTIALS
    potential->n_comp = 0;
    potential->calc_matrix = (int*) allocate_1d_array(properties->bonds.n_bonds, sizeof(int));

    properties->time = 0.0;

    /* Initialize control structure. */
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

}
