// Implementaiton of unit test of lookup tables in multiple ways

#include "bob.h"
#include "correlation_data.h"
#include "kinetochore.h"
#include "integration_test_chromosomes.h"

#include <fstream>
#include <iostream>

// Run the tests and all their variants
void IntegrationTestChromosomes::RunTests() {
    std::cout << "****************\n";
    std::cout << "Integraiton Test Chromosomes run tests\n";
    std::cout << "****************\n";

    for (auto &kv : tests_) {
        std::cout << "----------------\n";
        std::cout << "Test : " << kv.first << std::endl;
        std::cout << "----------------\n";

        for (int iv = 0; iv < node_["Tests"][kv.first].size(); ++iv) {
            std::cout << "  Variant : " << iv << std::endl;
            var_subnode_ = node_["Tests"][kv.first][iv];
            iv_ = iv;
            auto result = kv.second();

            if (!result) {
                std::cout << "Test : " << kv.first << " failed, check output!\n";
                exit(1);
            }
        }
    }
}

// Set the possible tests to run
void IntegrationTestChromosomes::SetTests() {
    node_ = YAML::LoadFile(filename_);

    // Check against names to bind method calls
    if (node_["Tests"]["GenerateInvertDistribution"]) {
        tests_["GenerateInvertDistribution"] = std::bind(&IntegrationTestChromosomes::GenerateInvertDistribution, this);
    }
    if (node_["Tests"]["Stage12Probability"]) {
        tests_["Stage12Probability"] = std::bind(&IntegrationTestChromosomes::Stage12Probability, this);
    }
}

// Initialize the system of chromosomes (with other stuff toooo)
void IntegrationTestChromosomes::InitSystem(system_parameters *parameters,
                                            system_properties *properties) {

    // Generic parameter initialization
    parse_parameters_node(var_subnode_["params"], parameters);
    gsl_rng_env_setup();
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    // Generate the unit cell
    int ndim = parameters->n_dim;
    properties->unit_cell.h = (double **) allocate_2d_array(ndim, ndim, sizeof(double));
    for (int i = 0; i < ndim; ++i) {
        properties->unit_cell.h[i][i] = 100;
    }

    // Get the microtubule information
    {
        int nmts = (int)var_subnode_["microtubules"].size();
        init_unit_cell_structure(parameters, properties);
        int nsites = 2*nmts;
        parameters->n_spheros = properties->bonds.n_bonds = nmts;
        properties->sites.n_sites = nsites;
        properties->sites.v = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
        properties->sites.r = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
        init_site_structure_sphero(parameters, properties);
        init_bond_structure_sphero(parameters, properties);
        for (int imt = 0; imt < nmts; ++imt) {
            for (int i = 0; i < ndim; ++i) {
                properties->bonds.r_bond[imt][i] = var_subnode_["microtubules"][imt]["r"][i].as<double>();
                properties->bonds.u_bond[imt][i] = var_subnode_["microtubules"][imt]["u"][i].as<double>();
            }
            properties->bonds.length[imt] = var_subnode_["microtubules"][imt]["l"].as<double>();
        }
        init_dynamic_instability(parameters, properties);
        // Print the bond information
        for (int imt = 0; imt < nmts; ++imt) {
            std::cout << "New Microtubule:\n";
            std::cout << "  r(" << properties->bonds.r_bond[imt][0] << ", "
                                << properties->bonds.r_bond[imt][1] << ", "
                                << properties->bonds.r_bond[imt][2] << ")\n";
            std::cout << "  u(" << properties->bonds.u_bond[imt][0] << ", "
                                << properties->bonds.u_bond[imt][1] << ", "
                                << properties->bonds.u_bond[imt][2] << ")\n";
            std::cout << "  l " << properties->bonds.length[imt] << std::endl;
        }
    }
}

// Generate the invert distribution of attachments along MT
bool IntegrationTestChromosomes::GenerateInvertDistribution() {
    bool success = true;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);

    // Create the chromosomes
    // First, create a node based on the basic information
    // Then override what we want for this particular parameter set
    std::string c_file = parameters.chromosome_config;
    YAML::Node chromosome_node = YAML::LoadFile(c_file.c_str());

    // Override the values
    chromosome_node["chromosomes"]["properties"]["AF_spring"]                = var_subnode_["chromosome"]["AF_spring"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_spring_angular"]        = var_subnode_["chromosome"]["AF_spring_angular"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_number_complexes"]      = var_subnode_["chromosome"]["AF_number_complexes"].as<int>();
    chromosome_node["chromosomes"]["properties"]["AF_rest_length"]           = var_subnode_["chromosome"]["AF_rest_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_characteristic_length"] = var_subnode_["chromosome"]["AF_characteristic_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["nkmc"]                     = var_subnode_["chromosome"]["nkmc"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_tip_concentration"]     = var_subnode_["chromosome"]["AF_tip_concentration"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_tip_on_rate"]           = var_subnode_["chromosome"]["AF_tip_on_rate"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_side_concentration"]    = var_subnode_["chromosome"]["AF_side_concentration"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_side_on_rate"]          = var_subnode_["chromosome"]["AF_side_on_rate"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_tip_distance"]          = var_subnode_["chromosome"]["AF_tip_distance"].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][0]                 = var_subnode_["chromosome"]["rA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][1]                 = var_subnode_["chromosome"]["rA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][2]                 = var_subnode_["chromosome"]["rA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][0]                 = var_subnode_["chromosome"]["uA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][1]                 = var_subnode_["chromosome"]["uA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][2]                 = var_subnode_["chromosome"]["uA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][0]                 = var_subnode_["chromosome"]["uB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][1]                 = var_subnode_["chromosome"]["uB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][2]                 = var_subnode_["chromosome"]["uB"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][0]                 = var_subnode_["chromosome"]["rB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][1]                 = var_subnode_["chromosome"]["rB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][2]                 = var_subnode_["chromosome"]["rB"][2].as<double>();

    properties.chromosomes.Init(&parameters,
                                &properties,
                                &chromosome_node);

    // Set up some correlation data to track the distribution
    CorrelationData af_dist_1d;
    {
        double bin_size[] = {0.05};
        double bin_min[] = {0.0};
        double bin_max[] = {properties.bonds.length[0]};
        af_dist_1d.Init(1, bin_size, bin_min, bin_max);
    }

    int nsteps = parameters.n_steps;
    std::cout << "Running " << nsteps << " steps\n";
    for (int i = 0; i < nsteps; ++i) {
        properties.i_current_step = i;    

        // Run chromosome monte carlo cycle
        properties.chromosomes.StepKMC(&parameters, &properties);

        // Check on the kcs for binding
        for (int ikc = 0; ikc < properties.chromosomes.nkcs_; ++ikc) {
            for (int isite = 0; isite < properties.chromosomes.naf_; ++isite) {
                if (properties.chromosomes.kinetochores_[ikc].attach_[isite] != -1) {
                    // Where we attached?
                    double crosspos = properties.chromosomes.kinetochores_[ikc].cross_pos_[isite]; 
                    af_dist_1d.Bin(1, &crosspos, 1.0);
                }
            }
        }

        if (i % 100000 == 0) {
            std::cout << "Step[" << i << "] readout\n";
            for (int ikc = 0; ikc < properties.chromosomes.nkcs_; ++ikc) {
                std::cout << "   nexp[" << ikc << "] = " << properties.chromosomes.n_exp_[ikc] << std::endl;
                std::cout << "   nbound[" << ikc << "] = " << properties.chromosomes.n_bound_[ikc] << std::endl;
            }
        }
    }

    std::string fname = var_subnode_["result_file"].as<std::string>();
    //af_dist_1d.NormalizeByConstant(1.0/parameters.n_steps);
    af_dist_1d.OutputBinary(fname.c_str());

    return success;
}

bool IntegrationTestChromosomes::Stage12Probability() {
    bool success = true;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);

    // Create the chromosomes
    // First, create a node based on the basic information
    // Then override what we want for this particular parameter set
    std::string c_file = parameters.chromosome_config;

    // Initialize the chromosomes
    YAML::Node chromosome_node = YAML::LoadFile(c_file.c_str());

    // Override the values
    chromosome_node["chromosomes"]["properties"]["AF_spring"]                = var_subnode_["chromosome"]["AF_spring"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_spring_angular"]        = var_subnode_["chromosome"]["AF_spring_angular"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_number_complexes"]      = var_subnode_["chromosome"]["AF_number_complexes"].as<int>();
    chromosome_node["chromosomes"]["properties"]["AF_rest_length"]           = var_subnode_["chromosome"]["AF_rest_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_characteristic_length"] = var_subnode_["chromosome"]["AF_characteristic_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_tip_concentration"]     = var_subnode_["chromosome"]["AF_tip_concentration"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_tip_on_rate"]           = var_subnode_["chromosome"]["AF_tip_on_rate"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_tip_diffusion"]         = var_subnode_["chromosome"]["AF_tip_diffusion"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_side_concentration"]    = var_subnode_["chromosome"]["AF_side_concentration"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_side_on_rate"]          = var_subnode_["chromosome"]["AF_side_on_rate"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_side_diffusion"]        = var_subnode_["chromosome"]["AF_side_diffusion"].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][0]                 = var_subnode_["chromosome"]["rA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][1]                 = var_subnode_["chromosome"]["rA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][2]                 = var_subnode_["chromosome"]["rA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][0]                 = var_subnode_["chromosome"]["uA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][1]                 = var_subnode_["chromosome"]["uA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][2]                 = var_subnode_["chromosome"]["uA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][0]                 = var_subnode_["chromosome"]["rB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][1]                 = var_subnode_["chromosome"]["rB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][2]                 = var_subnode_["chromosome"]["rB"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][0]                 = var_subnode_["chromosome"]["uB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][1]                 = var_subnode_["chromosome"]["uB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][2]                 = var_subnode_["chromosome"]["uB"][2].as<double>();

    properties.chromosomes.Init(&parameters,
                                &properties,
                                &chromosome_node);

    std::string fname = var_subnode_["result_file"].as<std::string>();
    std::ofstream fout(fname);

    // Orientations of the kinetochore
    std::map<std::string, std::vector<double>> orientations;
    orientations["set0"] = std::vector<double>{0.0, 0.0, -1.0};
    orientations["set1"] = std::vector<double>{1.0, 0.0, 0.0};
    orientations["set2"] = std::vector<double>{0.0, 0.0, 1.0};
    orientations["set3"] = std::vector<double>{-1.0, 0.0, 0.0};

    int nvars = var_subnode_["nvars"].as<int>();
    // Start generating points....
    for (auto iorient : orientations) {
        for (int ivar = 1; ivar < nvars; ++ivar) {
            double ir = ((double)ivar / (double)(nvars - 1)) * 4.0;
            // Change microtubule x value to this
            properties.bonds.r_bond[0][0] = ir;

            //std::cout << "ivar: " << ivar << std::endl;
            //std::cout << "  r(" << properties.bonds.r_bond[0][0] << ", "
            //                    << properties.bonds.r_bond[0][1] << ", "
            //                    << properties.bonds.r_bond[0][2] << ")\n";
            //std::cout << "  u(" << properties.bonds.u_bond[0][0] << ", "
            //                    << properties.bonds.u_bond[0][1] << ", "
            //                    << properties.bonds.u_bond[0][2] << ")\n";
            //std::cout << "  l " << properties.bonds.length[0] << std::endl;

            // Directly access the kinetochore and force it to run the update_1_2 cycle
            int nkc = properties.chromosomes.nkcs_;
            for (int ikc = 0; ikc < nkc; ++ikc) {
                Kinetochore *kc = &(properties.chromosomes.kinetochores_[ikc]);
                // Change the orientation of the kinetochore
                kc->u_[0] = iorient.second[0];
                kc->u_[1] = iorient.second[1];
                kc->u_[2] = iorient.second[2];
                kc->ForceUnbind(&parameters,
                                &properties);
                kc->Update_1_2(parameters.n_dim,
                               0,
                               properties.bonds.n_bonds,
                               properties.unit_cell.h,
                               properties.bonds.r_bond,
                               properties.bonds.s_bond,
                               properties.bonds.u_bond,
                               properties.bonds.length);
                if (fout.is_open() && ikc == 1) {
                    fout << iorient.first << " " << properties.chromosomes.af_kr_ << " " << ir << " " << kc->n_exp_tot_ << std::endl;
                }
            }
        }
    }

    fout.close();

    return success;
}
