// Implementaiton of integration tests for xlinks

#include "bob.h"
#include "correlation_data.h"
#include "integration_test_xlinks.h"
#include "xlink_entry.h"
#include "lookup_table.h"
#include "unit_test_lookup_table.h"

#include <fstream>
#include <iostream>
#include <numeric>

// Run the tests and all their variants
void IntegrationTestXlinks::RunTests() {
    std::cout << "****************\n";
    std::cout << "Integraiton Test Xlinks run tests\n";
    std::cout << "****************\n";

    for (auto &kv : tests_) {
        std::cout << "----------------\n";
        std::cout << "Test : " << kv.first << std::endl;
        std::cout << "----------------\n";

        for (int iv = 0; iv < node_["Tests"][kv.first].size(); ++iv) {
        std::cout << "----------------\n";
            //std::cout << "  Variant : " << iv << std::endl;
            var_subnode_ = node_["Tests"][kv.first][iv];
            iv_ = iv;
            auto result = kv.second();

            if (!result) {
                std::cout << "Test : " << kv.first << " failed, check output!\n";
                exit(1);
            }
        }
    }
}

// Set the possible tests to run
void IntegrationTestXlinks::SetTests() {
    node_ = YAML::LoadFile(filename_);

    // Check against names to bind method calls
    if (node_["Tests"]["Generate2DXlinkDistribution"]) {
        tests_["Generate2DXlinkDistribution"] = std::bind(&IntegrationTestXlinks::Generate2DXlinkDistribution, this);
    }
    if (node_["Tests"]["PolarAffinityUnitTest"]) {
        tests_["PolarAffinityUnitTest"] = std::bind(&IntegrationTestXlinks::PolarAffinityUnitTest, this);
    }
    if (node_["Tests"]["BindingRatioStats"]) {
        tests_["BindingRatioStats"] = std::bind(&IntegrationTestXlinks::BindingRatioStats, this);
    }
    if (node_["Tests"]["CrosslinkLookupTables"]) {
        tests_["CrosslinkLookupTables"] = std::bind(&IntegrationTestXlinks::CrosslinkLookupTables, this);
    }
}

// Initialize the system of chromosomes (with other stuff toooo)
void IntegrationTestXlinks::InitSystem(system_parameters *parameters,
                                       system_properties *properties) {

    // Generic parameter initialization
    parse_parameters_node(var_subnode_["params"], parameters);
    gsl_rng_env_setup();
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    // Generate the unit cell
    int ndim = parameters->n_dim;
    properties->unit_cell.h = (double **) allocate_2d_array(ndim, ndim, sizeof(double));
    for (int i = 0; i < ndim; ++i) {
        properties->unit_cell.h[i][i] = 100;
    }

    // Get the microtubule information
    {
        int nmts = 0;
        if (var_subnode_["microtubules"]) {
            nmts = var_subnode_["microtubules"].size();
            init_unit_cell_structure(parameters, properties);
            int nsites = 2*nmts;
            parameters->n_spheros = properties->bonds.n_bonds = nmts;
            properties->sites.n_sites = nsites;
            properties->sites.v = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
            properties->sites.r = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
            init_site_structure_sphero(parameters, properties);
            init_bond_structure_sphero(parameters, properties);
            for (int imt = 0; imt < nmts; ++imt) {
                for (int i = 0; i < ndim; ++i) {
                    properties->bonds.r_bond[imt][i] = var_subnode_["microtubules"][imt]["r"][i].as<double>();
                    properties->bonds.u_bond[imt][i] = var_subnode_["microtubules"][imt]["u"][i].as<double>();
                }
                properties->bonds.length[imt] = var_subnode_["microtubules"][imt]["l"].as<double>();
            }
            // Print the bond information
            PrintMTs(parameters, properties);
        }
        else{ // Make default MTs
            nmts = 2;
            init_unit_cell_structure(parameters, properties);
            int nsites = 2*nmts;
            parameters->n_spheros = properties->bonds.n_bonds = nmts;
            properties->sites.n_sites = nsites;
            properties->sites.v = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
            properties->sites.r = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
            init_site_structure_sphero(parameters, properties);
            init_bond_structure_sphero(parameters, properties);

            double **r = properties->bonds.r_bond;
            double **u = properties->bonds.u_bond;
            //properties->bonds.r_bond[imt][i] = var_subnode_["microtubules"][imt]["r"][i].as<double>();
            if (ndim == 2){
                r[0][0] = r[0][1] = 0.0;
                r[1][0] = 0.0;
                r[1][1] = 3.0;
                // MTs are initialized parallel by default
                u[0][0] = u[1][0] =  1.0;
                u[0][1] = u[1][1] = 0.0;
            }
            else if ( ndim == 3){
                r[0][0] = r[0][1] = r[0][2] = 0.0;
                r[1][0] = r[1][2] = 0.0;
                r[1][1] = 3.0;
                // MTs are initialized parallel by default
                u[0][0] = u[0][1] = u[1][0] = u[1][1] = 0.0;
                u[0][2] = u[1][2] =  1.0;
            }
            for (int imt = 0; imt < nmts; ++imt) {
                properties->bonds.length[imt] = 80.0;
            }
        }
        // Create some force stuff
        f_comp_ = (double **) allocate_2d_array(nmts, ndim, sizeof(double));
        t_comp_ = (double **) allocate_2d_array(nmts, ndim, sizeof(double));
        virial_comp_ = (double **) allocate_2d_array(ndim, ndim, sizeof(double));
        calc_matrix_ = (int *) allocate_1d_array(nmts, sizeof(int));
    }

    // Load OMP settings
    set_omp_settings(parameters);
    print_omp_settings();

    #ifdef ENABLE_OPENMP
    #pragma omp parallel
    #pragma omp master
    #endif
    {
        int n_threads = properties->mp_local.n_threads = properties->rng_mt.n_threads = 1;

        #ifdef ENABLE_OPENMP
        n_threads = properties->mp_local.n_threads = properties->rng_mt.n_threads = omp_get_num_threads();
        #endif

        properties->rng_mt.rng = (rng_properties*)
            allocate_1d_array(properties->rng_mt.n_threads, sizeof(rng_properties));

        gsl_rng_env_setup();
        for (int i_thr = 0; i_thr < properties->rng_mt.n_threads; ++i_thr) {
            properties->rng_mt.rng[i_thr].T = gsl_rng_default;
            properties->rng_mt.rng[i_thr].r = gsl_rng_alloc(properties->rng_mt.rng[i_thr].T);
            gsl_rng_set(properties->rng_mt.rng[i_thr].r, (parameters->seed+1) * i_thr);
        }

        properties->mp_local.f_local = (double***)
            allocate_3d_array(n_threads, properties->bonds.n_bonds, parameters->n_dim, sizeof(double));

        properties->mp_local.t_local = (double***)
            allocate_3d_array(n_threads, properties->bonds.n_bonds, 3, sizeof(double));

        properties->mp_local.virial_local = (double***)
            allocate_3d_array(n_threads, parameters->n_dim, parameters->n_dim, sizeof(double));

        properties->mp_local.calc_local = (int**)
            allocate_2d_array(n_threads, properties->bonds.n_bonds, sizeof(int));
    }
}

void IntegrationTestXlinks::InitXlinkParams( system_parameters *parameters,
                                             system_properties *properties) {
    InitXlinkParams(parameters, properties, var_subnode_["xlink"]);
}

void IntegrationTestXlinks::InitXlinkParams(system_parameters *parameters,
                                            system_properties *properties,
                                            YAML::Node param_node) {
    std::string x_file = parameters->crosslink_file;
    YAML::Node xlink_node = GetXlinkParamNode(x_file, param_node);
    properties->crosslinks.Init(parameters, properties, &xlink_node);
    FinishInitSystem(parameters,properties);
}

YAML::Node IntegrationTestXlinks::GetXlinkParamNode(std::string x_file, YAML::Node mod_node){
    std::cout << "xlink file: " << x_file << std::endl;
    YAML::Node xlink_node = YAML::LoadFile(x_file.c_str());
    // Vector of all important xlink paramters with double values
    std::vector<std::string> param_strs = {  "spring_constant",
                                             "velocity",
                                             "velocity_polar_scale",
                                             "velocity_antipolar_scale",
                                             "equilibrium_length",
                                             "barrier_weight",
                                             "characteristic_length",
                                             "polar_affinity",
                                             "pa_exp_split",
                                             "diffusion_bound",
                                             "diffusion_bound_2",
                                             "concentration_1",
                                             "concentration_2",
                                             "on_rate_1",
                                             "on_rate_2",
                                             "stall_force"
                                           };
    if (mod_node["force_dependent"])
        xlink_node["force_dependent"] = mod_node["force_dependent"].as<bool>();
    if (mod_node["reservoir"])
        xlink_node["crosslink"][0]["reservoir"] = mod_node["reservoir"].as<int>();
    for (auto param : param_strs){
        // All double value parameters are checked to see if they exist or not in test yaml file
        // if so override the default ones from conventional xlink yaml file
        if (YAML::Node p_node = mod_node[param])
            xlink_node["crosslink"][0][param] = p_node.as<double>();
    }
    return xlink_node;
}

// Finish initializing the system
void IntegrationTestXlinks::FinishInitSystem(system_parameters *parameters,
                                            system_properties *properties) {
    // Set up the neighbor lists
    properties->neighbors.neighbs = new nl_list[properties->bonds.n_bonds];
    update_neighbor_lists_sphero_all_pairs_mp(parameters->n_dim, parameters->n_periodic,
                                              properties->unit_cell.h,
                                              parameters->skin, parameters->r_cutoff,
                                              properties->bonds.n_bonds,
                                              properties->bonds.r_bond,
                                              properties->bonds.s_bond,
                                              properties->bonds.u_bond,
                                              properties->bonds.length,
                                              properties->neighbors.neighbs,
                                              parameters->nl_twoway_flag);
    // Calculate distance between rods and place value into neighborlists
    brownian_sphero_neighbor_lists(parameters, properties,
                                   f_comp_, virial_comp_,
                                   t_comp_, calc_matrix_);

}

void IntegrationTestXlinks::EquilibrateXlinks( system_parameters *parameters,
                                               system_properties *properties){

    int nstepsequil = MAX(10 / (0.5 * (properties->crosslinks.on_rate_1_[0][0] + properties->crosslinks.on_rate_1_[0][1]) * parameters->delta),
                          10 / (0.5 * (properties->crosslinks.on_rate_2_[0][0] + properties->crosslinks.on_rate_2_[0][1]) * parameters->delta));

    std::cout << "Running " << nstepsequil << " equilibration steps\n";
    std::cout << "  equivalent time: " << nstepsequil*parameters->delta << std::endl;
    for (int i = 0; i < nstepsequil; ++i) {
        // Run xlink KMC cycle
        properties->crosslinks.StepKMC(parameters, properties);

        // Run a crosslink potential cycle (needed for doubly bound xlinks to move!)
        crosslink_interaction_bd_mp(parameters,
                                    properties,
                                    f_comp_,
                                    virial_comp_,
                                    t_comp_,
                                    calc_matrix_);

        if (i % 100000 == 0) {
            std::cout << "Step[" << i << "] readout\n";
            properties->crosslinks.PrintSimulationStep(false);
        }
    }
    std::cout << "Equilibration run complete, starting measurements\n";
}

// TODO Comment
void IntegrationTestXlinks::SetXlinksToSinglyBound( system_parameters *parameters,
                                                    system_properties *properties,
                                                    double r0){
    int i_type = 0;
    int ibond = 0;
    int nbonds = properties->bonds.n_bonds;
    XlinkManagement *xlink_mgt = &(properties->crosslinks);
    int nxlinks = xlink_mgt->n_tot_[i_type];
    // Clear all xlinks
    xlink_mgt->stage_0_xlinks_[i_type].clear();
    xlink_mgt->n_free_[i_type] = 0;
    for ( int jbond = 0; jbond < nbonds; jbond++){
        xlink_mgt->stage_1_xlinks_[i_type][jbond].clear();
        xlink_mgt->n_bound_1_[i_type][0] = 0;
        xlink_mgt->n_bound_1_[i_type][1] = 0;
        xlink_mgt->stage_2_xlinks_[i_type][jbond].clear();
        xlink_mgt->n_bound_2_[i_type] = 0;
    }
    for (int xit = 0; xit < nxlinks; ++xit) {
        int headtype = 0;
        int parent = ibond;
        XlinkEntry new_link(xlink_mgt);
        new_link.SetActive();
        new_link.type_ = i_type;
        new_link.head_type_[0] = headtype;
        new_link.head_parent_[0] = parent;
        new_link.cross_position_[0] = r0;
        xlink_mgt->PushStage1(&new_link);
    }
}

// TODO Comment
void IntegrationTestXlinks::SetXlinksToDoublyBound( system_parameters *parameters,
                                                    system_properties *properties,
                                                    double r0, double r1){
    int i_type = 0;
    int bond_1 = 0;
    int head_1 = 0;
    int bond_2 = 1;
    int head_2 = 1;
    int nbonds = properties->bonds.n_bonds;
    XlinkManagement *xlink_mgt = &(properties->crosslinks);
    int nxlinks = xlink_mgt->n_tot_[i_type];
    // Clear all xlinks
    xlink_mgt->stage_0_xlinks_[i_type].clear();
    xlink_mgt->n_free_[i_type] = 0;
    for ( int jbond = 0; jbond < nbonds; jbond++){
        xlink_mgt->stage_1_xlinks_[i_type][jbond].clear();
        xlink_mgt->n_bound_1_[i_type][0] = 0;
        xlink_mgt->n_bound_1_[i_type][1] = 0;
        xlink_mgt->stage_2_xlinks_[i_type][jbond].clear();
        xlink_mgt->n_bound_2_[i_type] = 0;
    }
    for (int xit = 0; xit < nxlinks; ++xit) {
        XlinkEntry new_link(xlink_mgt);
        new_link.SetActive();
        new_link.type_ = i_type;
        new_link.head_type_[0] = head_1;
        new_link.head_type_[1] = head_2;
        new_link.head_parent_[0] = bond_1;
        new_link.head_parent_[1] = bond_2;
        new_link.cross_position_[0] = r0;
        new_link.cross_position_[1] = r1;
        //Calculate force and more importantly energy for new xlink
        double f[3];
        double u = new_link.CalcForce(parameters->n_dim,
                           parameters->n_periodic,
                           properties->unit_cell.h,
                           properties->bonds.r_bond,
                           properties->bonds.s_bond,
                           properties->bonds.u_bond,
                           properties->bonds.length,
                           f);
        xlink_mgt->PushStage2(&new_link);
    }
}

bool IntegrationTestXlinks::CheckSelectedAnswersLookup(LookupTable *lut,
                                                     std::map<double, std::map<double, double>> *sa) {
    bool success = true;

    auto x = (*lut).x_;

    for (auto &kv0 : *sa) {
        for (auto &kv1 : kv0.second) {
            double x[2] = {kv0.first,kv1.first};
            double retval = lut->Lookup(x);
            double ans = kv1.second;
            std::cout << "    value  (x: " << kv0.first << ", y: " << kv1.first
                << "), = " << retval << std::endl;
            std::cout << "    answer (x: " << kv0.first << ", y: " << kv1.first
                << "), = " << ans << std::endl;
            if (!almost_equal(retval, ans, 0.00001)) success = success && false;
        }
    }

    return success;
}
// Generate the invert distribution of attachments along MT
bool IntegrationTestXlinks::Generate2DXlinkDistribution() {
    bool success = true;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);
    // Create the xlink node based on tradition crosslink yaml file and then overide with test yaml file
    InitXlinkParams(&parameters, &properties);
    //FinishInitSystem(&parameters, &properties);

    // Set up some correlation data to track the distribution
    CorrelationData motor_dist_1d;
    CorrelationData motor_dist_2d;
    CorrelationData n_exp_dist;
    CorrelationData psi1_dist;
    double fudge = 0.0;
    {
        double bin_size[] = {0.05};
        double bin_min[] = {-5};
        double bin_max[] = {5};
        motor_dist_1d.Init(1, bin_size, bin_min, bin_max);
    }
    {
        double bin_size[] = {0.05, 0.05};
        double bin_min[] = {-fudge, -fudge};
        double bin_max[] = {properties.bonds.length[0] + fudge, properties.bonds.length[1] + fudge};
        motor_dist_2d.Init(2, bin_size, bin_min, bin_max);
    }
    {
        double bin_size[] = {0.05};
        double bin_min[] = {0-fudge};
        double bin_max[] = {properties.bonds.length[0]+fudge};
        psi1_dist.Init(1, bin_size, bin_min, bin_max);
        n_exp_dist.Init(1, bin_size, bin_min, bin_max);
    }

    EquilibrateXlinks(&parameters, &properties);

    int nsteps = parameters.n_steps;
    std::cout << "Running " << nsteps << " steps\n";
    std::cout << "  equivalent time: " << nsteps*parameters.delta << std::endl;
    for (int i = 0; i < nsteps; ++i) {
        properties.i_current_step = i;

        // Run xlink KMC cycle
        properties.crosslinks.StepKMC(&parameters, &properties);

        // Run a crosslink potential cycle (needed for doubly bound xlinks to move!)
        crosslink_interaction_bd_mp(&parameters,
                                    &properties,
                                    f_comp_,
                                    virial_comp_,
                                    t_comp_,
                                    calc_matrix_);

        int itype = 0; int ibond = 0;

        for (xlink_list::iterator xlink = properties.crosslinks.stage_2_xlinks_[itype][ibond].begin();
                                  xlink < properties.crosslinks.stage_2_xlinks_[itype][ibond].end();
                                  ++xlink) {
            if (xlink->IsActive()) {
                double f[3] = {0.0};
                xlink->CalcForce(parameters.n_dim,
                                 0,
                                 properties.unit_cell.h,
                                 properties.bonds.r_bond,
                                 properties.bonds.s_bond,
                                 properties.bonds.u_bond,
                                 properties.bonds.length,
                                 f);

                // Should stil be active
                if ((xlink->cross_position_[0] > 10) &&
                    (xlink->cross_position_[1] > 10)) {
                    double s = xlink->cross_position_[1] - xlink->cross_position_[0];
                    motor_dist_1d.Bin(1, &s, 1.0);
                }
                motor_dist_2d.Bin(2, xlink->cross_position_, 1.0);
            }
        }

        // Do the stage1 calculations too
        for (xlink_list::iterator xlink = properties.crosslinks.stage_1_xlinks_[itype][ibond].begin();
                                  xlink < properties.crosslinks.stage_1_xlinks_[itype][ibond].end();
                                  ++xlink) {
            if (xlink->IsActive()) {
                psi1_dist.Bin(1, &xlink->cross_position_[0], 1.0);
                n_exp_dist.Bin(1, &xlink->cross_position_[0], xlink->n_exp_);
            }
        }

        if (i % 100000 == 0) {
            std::cout << "Step[" << i << "] readout\n";
            properties.crosslinks.PrintSimulationStep(false);
        }
    }

    std::string fname_base = var_subnode_["result_file"].as<std::string>();
    std::string fname_1d    = "xlink_" + fname_base + "_1d.dat";
    std::string fname_2d    = "xlink_" + fname_base + "_2d.dat";
    std::string fname_psi1  = "xlink_" + fname_base + "_psi1.dat";
    std::string fname_nexp  = "xlink_" + fname_base + "_nexp.dat";

    motor_dist_1d.NormalizeByConstant(1.0/parameters.n_steps);
    motor_dist_2d.NormalizeByConstant(1.0/parameters.n_steps);
    psi1_dist.NormalizeByConstant(1.0/parameters.n_steps);
    n_exp_dist.NormalizeByConstant(1.0/parameters.n_steps);

    motor_dist_1d.OutputBinary(fname_1d.c_str());
    motor_dist_2d.OutputBinary(fname_2d.c_str());
    psi1_dist.OutputBinary(fname_psi1.c_str());
    n_exp_dist.OutputBinary(fname_nexp.c_str());

    return success;
}

bool IntegrationTestXlinks::PolarAffinityUnitTest() {
    bool success = true;
    double SMALL = .00001;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);

    // Create the xlink node based on tradition crosslink yaml file and then overide with test yaml file
    if (!var_subnode_["input"]["polar_affinity"])
        var_subnode_["input"]["polar_affinity"].push_back(1.0);
    if (!var_subnode_["input"]["pa_exp_split"])
        var_subnode_["input"]["pa_exp_split"].push_back(1.0);

    var_subnode_["xlink"]["polar_affinity"] = var_subnode_["input"]["polar_affinity"].as<double>();
    var_subnode_["xlink"]["pa_exp_split"] = var_subnode_["input"]["pa_exp_split"].as<double>();
    InitXlinkParams(&parameters, &properties);
    //FinishInitSystem(&parameters, &properties);

    // Crosslink management polar affinity parameters
    double pa = properties.crosslinks.polar_affinity_[0];
    double pa_off = properties.crosslinks.pa_off_[0];
    double pa_on = pa * pa_off;
    // Orientation vectors
    double u1[3] = {0.0, 0.0, 1.0};
    double u2[3] = {0.0, 0.0, -1.0};
    // Make sure crosslinks pa_off calculation was accurate
    if (ABS(pa_off - var_subnode_["output"]["pa_off"].as<double>()) > SMALL) {
        std::cout<< "**pa_off original failed** "<<pa_off<<"\n";
        return false;
    }
    // Parallel check
    double pa_on_calc = properties.crosslinks.PolarAffinityOn(parameters.n_dim, 0, u1, u1);
    if (ABS(pa_on_calc-pa_on) > SMALL) {
        std::cout<< "**pa_on parallel method failed**\n";
        return false;
    }
    double pa_off_calc = properties.crosslinks.PolarAffinityOff(parameters.n_dim, 0, u1, u1);
    if (ABS(pa_off_calc-pa_off) > SMALL) {
        std::cout<< "**pa_off parallel method failed**\n";
        return false;
    }
    // Anti-parallel check, should just be 1.0 for both of these
    pa_on_calc = properties.crosslinks.PolarAffinityOn(parameters.n_dim, 0, u1, u2);
    if (pa_on_calc != 1.0 ){
        std::cout<< "**pa_on anti-arallel method failed**\n";
        return false;
    }
    pa_off_calc = properties.crosslinks.PolarAffinityOff(parameters.n_dim, 0, u1, u2);
    if (pa_on_calc != 1.0 ) {
        std::cout<< "**pa_on anti-arallel method failed**\n";
        return false;
    }
    return success;
}

bool IntegrationTestXlinks::BindingRatioStats() {
    bool success = true;
    double SMALL = .00001;
    double nexp_on = -1;
    double nexp_off = -1;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);

    int nsteps = parameters.n_steps; // Total steps of test
    std::cout << "Running " << nsteps << " steps\n";
    // On rate testing
    std::vector<binding_data> Non_data;
    if (var_subnode_["xlink"].IsSequence()){
        for (int i_yn=0; i_yn< var_subnode_["xlink"].size(); i_yn++){
            binding_data data; 
            InitXlinkParams(&parameters, &properties, var_subnode_["xlink"][i_yn]);
            std::cout << "      On Rate Testing: " << i_yn<< "/"<< var_subnode_["xlink"].size() 
                      << std::endl;
            SingleBindingTest(&parameters, &properties, &data);
            data.result = var_subnode_["xlink"][i_yn]["results"]["n_on_exp"].as<double>();
            Non_data.push_back(data);
        }
    }
    else{
        binding_data data; 
        InitXlinkParams(&parameters, &properties, var_subnode_["xlink"]);
        std::cout << "      On Rate Testing " << std::endl;
        SingleBindingTest(&parameters, &properties, &data);
        data.result = var_subnode_["xlink"]["results"]["n_on_exp"].as<double>();
        Non_data.push_back(data);
    }
    // Off rate testing
    std::vector<binding_data> Noff_data;
    if (var_subnode_["xlink"].IsSequence()){
        for (int i_yn=0; i_yn< var_subnode_["xlink"].size(); i_yn++){
            binding_data data; 
            InitXlinkParams(&parameters, &properties, var_subnode_["xlink"][i_yn]);
            std::cout << "      Off Rate Testing: " << i_yn<< "/"<< var_subnode_["xlink"].size() 
                      << std::endl;
            DoubleBindingTest(&parameters, &properties, &data);
            data.result = var_subnode_["xlink"][i_yn]["results"]["n_off_exp"].as<double>();
            Noff_data.push_back(data);
        }
    }
    else{
        binding_data data; 
        InitXlinkParams(&parameters, &properties, var_subnode_["xlink"]);
        std::cout << "      Off Rate Testing " << std::endl;
        DoubleBindingTest(&parameters, &properties, &data);
        data.result = var_subnode_["xlink"]["results"]["n_off_exp"].as<double>();
        Non_data.push_back(data);
    }
    std::cout << " --------- End of Varient Run: " << iv_ << " ---------- "<< std::endl;
    std::cout << std::endl << "      On Rate Results" << std::endl;
    for (auto d : Non_data){
        std::cout << "      --- N_on calc: " << d.result << ", N_on exp: " << d.N_exp << ", N_on Avg: " << d.N_avg
                  << ", N_on Error: " << d.N_err;
        // Test lookup table prediction
        if (fabs(d.N_exp - d.result)/d.result > SMALL){
            std::cout << " (FAILED) ";
            success = false;
        }
        // Test actual binding averages
        else if (fabs(d.N_avg - d.result) > (4.0*d.N_err)){
            std::cout << " (FAILED) ";
            success = false;
        }
        std::cout << std::endl;
    }
    std::cout << std::endl << "      Off Rate Results "<< std::endl;
    for (auto d : Noff_data){
        std::cout << "      --- N_off Calc: "<< d.result << ", N_off Avg: " << d.N_avg 
                  << ", N_off Error: " << d.N_err << std::endl;
        if (fabs(d.N_avg - d.result) > (4.0*d.N_err)){
            std::cout << " (FAILED) ";
            success = false;
        }
    }
    std::cout << std::endl;
    //std::cout << "      --- Association Constant: " << assoc_const << std::endl;

    return success;
}

void IntegrationTestXlinks::PrintMTs(system_parameters *parameters, system_properties *properties ) {
    int ndim = parameters->n_dim;
    int nmts = properties->bonds.n_bonds;
    for (int imt = 0; imt < nmts; ++imt) {
        std::cout << "New Microtubule:\n";
        std::cout << "  r(" << properties->bonds.r_bond[imt][0] << ", "
                            << properties->bonds.r_bond[imt][1] << ", "
                            << properties->bonds.r_bond[imt][2] << ")\n";
        std::cout << "  u(" << properties->bonds.u_bond[imt][0] << ", "
                            << properties->bonds.u_bond[imt][1] << ", "
                            << properties->bonds.u_bond[imt][2] << ")\n";
        std::cout << "  l " << properties->bonds.length[imt] << std::endl;
    }
}


bool IntegrationTestXlinks::CrosslinkLookupTables() {

    bool success = true;
    double SMALL = .00001;
    double nexp_on = -1;
    double nexp_off = -1;
    int i_type = 0;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);
    InitXlinkParams(&parameters, &properties);
    //FinishInitSystem(&parameters, &properties);

    XlinkManagement *xlink_mgt = &(properties.crosslinks);
    //// Make answers to check against
    //std::map<double, std::map<double, double>> selected_answers;
    //for (auto nit = var_subnode_["results"].begin();
    //          nit != var_subnode_["results"].end();
    //          ++nit) {
    //    YAML::Node mysubnode = *nit;
    //    double a = mysubnode["a"].as<double>();
    //    //double b = mysubnode["b"].as<double>();
    //    double ans = mysubnode["ans"].as<double>();
    //    selected_answers[a][b] = ans;
    //}
    //success = CheckSelectedAnswersLookup( &xlink_mgt->n_exp_lookup_[i_type],
    //                                      &selected_answers);

    double a = 7.0;
    double b = 3;
    double x[2] = {fabs(a), b};
    double term0 = xlink_mgt->n_exp_lookup_[i_type].Lookup(x) * ((a < 0) ? -1.0 : 1.0);
    std::cout << "    value  (x: " << a << ", y: " << b << "), = " << term0 << std::endl;

    return success;
}




void IntegrationTestXlinks::SingleBindingTest(system_parameters *parameters, system_properties *properties,
                                              binding_data* data){
    double SMALL = .00001;
    double nexp_on = -1;

    int nsteps = parameters->n_steps;
    data->ZeroData(nsteps);
    //std::vector<int> n2_bound(nsteps, 0);
    double *l = properties->bonds.length;
    double r0 = l[0]/2.0; double r1 = l[1]/2.0;
    for (int i = 0; i < nsteps; ++i) {
        properties->i_current_step = i;
        SetXlinksToSinglyBound( parameters, properties, r0);
        // Run xlink KMC cycle for two stage crosslinks
        properties->crosslinks.PrepKMC(parameters, properties);
        if (nexp_on == -1){
            nexp_on = properties->crosslinks.n_exp_[0];
            //std::cout << "      --- N_exp On: " << nexp_on << std::endl;
            data->N_exp = nexp_on;
        }
        properties->crosslinks.TwoStageKMC(parameters, properties);

        // Run a crosslink potential cycle (needed for doubly bound xlinks to move!)
        crosslink_interaction_bd_mp(parameters,
                                    properties,
                                    f_comp_,
                                    virial_comp_,
                                    t_comp_,
                                    calc_matrix_);
        int itype = 0; int ibond = 0;
        for (xlink_list::iterator xlink = properties->crosslinks.stage_2_xlinks_[itype][ibond].begin();
                                  xlink < properties->crosslinks.stage_2_xlinks_[itype][ibond].end();
                                  ++xlink) {
            if (xlink->IsActive()) {
                double f[3] = {0.0};
                xlink->CalcForce(parameters->n_dim,
                                 0,
                                 properties->unit_cell.h,
                                 properties->bonds.r_bond,
                                 properties->bonds.s_bond,
                                 properties->bonds.u_bond,
                                 properties->bonds.length,
                                 f);
                data->N_bound[i]++;
            }
        }
    }
    data->CalcStats();
}

void IntegrationTestXlinks::DoubleBindingTest(system_parameters *parameters, system_properties *properties,
                                              binding_data* data){
    double SMALL = .00001;
    double nexp_off = -1;
    int itype = 0;

    int nsteps = parameters->n_steps;
    data->ZeroData(nsteps);

    double *l = properties->bonds.length;
    double r0 = l[0]/2.0; double r1 = l[1]/2.0;
    for (int i = 0; i < nsteps; ++i) {
        properties->i_current_step = i;
        SetXlinksToDoublyBound(parameters, properties, r0, r1);
        properties->crosslinks.StepKMC(parameters, properties);
        crosslink_interaction_bd_mp(parameters,
                                    properties,
                                    f_comp_,
                                    virial_comp_,
                                    t_comp_,
                                    calc_matrix_);
        // Find out how many xlinks stepped off
        for (int ibond = 0; ibond < properties->bonds.n_bonds; ibond++){
            for (xlink_list::iterator xlink = properties->crosslinks.stage_1_xlinks_[itype][ibond].begin();
                                      xlink < properties->crosslinks.stage_1_xlinks_[itype][ibond].end();
                                      ++xlink) {
                if (xlink->IsActive()) {
                    data->N_bound[i]++;
                }
            }
        }
    }
    data->CalcStats();
}

void binding_data::CalcStats(){
    // Bound average per step
    N_avg = double(std::accumulate(N_bound.begin(), N_bound.end(), 0)); // Sums all points
    N_avg /= double(N_bound.size()); // Get the average
    // Get standard error of bound per step
    N_err = 0;
    for (auto& n : N_bound)
        N_err += SQR(n - N_avg);
    N_err = sqrt(N_err)/double(N_bound.size());
}

