// Kinetochore entries

#include "bob.h"
#include "kinetochore.h"
#include "lookup_table_advanced.h"
#include "minimum_distance.h"
#include "probabilities.h"
#include "triangle_mesh.h"

#include <iostream>

#define SKIN_ 8.0

Kinetochore::Kinetochore() : parent_(NULL), idx_(-1), cidx_(-1) {

}

void Kinetochore::Init(ChromosomeManagement *parent,
                       int idx,
                       int cidx,
                       bool *sister,
                       double *r,
                       double *u,
                       double *ueff,
                       double *v,
                       double *w,
                       double vshift) {
    parent_ = parent;
    idx_ = idx;
    cidx_ = cidx;

    ndim_ = parent_->ndim_;
    nsites_ = parent_->naf_;
    nbonds_ = parent_->properties_->bonds.n_bonds;
    if (parent_->properties_->anchors.n_anchors > 0) {
        nmt_per_spb_ = parent_->properties_->bonds.n_bonds / parent_->properties_->anchors.n_anchors;
    } else {
        nmt_per_spb_ = 1;
    }

    second_sister_ = sister;
    r_ = r;
    u_ = u;
    ueff_ = ueff;
    v_ = v;
    w_ = w;

    // Assign myself the proper triangulation
    tri_ = &(parent_->tris_[idx]);
    vertex_rel_ = parent_->vertex_rel_[idx];
    nverts_ = parent_->nkctriverts_;
    attach_status_ = &(parent_->kcattachmentstatus_[idx]);

    // Assign the proper type of rate calculation
    if (parent_->af_potential_type_ == "harmonic") {
        force_type_ = 0;
        prob_1_2_func = &prob_1_2_advanced_full;
    } else if (parent_->af_potential_type_ == "quartic") {
        force_type_ = 1;
        prob_1_2_func = &prob_1_2_advanced_quartic;
    } else {
        std::cerr << "Wrong type of potential rate calculation: " << parent_->af_potential_type_ << std::endl;
        exit(1);
    }

    CreateRefVectors(vshift);

    CreateBindingSites();

    CreateKinetochoreTriangulation();

    Allocate();
}

// Reinit the kinetochore, which means reconfigure without reallocating everything and setting up stuff
void Kinetochore::ReInit() {
    CreateRefVectors(0.0);
    CreateBindingSites();
    CreateKinetochoreTriangulation();
}

void Kinetochore::PrintFrame() {
    std::cout << "  KC[" << idx_ << "]\n";
    // Position/force information
    double *f = parent_->f_[idx_];
    std::cout << "    secondsister: " << *second_sister_ << std::endl;
    std::cout << "    r (" << r_[0] << ", " << r_[1] << ", " << r_[2] << ")\n";
    std::cout << "    u (" << u_[0] << ", " << u_[1] << ", " << u_[2] << ")\n";
    std::cout << "    ueff(" << ueff_[0] << ", " << ueff_[1] << ", " << ueff_[2] << ")\n";
    std::cout << "    f (" << f[0] << ", " << f[1] << ", " << f[2] << ")\n";

    // KMC/site stuff
    std::cout << "    attachment_status: " << *attach_status_ << std::endl;
    std::cout << "    nbound: " << parent_->n_bound_[idx_] << ", nexptot: " << n_exp_tot_ << std::endl;
    for (int isite = 0; isite < nsites_; ++isite) {
        if (attach_[isite] == -1) {
            std::cout << "    site[" << isite << "] nexp: " << n_exp_[isite] << std::endl;
        } else {
            std::cout << "    site[" << isite << "] bond: " << attach_[isite] << ", loc: " << cross_pos_[isite]
                << ", ";
            int ibond = attach_[isite];
            double dstab = parent_->properties_->bonds.length[ibond] - cross_pos_[isite];
            if (dstab < parent_->af_tip_distance_) {
                std::cout << "TIP  ";
            } else {
                std::cout << "SIDE ";
            }

            if (parent_->properties_->bonds.poly_state[ibond] == GROW) {
                std::cout << "GROW\n";
            } else if (parent_->properties_->bonds.poly_state[ibond] == SHRINK) {
                std::cout << "SHRINK\n";
            } else {
                std::cout << "PAUSE\n";
            }
        }
    }
}

void Kinetochore::ConsistencyCheck() {
    // First, check out position, and if we're outside the boundary or not...
    double conf_rad = 0.5 * parent_->properties_->unit_cell.h[0][0];
    double conf_rad2 = SQR(conf_rad);
    double rmag2 = dot_product(ndim_, r_, r_);
    if (rmag2 > conf_rad2) {
        PrintFrame();
        std::cerr << "  ERROR: outside boundary, rmag: " << sqrt(rmag2) << std::endl;
        exit(1);
    }
}

double Kinetochore::GetDrawDiameter() {
    return parent_->kc_diameter_;
}

float *Kinetochore::GetColor() {
    return parent_->color_[cidx_];
}

float *Kinetochore::GetColorAF() {
    return parent_->color_af_[cidx_];
}

void Kinetochore::CreateRefVectors(double vshift) {
    // Make sure that u is actually a unit vector
    double norm_factor = 1.0/sqrt(dot_product(ndim_, u_, u_));
    for (int i = 0; i < ndim_; ++i) {
        u_[i] *= norm_factor;
    }

    double vect1[3] = { 1.0, 0, vshift };
    double vect2[3] = { 0.0, 1.0, vshift };
    if (1.0 - ABS(u_[0]) > 1e-2)
        cross_product(u_, vect1, v_, 3);
    else
        cross_product(u_, vect2, v_, 3);

    norm_factor = sqrt(1.0/dot_product(3, v_, v_));
    for (int i = 0; i < 3; ++i)
        v_[i] *= norm_factor;
    
    cross_product(u_, v_, w_, 3);
}

void Kinetochore::CreateKinetochoreTriangulation() {
    //std::cout << "Creating kinetochore triangulation\n";
    std::vector<std::array<double, 3>> verts;
    std::vector<int> indicies;

    double rkc = 0.5 * parent_->kc_diameter_;

    if (parent_->kc_triangulation_type_ == "hexagon") {
        std::cout << "Kinetochore creating standard hexagon\n";
        for (int iv = 0; iv < nverts_; ++iv) {
            std::array<double, 3> vertex;

            vertex_rel_[iv][1] = rkc * cos(2.*M_PI/nverts_*iv);
            vertex_rel_[iv][2] = rkc * sin(2.*M_PI/nverts_*iv);
            vertex_rel_[iv][0] = 0.0;

            // Now, generate the lab frame coordinates
            for (int i = 0; i < ndim_; ++i) {
                vertex[i] = r_[i] +
                            vertex_rel_[iv][0]*u_[i] +
                            vertex_rel_[iv][1]*v_[i] +
                            vertex_rel_[iv][2]*w_[i];
            }
            verts.push_back(vertex);
        }
    } else if ((parent_->kc_triangulation_type_ == "rectangle") || 
               (parent_->kc_triangulation_type_ == "point")) {
        // 4 vertex points for a rectangle
        // Corner 0
        vertex_rel_[0][0] = 0.0;
        vertex_rel_[0][1] = parent_->kc_dimensions_[0]/2.;
        vertex_rel_[0][2] = parent_->kc_dimensions_[1]/2.;

        // Corner 1
        vertex_rel_[1][0] = 0.0;
        vertex_rel_[1][1] = -parent_->kc_dimensions_[0]/2.;
        vertex_rel_[1][2] = parent_->kc_dimensions_[1]/2.;

        // Corner 2
        vertex_rel_[2][0] = 0.0;
        vertex_rel_[2][1] = -parent_->kc_dimensions_[0]/2.;
        vertex_rel_[2][2] = -parent_->kc_dimensions_[1]/2.;

        // Corner 3
        vertex_rel_[3][0] = 0.0;
        vertex_rel_[3][1] = parent_->kc_dimensions_[0]/2.;
        vertex_rel_[3][2] = -parent_->kc_dimensions_[1]/2.;

        // Generate lab frame coordinates
        for (int iv = 0; iv < nverts_; ++iv) {
            std::array<double, 3> vertex;

            for (int i = 0; i < ndim_; ++i) {
                vertex[i] = r_[i] +
                            vertex_rel_[iv][0]*u_[i] +
                            vertex_rel_[iv][1]*v_[i] +
                            vertex_rel_[iv][2]*w_[i];
            }
            verts.push_back(vertex);
        }
    } 

    create_polygon_mesh(nverts_, indicies);
    // create the triangle mesh
    create_polygon(tri_, verts, indicies);
    //print_polygon(tri_);
}

void Kinetochore::CreateBindingSites() {
    // New way to doing kinetochores where they are discs
    //std::cout << "Creating binding sites\n";

    //double rallowed = parent_->af_location_radius_;
    double rexclude = parent_->af_exclusion_radius_;
   
    *second_sister_ = (idx_ % 2 != 0);

    // Assume that the allowed radius is the radius around the u_eff vector
    // which is reversed for the second sister kinetochore
    for (int i = 0; i < ndim_; ++i) {
        if (*second_sister_) {
            ueff_[i] = -u_[i];
        } else {
            ueff_[i] = u_[i]; 
        }
    }

    double rkc = 0.5 * parent_->kc_diameter_;
    // 2d object, just make sure that we're good inside of this
    bool overlap = true;
    double pos_vec[3] = {0.0};

    // How many to insert?
    for (int isite = 0; isite < parent_->naf_; ++isite) {
        if (parent_->kc_triangulation_type_ == "hexagon") {
            if (!parent_->sites_specified_) {
                int itrial = 0;
                do {
                    overlap = false;
                    itrial++;
                    // Generate a random location in the kinetochore radius
                    double theta = gsl_rng_uniform(parent_->properties_->rng.r) * 2*M_PI;
                    double rho = sqrt(gsl_rng_uniform(parent_->properties_->rng.r))*rkc;
                    pos_vec[1] = rho*cos(theta);
                    pos_vec[2] = rho*sin(theta);
                    pos_vec[0] = 0.0; // no x extent

                    // Now check for overlaps with other sites
                    for (int isite2 = 0; isite2 < parent_->anchors_[idx_].size(); ++isite2) {
                        auto otheranchor = parent_->anchors_[idx_][isite2]; 
                        double rmag2 = 0.0;
                        for (int i = 0; i < ndim_; ++i) {
                            double drmag = otheranchor.pos_rel[i] - pos_vec[i];
                            rmag2 += SQR(drmag);
                        }
                        //std::cout << "site2: " << isite2 << ", rmag2: " << rmag2 << std::endl;
                        overlap = overlap || (rmag2 < SQR(rexclude));
                    }

                    //std::cout << "site: " << isite << " trial: " << itrial << std::endl;
                    
                    if (parent_->anchors_[idx_].size() == 0) {
                        overlap = false;
                    } 

                } while (overlap);
            } else {
                // Dirty hack, ugh, set to just along x axis for the binding sites
                // Set to 2pi/nsites along outside edge of kinetochore
                //std::cout << "Directly setting site information\n";
                pos_vec[0] = 0.0;
                pos_vec[1] = rkc * cos(2*M_PI*isite/parent_->naf_);
                pos_vec[2] = rkc * sin(2*M_PI*isite/parent_->naf_);
            }
        } else if (parent_->kc_triangulation_type_ == "rectangle") {
            // Align along the kinetochore in a linear fashion
            double myoffset = ( parent_->naf_ % 2 == 0 ? 0.5 * rexclude : 0.0);
            pos_vec[0] = 0.0;
            pos_vec[1] = (isite - parent_->naf_/2) * rexclude + myoffset;
            pos_vec[2] = 0.0; // No distance along what
        } else if (parent_->kc_triangulation_type_ == "point") {
            // Align the binding site with the center of the kinetochore
            pos_vec[0] = pos_vec[1] = pos_vec[2] = 0.0;
        } else {
            std::cerr << "Wrong kind of triangulation type: " << parent_->kc_triangulation_type_ << std::endl;
            exit(1);
        }

        //std::cout << "Chose pos_vec: (" << pos_vec[0] << ", "
        //                                << pos_vec[1] << ", "
        //                                << pos_vec[2] << ")\n";
        //std::cout << "u_eff: (" << u_eff[0] << ", "
        //                        << u_eff[1] << ", "
        //                        << u_eff[2] << ")\n";
        // This position is relative inside the frame of the kinetochore, so we
        // need to make sure to update it appropriately when it moves
        al_entry new_anchor;
        new_anchor.label = idx_; // kc label
        // Generate the lab frame position
        // Really, first generate the relative position in the lab frame, since we have the
        // distance of the binding site in the frame of the kinetochore
        for (int idim = 0; idim < ndim_; ++idim) {
            new_anchor.pos_rel[idim] = pos_vec[idim];
        }
        // Now, generate the lab frame coordinates
        for (int i = 0; i < ndim_; ++i) {
            new_anchor.pos[i] = r_[i] + 
                                new_anchor.pos_rel[0]*u_[i] +
                                new_anchor.pos_rel[1]*v_[i] +
                                new_anchor.pos_rel[2]*w_[i];
        }

        //std::cout << "Final anchor position (lab frame): ("
        //    << new_anchor.pos[0] << ", "
        //    << new_anchor.pos[1] << ", "
        //    << new_anchor.pos[2] << ")\n";


        parent_->anchors_[idx_].push_back(new_anchor);
    }
}

// Create the number of binding sites on the surface of the sister kcs
void Kinetochore::CreateBindingSites_old() {
//    // Get the allowed radius of af complexes from parent
//    // along with exclusion radius
//    //std::cout << "Creating binding sites\n";
//    double rallowed = parent_->af_location_radius_;
//    double rexclude = parent_->af_exclusion_radius_;
//
//    bool second_sister_ = (idx_ % 2 != 0);
//
//    //std::cout << "Second sister: " << second_sister << std::endl;
//
//    double u_eff[3] = {0.0};
//
//    // Assume that the allowed radius is the radius around the u_eff vector
//    // which is reversed for the second sister kinetochore
//    for (int i = 0; i < ndim_; ++i) {
//        if (second_sister_) {
//            u_eff[i] = -u_[i];
//        } else {
//            u_eff[i] = u_[i]; 
//        }
//    }
//
//    double rkc = 0.5 * parent_->kc_diameter_;
//
//    // Generate random unit vector inside this
//    double alpha;
//    double alpha_max = asin(rallowed / rkc);
//    double pos_vec[3] = {0.0};
//    bool overlap = true;
//
//    // How many to insert?
//    for (int isite = 0; isite < parent_->naf_; ++isite) {
//        if (!parent_->sites_specified_) {
//            int itrial = 0;
//            do {
//                overlap = false;
//                itrial++;
//                // Check against alpha_max
//                do {
//                    generate_random_unit_vector(ndim_, pos_vec, parent_->properties_->rng.r);
//                    alpha = acos(dot_product(ndim_, u_eff, pos_vec));
//                } while (alpha > alpha_max);
//
//                // Now check for overlaps with other sites
//                for (int isite2 = 0; isite2 < parent_->anchors_[idx_].size(); ++isite2) {
//                    auto otheranchor = parent_->anchors_[idx_][isite2]; 
//                    double rmag2 = 0.0;
//                    for (int i = 0; i < ndim_; ++i) {
//                        double drmag = otheranchor.pos[i] - r_[i] - rkc * pos_vec[i];
//                        rmag2 += SQR(drmag);
//                    }
//                    //std::cout << "site2: " << isite2 << ", rmag2: " << rmag2 << std::endl;
//                    overlap = overlap || (rmag2 < SQR(rexclude));
//                }
//
//                //std::cout << "site: " << isite << " trial: " << itrial << std::endl;
//                
//                if (parent_->anchors_[idx_].size() == 0) {
//                    overlap = false;
//                } 
//
//            } while (overlap);
//        } else {
//            // Dirty hack, ugh, set to just along x axis for the binding sites
//            pos_vec[0] = 1.0;
//        }
//
//        //std::cout << "Chose pos_vec: (" << pos_vec[0] << ", "
//        //                                << pos_vec[1] << ", "
//        //                                << pos_vec[2] << "), alpha: "
//        //                                << alpha << std::endl;
//        //std::cout << "u_eff: (" << u_eff[0] << ", "
//        //                        << u_eff[1] << ", "
//        //                        << u_eff[2] << ")\n";
//        // This position is relative inside the frame of the kinetochore, so we
//        // need to make sure to update it appropriately when it moves
//        al_entry new_anchor;
//        new_anchor.label = idx_; // kc label
//        // Generate the lba frame position
//        for (int idim = 0; idim < ndim_; ++idim) {
//            new_anchor.pos[idim] = r_[idim] + rkc * pos_vec[idim];
//        }
//        //std::cout << "pos: ("   << new_anchor.pos[0] << ", "
//        //                        << new_anchor.pos[1] << ", "
//        //                        << new_anchor.pos[2] << ")\n";
//        // Generate the relative position
//        double r_rel[3] = {0.0};
//        for (int idim = 0; idim < ndim_; ++idim) {
//            r_rel[idim] = new_anchor.pos[idim] - r_[idim];
//        }
//        //std::cout << "rel_pos: (" << r_rel[0] << ", "
//        //                          << r_rel[1] << ", "
//        //                          << r_rel[2] << ")\n";
//        double u_proj = dot_product(ndim_, r_rel, u_);
//        double v_proj = dot_product(ndim_, r_rel, v_);
//        double w_proj = dot_product(ndim_, r_rel, w_);
//        new_anchor.pos_rel[0] = u_proj;
//        new_anchor.pos_rel[1] = v_proj;
//        new_anchor.pos_rel[2] = w_proj;
//
//        parent_->anchors_[idx_].push_back(new_anchor);
//    }
//    //std::cout << "Done creating binding sites\n";
}

void Kinetochore::Allocate() {
    // Allocate space
    neighbors_ = new nl_list[nsites_];
    dr_tot_ = (double**) allocate_2d_array(nsites_, ndim_, sizeof(double));
    n_exp_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    attach_ = (int*) allocate_1d_array(nsites_, sizeof(int));
    cross_pos_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    r_cross_ = (double**) allocate_2d_array(nsites_, ndim_, sizeof(double));
    upot_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    feff_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    fangular_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    flink_ = (double**) allocate_2d_array(nsites_, ndim_, sizeof(double));
    lbond_last_ = (double*) allocate_1d_array(nsites_, sizeof(double));

    // Attach to any memory locations we need
    // Attach to the position in the anchor array
    anchors_ = &(parent_->anchors_[idx_]);

    // Set any defaults
    for (int isite = 0; isite < nsites_; ++isite) {
        attach_[isite] = -1;
        lbond_last_[isite] = -1;
    }
}

void Kinetochore::Update_1_2(int ndim,
                             int nperiodic,
                             int nbonds,
                             double **h,
                             double **rbond,
                             double **sbond,
                             double **ubond,
                             double *lbond) {
    // Already have neighbor list
    n_exp_tot_ = 0.0;
    for (int isite = 0; isite < nsites_; ++isite) {
        //std::cout << "  af[" << isite << "]\n";
        n_exp_[isite] = 0.0;
        if (attach_[isite] != -1) {
            continue;
        }

        // Each site has it's own neighbor list and everything!
        for (nl_list::iterator p = neighbors_[isite].begin(); p != neighbors_[isite].end(); ++p) {
            if (parent_->af_tip_distance_ > 0.0) {
                // Tip enhancement works slightly differently
                double dtip = parent_->af_tip_distance_;
                if (!p->duplicate_) {
                    // Side of the microtubule
                    double lside = lbond[p->label] - dtip;
                    double rside[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        rside[i] = rbond[p->label][i] - 0.5 * dtip * ubond[p->label][i]; // set the new r distance
                    }
                    double dr[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        dr[i] = (*anchors_)[isite].pos[i] - rside[i];
                    }
                    double binding_affinity = 0.0;
                    // The side also cares about the SAC being on or off and adjusts its rate accordingly
                    if (parent_->sac_status_ == 1) {
                        binding_affinity = parent_->af_side_eps_eff_ * parent_->af_side_on_rate_;
                    } else {
                        binding_affinity = parent_->af_side_eps_eff_ * parent_->anaphase_rate_;
                    }
                    // Add the aurora B contribution
                    if (parent_->aurora_b_effects_ == 1 && parent_->chromosome_orientation_status_[cidx_] != 4 && parent_->sac_status_ == 1) {
                        binding_affinity = parent_->aurora_b_factor_ * binding_affinity;
                    } else if (parent_->aurora_b_effects_ == 3 && parent_->sac_status_ == 1) {
                        // Inter-kinetochore force-dependent on-rate
                        double fmag_interkc = 0.0;
                        for (int i = 0; i < 3; ++i) {
                            fmag_interkc += SQR(parent_->f_interkc_[idx_][i]);
                        }
                        fmag_interkc = std::sqrt(fmag_interkc);
                        // Determine if the interkinetochore force is towards the direction of ueff of the kinetochore, or not
                        double interkcforce_dot_ueff = dot_product(3, parent_->f_interkc_[idx_], ueff_);

                        double aurora_b_force_factor = 0.0;
                        if (fmag_interkc <= parent_->aurora_b_force_plateau_) {
                            aurora_b_force_factor = 1.0;
                        } else if (interkcforce_dot_ueff > 0.0) {
                            aurora_b_force_factor = 1.0;
                        } else {
                            aurora_b_force_factor = 1.0 / (1.0 + parent_->aurora_b_force_dependence_ * (fmag_interkc - parent_->aurora_b_force_plateau_));
                            //std::cout << "kc[" << idx_ << "] fmag = " << fmag_interkc << ", direction = " << interkcforce_dot_ueff << std::endl;
                            //std::cout << "   aurora b stabilization factor (onrate) = " << aurora_b_force_factor << std::endl;
                        }
                        binding_affinity = binding_affinity * aurora_b_force_factor;
                    }
                    p->value = binding_affinity * Stage_1_2_Probability(ndim_,
                                                                        p->label,
                                                                        false,
                                                                        dr,
                                                                        ubond[p->label],
                                                                        lside);
                    //if (p->value > 0.0) {
                    //    std::cout << "Side contribution[" << p->label << "] = " << p->value << std::endl;
                    //}
                } else {
                    // Tip region
                    double ltip = dtip;
                    double rtip[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        rtip[i] = rbond[p->label][i] + 0.5 * lbond[p->label] * ubond[p->label][i]
                                                     - 0.5 * dtip * ubond[p->label][i]; // set the new r distance
                    }
                    double dr[3] = {0.0};
                    for (int i = 0; i < ndim_; ++i) {
                        dr[i] = (*anchors_)[isite].pos[i] - rtip[i];
                    }
                    double binding_affinity = 0.0;
                    if (parent_->properties_->bonds.poly_state[p->label] == GROW) {
                        // Change based on the SAC status
                        if (parent_->sac_status_ == 1) {
                            binding_affinity = parent_->af_tip_eps_eff_ * parent_->af_tip_on_rate_assemble_;
                        } else {
                            binding_affinity = parent_->af_tip_eps_eff_ * parent_->anaphase_rate_;
                        }
                    } else {
                        // Change the lockdown procedure and amount based on the SAC status
                        if (parent_->sac_status_ == 1) {
                            binding_affinity = parent_->af_tip_eps_eff_ * parent_->af_tip_on_rate_disemble_;
                        } else {
                            binding_affinity = parent_->af_tip_eps_eff_ * parent_->anaphase_rate_;
                        }
                    }
                    // Add the aurora B contribution
                    if (parent_->aurora_b_effects_ == 1 && parent_->chromosome_orientation_status_[cidx_] != 4 && parent_->sac_status_ == 1) {
                        binding_affinity = parent_->aurora_b_factor_ * binding_affinity;
                    } else if (parent_->aurora_b_effects_ == 3 && parent_->sac_status_ == 1) {
                        // Inter-kinetochore force-dependent on-rate
                        double fmag_interkc = 0.0;
                        for (int i = 0; i < 3; ++i) {
                            fmag_interkc += SQR(parent_->f_interkc_[idx_][i]);
                        }
                        fmag_interkc = std::sqrt(fmag_interkc);
                        // Determine if the interkinetochore force is towards the direction of ueff of the kinetochore, or not
                        double interkcforce_dot_ueff = dot_product(3, parent_->f_interkc_[idx_], ueff_);

                        double aurora_b_force_factor = 0.0;
                        if (fmag_interkc <= parent_->aurora_b_force_plateau_) {
                            aurora_b_force_factor = 1.0;
                        } else if (interkcforce_dot_ueff > 0.0) {
                            aurora_b_force_factor = 1.0;
                        } else {
                            aurora_b_force_factor = 1.0 / (1.0 + parent_->aurora_b_force_dependence_ * (fmag_interkc - parent_->aurora_b_force_plateau_));
                            //std::cout << "kc[" << idx_ << "] fmag = " << fmag_interkc << ", direction = " << interkcforce_dot_ueff << std::endl;
                            //std::cout << "   aurora b stabilization factor (onrate) = " << aurora_b_force_factor << std::endl;
                        }
                        binding_affinity = binding_affinity * aurora_b_force_factor;
                    }
                    p->value = binding_affinity * Stage_1_2_Probability(ndim_,
                                                                        p->label,
                                                                        true,
                                                                        dr,
                                                                        ubond[p->label],
                                                                        ltip);
                    //if (p->value > 0.0) {
                    //    std::cout << "Tip contribution[" << p->label << "] = " << p->value << std::endl;
                    //}
                }
            } else {
                double dr[3] = {0.0};
                for (int i = 0; i < ndim_; ++i) {
                    dr[i] = (*anchors_)[isite].pos[i] - rbond[p->label][i];
                }
                double binding_affinity = 0.0;
                binding_affinity = parent_->af_side_eps_eff_ * parent_->af_side_on_rate_;
                // Add the aurora B contribution
                if (parent_->aurora_b_effects_ == 1 && parent_->chromosome_orientation_status_[cidx_] != 4 && parent_->sac_status_ == 1) {
                    binding_affinity = parent_->aurora_b_factor_ * binding_affinity;
                } else if (parent_->aurora_b_effects_ == 3 && parent_->sac_status_ == 1) {
                    // Inter-kinetochore force-dependent on-rate
                    double fmag_interkc = 0.0;
                    for (int i = 0; i < 3; ++i) {
                        fmag_interkc += SQR(parent_->f_interkc_[idx_][i]);
                    }
                    fmag_interkc = std::sqrt(fmag_interkc);
                    // Determine if the interkinetochore force is towards the direction of ueff of the kinetochore, or not
                    double interkcforce_dot_ueff = dot_product(3, parent_->f_interkc_[idx_], ueff_);

                    double aurora_b_force_factor = 0.0;
                    if (fmag_interkc <= parent_->aurora_b_force_plateau_) {
                        aurora_b_force_factor = 1.0;
                    } else if (interkcforce_dot_ueff > 0.0) {
                        aurora_b_force_factor = 1.0;
                    } else {
                        aurora_b_force_factor = 1.0 / (1.0 + parent_->aurora_b_force_dependence_ * (fmag_interkc - parent_->aurora_b_force_plateau_));
                        //std::cout << "kc[" << idx_ << "] fmag = " << fmag_interkc << ", direction = " << interkcforce_dot_ueff << std::endl;
                        //std::cout << "   aurora b stabilization factor (onrate) = " << aurora_b_force_factor << std::endl;
                    }
                    binding_affinity = binding_affinity * aurora_b_force_factor;
                }
                p->value = binding_affinity * Stage_1_2_Probability(ndim_,
                                                                    p->label,
                                                                    false,
                                                                    dr,
                                                                    ubond[p->label],
                                                                    lbond[p->label]);
            }

            // Check to see if the probability of attaching will be greater than 1.0, this will lead to
            // problems down the road
            if (p->value * parent_->delta_kmc_ > 1.0) {
                std::cerr << "ERROR: kMC on-rate probability for chromosomes is > 1.0, p->value = " << p->value << std::endl;
                std::cerr << "   Not exiting, continuing on\n";
                //exit(1);
            }

            n_exp_[isite] += p->value;
            n_exp_tot_ += p->value;
        }
    }
}


double Kinetochore::Stage_1_2_Probability(int n_dim,
                                          int ibond,
                                          bool tip,
                                          double *dr,
                                          double *uline,
                                          double length2) {
    double dr_cross_line[3];
    for (int i = 0; i < n_dim; ++i) {
        dr_cross_line[i] = dr[i];
    }
    double mu0 = -dot_product(n_dim, dr_cross_line, uline);

    double r_min_mag2 = 0.0;
    double rmin[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        double dri = uline[i] * mu0 + dr_cross_line[i];
        rmin[i] = -dri;
        r_min_mag2 += SQR(dri);
    }
    if (r_min_mag2 > parent_->rcutoff2_) {
        return 0.0;
    }

    if (parent_->af_r0_ == 0.0 && parent_->af_xc_assemble_ == 0.0) {
        std::cout << "Don't have zero rest length on chromosomes, exiting!\n";
        exit(1);
    } else {
        // Does this regardless of attachment state
        double lim0 = -mu0 - 0.5 * length2;
        double lim1 = -mu0 + 0.5 * length2;
        // Do something fancy with the cutoffs...
        if (lim0 < -parent_->acutoff_) {
            lim0 = -parent_->acutoff_;
        } else if (lim0 > parent_->acutoff_) {
            lim0 = parent_->acutoff_;
        }
        if (lim1 < -parent_->acutoff_) {
            lim1 = -parent_->acutoff_;
        } else if (lim1 > parent_->acutoff_) {
            lim1 = parent_->acutoff_;
        }
        double r_min_mag = sqrt(r_min_mag2);
        double costhetaminA = dot_product(n_dim, rmin, ueff_) / r_min_mag;
        double costheta2A = dot_product(n_dim, uline, ueff_);
        // Corase grained way of doing this, directly consult the integral
        double x[5] = {lim0, lim1, r_min_mag, costhetaminA, costheta2A};
        double alpha  = parent_->af_k_ / 2.;
        double alpha2 = 0.0;
        if (parent_->prog_lock_ == 0) {
            alpha2 = parent_->af_kr_[0] / 2.;
        } else {
            alpha2 = parent_->af_kr_[parent_->n_bound_[idx_]] / 2.;
        }
        kc_params params;
        params.alpha = alpha;
        params.alpha2 = alpha2;
        params.r0 = parent_->af_r0_;
        if (tip) {
            if (parent_->properties_->bonds.poly_state[ibond] == GROW) {
                params.xc = parent_->af_xc_assemble_;
            } else {
                params.xc = parent_->af_xc_disemble_;
            }
        } else {
            params.xc = parent_->af_xc_side_;
        }
        params.chic = parent_->af_chic_;
        int pstatus;
        //double nexp = prob_1_2_advanced_full(x, &params);
        int ithread = 0;
        #ifdef ENABLE_OPENMP
        ithread = omp_get_thread_num();
        #endif
        double nexp = prob_1_2_func(x, &params, &pstatus, parent_->gsl_workspaces_[ithread]);

        if (pstatus) {
            // Something has gone wrong, terminate the simulation
            std::cerr << "Failed in Kinetochore::Stage_1_2_Probability KC[" << idx_ << "]\n";
            parent_->PrintFrame();
            exit(1);
        }

        if (nexp > 1E-5) {
            return nexp;
        }
        else
            return 0.0;
    }
    return 0.0;
}

double Kinetochore::Stage_1_2_Probability(double rminmag2,
                                          double lline) {
    // no polar affinity, also, have exact position of first 'head'
    // Limits are just the limits of integration for the rod, since
    // the closest approach has no mu value, and it will always be
    // perpendicular
    if (rminmag2 > parent_->rcutoff2_)
        return 0.0;

    double kb = parent_->af_k_;

    if (parent_->af_r0_ == 0.0 && parent_->af_xc_assemble_ == 0.0) {
        std::cout << "KC Shouldn't be here...\n";
        double scale_factor = sqrt(0.5 * kb);

        double lim0 = scale_factor * (-0.5 * lline);
        double term0 = erf(lim0);
        double lim1 = scale_factor * (0.5 * lline);
        double term1 = erf(lim1);

        double result = sqrt(M_PI_2 / kb) * exp(-0.5*kb*rminmag2) * (term1 - term0);
        return result;
    } else {
        double lim0 = -0.5 * lline;
        double lim1 = 0.5 * lline;
        double rminmag = sqrt(rminmag2);
        double x[2] = {fabs(lim0), rminmag};
        double term0 = parent_->n_exp_lookup_->Lookup(x) * ((lim0 < 0) ? -1.0 : 1.0);
        x[0] = fabs(lim1);
        double term1 = parent_->n_exp_lookup_->Lookup(x) * ((lim1 < 0) ? -1.0 : 1.0);
        double nexp = (term1 - term0);
        if (nexp > 1E-3)
            return nexp;
        else
            return 0.0;
    }
    return 0.0;
}

void Kinetochore::UpdateNeighbors(int ndim,
                                  int nperiodic,
                                  int nbonds,
                                  double **h,
                                  double **rbond,
                                  double **sbond,
                                  double **ubond,
                                  double *lbond,
                                  double rcutoff2) {

    //if (parent_->properties_->i_current_step >= 3136845) {
    //    std::cout << "KC [" << idx_ << "] UpdateNeighbors\n";
    //}
    // Loop over sites
    for (int isite = 0; isite < nsites_; ++isite) {
        dr_tot_[isite][0] = dr_tot_[isite][1] = dr_tot_[isite][2] = 0.0;
        neighbors_[isite].clear();
        if (attach_[isite] != -1)
            continue;

        // Loop over bonds
        for (int ibond = 0; ibond < nbonds_; ++ibond) {
            nl_entry neighb;
            double *s_ = NULL;
            double mu, r_min[3];
            min_distance_sphere_sphero(ndim,
                                       nperiodic,
                                       h,
                                       (*anchors_)[isite].pos,
                                       s_,
                                       rbond[ibond],
                                       sbond[ibond],
                                       ubond[ibond],
                                       lbond[ibond],
                                       r_min,
                                       &neighb.r_min_mag2,
                                       &mu);
            if (neighb.r_min_mag2 < rcutoff2) {
                nl_entry new_neighbor;
                new_neighbor.label = ibond;
                new_neighbor.value = 0.0;
                new_neighbor.duplicate_ = false;

                neighbors_[isite].push_back(new_neighbor);

                // If we have tip enhancement, add an extra fake neighbor onto the end
                if (parent_->af_tip_distance_ > 0.0) {
                    nl_entry new_neighbor_tip;
                    new_neighbor_tip.label = ibond;
                    new_neighbor_tip.value = 0.0;
                    new_neighbor_tip.duplicate_ = true;

                    neighbors_[isite].push_back(new_neighbor_tip);
                }
            }
        }
    }

    //// DEBUG, print the neighbor list
    //for (int isite = 0; isite < nsites_; ++isite) {
    //    for (auto p = neighbors_[isite].begin(); p != neighbors_[isite].end(); ++p) {
    //        std::cout << " neighbor: " << p->label << std::endl;
    //    }
    //}
}

void Kinetochore::ClearNeighbors(int isite) {
    n_exp_tot_ -= n_exp_[isite];
    n_exp_[isite] = 0.0;
    neighbors_[isite].clear();
}

bool Kinetochore::Insert_1_2(system_parameters *parameters,
                             system_properties *properties) {

    double ran_loc = gsl_rng_uniform(properties->rng.r) * n_exp_tot_;

    // Loop over sites to pick which one
    double loc = 0.0;
    for (int isite = 0; isite < nsites_; ++isite) {
        loc += n_exp_[isite]; 

        // If we find a site, reset the counter to the beginning of the list
        if (loc > ran_loc) {
            loc -= n_exp_[isite];

            // Look at my neighbors for which one to fall onto
            for (nl_list::iterator nl = neighbors_[isite].begin(); nl != neighbors_[isite].end(); ++nl) {
                loc += nl->value;

                if (loc > ran_loc) {
                    // We found which one to fall onto, now figure out where

                    bool didconvert = Convert_1_2(parameters,
                                                  properties,
                                                  nl->label,
                                                  isite,
                                                  nl->duplicate_);

                    if (!didconvert) {
                        std::cout << "Obviously failed, what else can we look at?\n";
                        std::cout << "   loc: " << loc << ", ran_loc: " << ran_loc << ", nexp[" << isite << "]: "
                            << n_exp_[isite] << ", nexptot: " << n_exp_tot_ << std::endl;
                    }
                    return didconvert;
                }
            }
        }
    }

    // If we make it here, something has gone wrong with attaching a trial
    //std::cout << "Something has gone horribly wrong KC Insert_1_2\n";
    //std::cout << "  Step[" << properties->i_current_step << "], kc[" << idx_ << "]\n";
    //exit(1);
    return false;
}

bool Kinetochore::Convert_1_2(system_parameters *parameters,
                              system_properties *properties,
                              int ibond,
                              int isite,
                              bool tip) {
    bool success = false;
    //std::cout << "Convert_1_2: bond:" << ibond << ", site: " << isite << std::endl;

    ClearNeighbors(isite); 
    attach_[isite] = ibond;

    double **rbond = properties->bonds.r_bond;
    double **ubond = properties->bonds.u_bond;
    double *lbond  = properties->bonds.length;

    //std::cout << "nexp now: " << n_exp_tot_ << std::endl;

    // Pick a random position for this site on the bond
    double s_1[3] = {0.0};
    if (parent_->af_tip_distance_ > 0.0) {
        double dtip = parent_->af_tip_distance_;
        if (!tip) {
            // Side location
            double lside = lbond[ibond] - dtip;
            double rside[3] = {0.0};
            for (int i = 0; i < ndim_; ++i) {
                rside[i] = rbond[ibond][i] - 0.5 * dtip * ubond[ibond][i]; // set the new r distance
            }
            cross_pos_[isite] = RandomPositionOld(parameters->n_dim,
                                               parameters->n_periodic,
                                               ibond,
                                               false,
                                               properties->unit_cell.h,
                                               (*anchors_)[isite].pos,
                                               s_1,
                                               rside,
                                               properties->bonds.s_bond[ibond],
                                               properties->bonds.u_bond[ibond],
                                               lside,
                                               parent_->af_k_,
                                               properties->rng.r);
            //std::cout << "Convert_1_2: bond: " << ibond << ", site: " << isite << " side attach pos: " << cross_pos_[isite] << std::endl;
            //std::cout << "KC[" << idx_ << "]{" << isite << "} MT[" << ibond << "] attach SIDE position: " << cross_pos_[isite] << std::endl;
        } else {
            double ltip = dtip;
            double rtip[3] = {0.0};
            for (int i = 0; i < ndim_; ++i) {
                rtip[i] = rbond[ibond][i] + 0.5 * lbond[ibond] * ubond[ibond][i]
                                          - 0.5 * dtip * ubond[ibond][i]; // set the new r distance
            }
            cross_pos_[isite] = RandomPositionOld(parameters->n_dim,
                                               parameters->n_periodic,
                                               ibond,
                                               true,
                                               properties->unit_cell.h,
                                               (*anchors_)[isite].pos,
                                               s_1,
                                               rtip,
                                               properties->bonds.s_bond[ibond],
                                               properties->bonds.u_bond[ibond],
                                               ltip,
                                               parent_->af_k_,
                                               properties->rng.r);
            // Modify this to actually be the location
            cross_pos_[isite] += lbond[ibond] - dtip;

            if (cross_pos_[isite] > lbond[ibond] || cross_pos_[isite] < 0.0) {
                std::cout << "Tip is somehow off the bond, exiting!\n";
                exit(1);
            }
            //std::cout << "KC[" << idx_ << "]{" << isite << "} MT[" << ibond << "] attach  TIP position: " << cross_pos_[isite] << std::endl;
            //std::cout << "Convert_1_2: bond: " << ibond << ", site: " << isite << " tip attach pos: " << cross_pos_[isite] << std::endl;
        }
    } else {
        // No tip enhancement, so just find the position
        cross_pos_[isite] = RandomPositionOld(parameters->n_dim,
                                           parameters->n_periodic,
                                           ibond,
                                           false,
                                           properties->unit_cell.h,
                                           (*anchors_)[isite].pos,
                                           s_1,
                                           properties->bonds.r_bond[ibond],
                                           properties->bonds.s_bond[ibond],
                                           properties->bonds.u_bond[ibond],
                                           properties->bonds.length[ibond],
                                           parent_->af_k_,
                                           properties->rng.r);
        //std::cout << "KC[" << idx_ << "]{" << isite << "} MT[" << ibond << "] attach      position: " << cross_pos_[isite] << std::endl;
    }
    if (cross_pos_[isite] != cross_pos_[isite]) {
        std::cout << "NaN encountered in KC Convert_1_2\n";
        std::cout << "  bond: " << ibond << ", site: " << isite << std::endl;
        // Instead of exiting, just don't attach and keep going (slightly unexpected behavior)
        attach_[isite] = -1;
        //exit(1);
        success = false;
    } else {
        //std::cout << "we good!\n";
        // Calculate the force/energy for later use
        double f[3] = {0.0};
        double t[3] = {0.0};
        if (force_type_ == 0) {
            CalcForceHarmonic(parameters->n_dim,
                              parameters->n_periodic,
                              properties->unit_cell.h,
                              isite,
                              properties->bonds.r_bond[ibond],
                              properties->bonds.s_bond[ibond],
                              properties->bonds.u_bond[ibond],
                              properties->bonds.length[ibond],
                              f,
                              t);
            success = true;
        } else if (force_type_ == 1) {
            CalcForceQuartic(parameters->n_dim,
                             parameters->n_periodic,
                             properties->unit_cell.h,
                             isite,
                             properties->bonds.r_bond[ibond],
                             properties->bonds.s_bond[ibond],
                             properties->bonds.u_bond[ibond],
                             properties->bonds.length[ibond],
                             f,
                             t);
            success = true;
        }
    }
    return success;
}

bool Kinetochore::ForceAttach(system_parameters *parameters,
                              system_properties *properties,
                              std::vector<int>& bondlist) {
    bool success = false;
    
    for (int isite = 0; isite < nsites_; ++isite) {
        ClearNeighbors(isite);
        int ibond = bondlist[isite];
        attach_[isite] = ibond; // please have at least this many bonds
        // Just the tip, just for a second.  It'll feel good.
        cross_pos_[isite] = properties->bonds.length[ibond];
        double f[3] = {0.0};
        double t[3] = {0.0};
        CalcForceHarmonic(parameters->n_dim,
                          parameters->n_periodic,
                          properties->unit_cell.h,
                          isite,
                          properties->bonds.r_bond[ibond],
                          properties->bonds.s_bond[ibond],
                          properties->bonds.u_bond[ibond],
                          properties->bonds.length[ibond],
                          f,
                          t);
        success = true;
    }

    return success;
}

bool Kinetochore::ForceAttachSingle(system_parameters *parameters,
                                    system_properties *properties,
                                    int ibond,
                                    int isite) {
    bool success = false;

    ClearNeighbors(isite);
    attach_[isite] = ibond;
    cross_pos_[isite] = properties->bonds.length[ibond];
    double f[3] = {0.0};
    double t[3] = {0.0};
    CalcForceHarmonic(parameters->n_dim,
                      parameters->n_periodic,
                      properties->unit_cell.h,
                      isite,
                      properties->bonds.r_bond[ibond],
                      properties->bonds.s_bond[ibond],
                      properties->bonds.u_bond[ibond],
                      properties->bonds.length[ibond],
                      f,
                      t);
    success = true;

    return success;
}

void Kinetochore::ForceUnbind(system_parameters *parameters,
                              system_properties *properties) {
    for (int isite = 0; isite < nsites_; ++isite) {
        ClearNeighbors(isite);
        attach_[isite] = -1;
    }
    double r_cutoff2 = SQR(parent_->rcutoff_ + SKIN_);
    UpdateNeighbors(parameters->n_dim,
                    parameters->n_periodic,
                    properties->bonds.n_bonds,
                    properties->unit_cell.h,
                    properties->bonds.r_bond,
                    properties->bonds.s_bond,
                    properties->bonds.u_bond,
                    properties->bonds.length,
                    r_cutoff2);
}

bool Kinetochore::Remove_2_1(system_parameters *parameters,
                             system_properties *properties) {
    // Remove one of the binding sites
    int curr_site = -1;
    int n_bound = parent_->n_bound_[idx_];
    int off_site = (int) (gsl_rng_uniform(properties->rng.r) * n_bound);

    for (int isite = 0; isite < nsites_; ++isite) {
        if (attach_[isite] != -1)
            curr_site++;
        if (curr_site == off_site) {
            //std::cout << "KC[ " << idx_ << "]{" << isite << "} removing\n";
            //std::cout << "  attach: " << attach_[isite] << std::endl;

            parent_->n_bound_[idx_]--;
            attach_[isite] = -1;
            cross_pos_[isite] = 0.0;
            
            return true;
        }
    }

    return false;
}

int Kinetochore::Remove_2_1_Fdep(system_parameters *parameters,
                                 system_properties *properties) {
    // Force dependent unbinding, including the tip enhancement changes
    double rcutoff_2 = SQR(SKIN_ + parent_->rcutoff_);
    int noff_tot = 0;

    for (int isite = 0; isite < nsites_; ++isite) {
        if (attach_[isite] != -1) {
            //double fmag = sqrt(dot_product(ndim_, flink_[isite], flink_[isite]));
            double fmag = feff_[isite]; // actually an adjusted force...
            double fangularmag = fangular_[isite];
            double kboltz = 0.0;

            // Determine is we're at the tip or not
            int ibond = attach_[isite];
            double dstab = properties->bonds.length[ibond] - cross_pos_[isite];
            double poff_base = 0.0;
            
            double kon = 0.0;
            double xc = 0.0;

            // Check for all the possible combinations.
            bool is_tip = dstab < parent_->af_tip_distance_;
            if (dstab < parent_->af_tip_distance_) {
                // Tip stuff
                // Still make it force dependent
                if (parent_->properties_->bonds.poly_state[ibond] == GROW) {
                    if ( parent_->sac_status_ == 1) {
                        kon = parent_->af_tip_on_rate_assemble_;
                    } else {
                        kon = parent_->anaphase_rate_;
                    }
                    xc = parent_->af_xc_assemble_;
                } else {
                    // Change the lifetime based on SAC status
                    if (parent_->sac_status_ == 1) {
                        kon = parent_->af_tip_on_rate_disemble_;
                    } else {
                        kon = parent_->anaphase_rate_;
                    }
                    xc = parent_->af_xc_disemble_;
                }
            } else {
                if (parent_->sac_status_ == 1) {
                    kon = parent_->af_side_on_rate_;
                } else {
                    kon = parent_->anaphase_rate_;
                }
                xc = parent_->af_xc_side_;
            }

            // Add the aurora B contribution
            if (parent_->aurora_b_effects_ == 1 && parent_->chromosome_orientation_status_[cidx_] != 4 && parent_->sac_status_ == 1) {
                kon = parent_->aurora_b_factor_ * kon;
            } else if (((parent_->aurora_b_effects_ == 2) || (parent_->aurora_b_effects_ == 3) || (parent_->aurora_b_effects_ == 4 && is_tip)) && parent_->sac_status_ == 1) {
                // The force-dependent way of doing aurora b effects
                // Get the magnitude of the inter-kinetochore force, bear in mind, this is from the LAST round of computations
                double fmag_interkc = 0.0;
                for (int i = 0; i < 3; ++i) {
                    fmag_interkc += SQR(parent_->f_interkc_[idx_][i]);
                }
                fmag_interkc = std::sqrt(fmag_interkc);
                // Determine if the interkinetochore force is towards the direction of ueff of the kinetochore, or not
                double interkcforce_dot_ueff = dot_product(3, parent_->f_interkc_[idx_], ueff_);

                // Compute the force-factor dependence
                double aurora_b_force_factor = 0.0;
                if (fmag_interkc <= parent_->aurora_b_force_plateau_) {
                    aurora_b_force_factor = 1.0; 
                } else if (interkcforce_dot_ueff > 0.0) {
                    aurora_b_force_factor = 1.0;
                } else {
                    aurora_b_force_factor = 1.0 / (1.0 + parent_->aurora_b_force_dependence_ * (fmag_interkc - parent_->aurora_b_force_plateau_));
                    //std::cout << "kc[" << idx_ << "] fmag = " << fmag_interkc << ", direction = " << interkcforce_dot_ueff << std::endl;
                    //std::cout << "   aurora b stabilization factor (offrate) = " << aurora_b_force_factor << std::endl;
                    //exit(1);
                }

                kon = kon * aurora_b_force_factor;
            }

            //std::cout << "Chromosome status: " << parent_->chromosome_orientation_status_[cidx_] << std::endl;
            //std::cout << "tip: " << (dstab < parent_->af_tip_distance_) << std::endl;
            //std::cout << "kon: " << kon << std::endl;
            //std::cout << "xc: " << xc << std::endl;
            
            poff_base = kon * parent_->delta_kmc_;
            kboltz = exp(xc * fmag + parent_->af_chic_ * fangularmag);

            // Check to make sure we aren't causing any problems with probabilities > 1
            if ((poff_base * kboltz > 1.0) && (parent_->use_poisson_process_ == 0)) {
                std::cerr << "ERROR: Major error in chromosome detachment code: poff > 1.0, poff_base = " << poff_base << ", kboltz = " << kboltz << ", combined = " << poff_base * kboltz << std::endl;
                std::cerr << "   Not exiting, continuing on\n";
                //exit(1);
            }

            if (parent_->use_poisson_process_ == 1) {
                // Use a poisson process for the off-rate, P = 1 - exp(-P)
                uint8_t off = gsl_rng_uniform(properties->rng.r) < (1.0 - exp(-poff_base * kboltz));
                //std::cout << "Using new form of off-rate " << (int)off << std::endl;
                if (off) {
                    // Unbind me
                    parent_->n_bound_[idx_]--;
                    attach_[isite] = -1;
                    cross_pos_[isite] = 0.0;
                    noff_tot++;
                }
            } else {
                uint8_t off = gsl_rng_uniform(properties->rng.r) < poff_base * kboltz;
                //std::cout << "Using old form of off-rate " << (int)off << std::endl;
                if (off) {
                    // Unbind me
                    parent_->n_bound_[idx_]--;
                    attach_[isite] = -1;
                    cross_pos_[isite] = 0.0;
                    noff_tot++;
                }
            }
            //if (off) {
            //    // Unbind me
            //    parent_->n_bound_[idx_]--;
            //    attach_[isite] = -1;
            //    cross_pos_[isite] = 0.0;
            //    noff_tot++;
            //}
        }
    }

    return noff_tot;
}

// Build the RandomPosition based on falling on the bond in the correct locaiton
// according to the CDF of the probability distributions
double Kinetochore::RandomPosition(int ndim,
                                   int nperiodic,
                                   int ibond,
                                   bool tip,
                                   double **h, 
                                   double *r1,
                                   double *s1,
                                   double *rline,
                                   double *sline,
                                   double *uline,
                                   double lline,
                                   double kb,
                                   gsl_rng *r) {
    // Calculate the minimum distance to the carrier line
    // for the cross position information
    double mu;
    double dr[3] = {0.0};
    min_distance_point_carrier_line(ndim,
                                    nperiodic,
                                    h,
                                    r1,
                                    s1,
                                    rline,
                                    sline,
                                    uline,
                                    lline,
                                    dr,
                                    &mu);

    double y02 = 0.0;
    double rmin[3] = {0.0};
    for (int i = 0; i < ndim; ++i) {
        double dri = uline[i] * mu + dr[i];
        rmin[i] = dri;
        y02 += SQR(dri);
    }

    // Diverge from previous method, generate a CDF of the NORMALIZED probability
    // distribution...
    double rminmag = sqrt(y02);
    double costhetaminA = dot_product(ndim, rmin, ueff_) / rminmag;
    double costheta2A = dot_product(ndim, uline, ueff_);
    double lim0 = -mu - 0.5 * lline;
    double lim1 = -mu + 0.5 * lline;
    double xvec[5] = {lim0, lim1, rminmag, costhetaminA, costheta2A};
    kc_params params;
    params.alpha = parent_->af_k_/2.;
    if (parent_->prog_lock_ == 0) {
        params.alpha2 = parent_->af_kr_[0] / 2.;
    } else {
        params.alpha2 = parent_->af_kr_[parent_->n_bound_[idx_]] / 2.;
    }
    params.r0 = parent_->af_r0_;
    if (tip) {
        if (parent_->properties_->bonds.poly_state[ibond] == GROW) {
            params.xc = parent_->af_xc_assemble_;
        } else {
            params.xc = parent_->af_xc_disemble_;
        }
    } else {
        params.xc = parent_->af_xc_side_;
    }
    params.chic = parent_->af_chic_;
    int ithread = 0;
    #ifdef ENABLE_OPENMP
    ithread = omp_get_thread_num();
    #endif
    double u = gsl_rng_uniform(r);
    double pos = InverseTransformSample(u, xvec, lline, prob_1_2_func, parent_->gsl_workspaces_[ithread], 1e-3, &params) + mu + 0.5*lline;

    //std::cout << "u = " << u << std::endl;
    //std::cout << "pos = " << pos << std::endl;
    if (pos >= 0 && pos <= lline) {
        return pos;
    } else {
        std::stringstream ess;
        ess << "Kinetochore::Convert_1_2: failed RandomPosition\n";
        dump_simulation(parent_->parameters_, parent_->properties_, ess.str());
        return nan("");
    }
}

double Kinetochore::RandomPositionOld(int ndim,
                                   int nperiodic,
                                   int ibond,
                                   bool tip,
                                   double **h,
                                   double *r1,
                                   double *s1,
                                   double *rline,
                                   double *sline,
                                   double *uline,
                                   double lline,
                                   double kb,
                                   gsl_rng *r) {
    //std::cout << "WARNING, random position not doing anything right now\n";
    double mu;
    double dr[3] = {0.0};
    min_distance_point_carrier_line(ndim,
                                    nperiodic,
                                    h,
                                    r1,
                                    s1,
                                    rline,
                                    sline,
                                    uline,
                                    lline,
                                    dr,
                                    &mu);

    //std::cout << "Random position mu: " << mu << std::endl;

    // Change form is we have rest length or not
    if (parent_->af_r0_ == 0.0) {
        do {
            double pos = gsl_ran_gaussian_ziggurat(r, sqrt(1.0/kb)) + mu + 0.5 * lline;
            if (pos >= 0 && pos < lline)
                return pos;
        } while (1);
    } else {
        double y02 = 0.0;
        double rmin[3] = {0.0};
        for (int i = 0; i < ndim; ++i) {
            double dri = uline[i] * mu + dr[i];
            rmin[i] = dri;
            y02 += SQR(dri);
        }
        int itrial = 0;
        //if (parent_->properties_->i_current_step >= 134143847) {
        //    std::cout << "Fine DEBUG, step: " << parent_->properties_->i_current_step << std::endl;
        //}
        do {
            itrial++;
            double u = gsl_rng_uniform(r);
            double rminmag = sqrt(y02);
            double costhetaminA = dot_product(ndim, rmin, ueff_) / rminmag;
            double costheta2A = dot_product(ndim, uline, ueff_);
            double lim0 = -mu - 0.5 * lline;
            double lim1 = -mu + 0.5 * lline;
            double xvec[5] = {lim0, lim1, rminmag, costhetaminA, costheta2A};
            kc_params params;
            params.alpha = parent_->af_k_/2.;
            if (parent_->prog_lock_ == 0) {
                params.alpha2 = parent_->af_kr_[0] / 2.;
            } else {
                params.alpha2 = parent_->af_kr_[parent_->n_bound_[idx_]] / 2.;
            }
            params.r0 = parent_->af_r0_;
            if (tip) {
                if (parent_->properties_->bonds.poly_state[ibond] == GROW) {
                    params.xc = parent_->af_xc_assemble_;
                } else {
                    params.xc = parent_->af_xc_disemble_;
                }
            } else {
                params.xc = parent_->af_xc_side_;
            }
            params.chic = parent_->af_chic_;
            int ithread = 0;
            #ifdef ENABLE_OPENMP
            ithread = omp_get_thread_num();
            #endif
            double pos = InvertBinary(u, xvec, prob_1_2_func, parent_->gsl_workspaces_[ithread], 1e-3, &params) + mu + 0.5 * lline;

            // Old improper way of doing things, try the new way
            //double xvec[4] = {0.0, sqrt(y02), costhetaminA, costheta2A};
            //double rand2 = gsl_rng_uniform(r);
            //// Combined for tip tracking algorithm, bias towards tip by some amount
            //double pos = ((rand2 < 0.5) ? -1.0 : 1.0) *
            //    parent_->n_exp_lookup_->Invert(0, u, xvec) + mu + 0.5 * lline;
            ////std::cout << "KC in x[" << xvec[0] << ", " << xvec[1] << "]\n";
            ////std::cout << "  u: " << u << std::endl;
            ////std::cout << "  res: " << parent_->n_exp_lookup_->Invert(0, u, xvec) << std::endl;
            if (pos >= 0 && pos <= lline)
                return pos;
        } while (itrial < 10000); // guard against pathological behavior
        std::stringstream ess;
        //ess << "Kinetochore::Convert_1_2: Too many trials, returning NaN\n";
        //ess << "   kc: " << idx_ << std::endl;
        //ess << "   min dr: (" << dr[0] << ", " << dr[1] << ", " << dr[2] << "), mu: " << mu << std::endl;
        //ess << "   y02: " << y02 << ", " << ", length: " << lline << std::endl;
        //dump_simulation(parent_->parameters_, parent_->properties_, ess.str());
        //return nan("");
        ess << "Kinetochore::Convert_1_2: Too many trials (over 10k), returning cross position 0 for consistency\n";
        std::cout << ess.str();
        return 0.0;
    }

    std::stringstream ess;
    ess << "Kinetochore::Convert_1_2: failed RandomPosition\n";
    dump_simulation(parent_->parameters_, parent_->properties_, ess.str());
    return nan("");
}

double Kinetochore::CalcForceHarmonic(int ndim,
                                      int nperiodic,
                                      double **h,
                                      int isite,
                                      double *rbond,
                                      double *sbond,
                                      double *ubond,
                                      double lbond,
                                      double *f,
                                      double *t) {
    UpdateSiteVector(ndim,
                     nperiodic,
                     h,
                     isite,
                     rbond,
                     sbond,
                     ubond,
                     lbond);
    upot_[isite] = 0.0;

    //std::cout << "rcross(" << r_cross_[isite][0] << ", "
    //                       << r_cross_[isite][1] << ", "
    //                       << r_cross_[isite][2] << ")\n";

    double k = parent_->af_k_;
    double linearfactor = k;
    double kr = 0.0;
    if (parent_->prog_lock_ == 0) {
        kr = parent_->af_kr_[0];
    } else {
        kr = parent_->af_kr_[parent_->n_bound_[idx_]];
    }
    double rmag = sqrt(dot_product(ndim, r_cross_[isite], r_cross_[isite]));
    //std::cout << "rmag: " << rmag << std::endl;
    double rhat[3] = {0.0};
    for (int i = 0; i < ndim; ++i) {
        rhat[i] = r_cross_[isite][i] / rmag;
    }
    // Linear factor
    if (parent_->af_r0_ == 0.0) {
        linearfactor = k;
    } else {
        linearfactor = k * (rmag - parent_->af_r0_);
    }
    //std::cout << "linearfactor: " << linearfactor << std::endl;
    //std::cout << "rhat(" << rhat[0] << ", "
    //                     << rhat[1] << ", "
    //                     << rhat[2] << ")\n";
    // Update the effective U, otherwise this is all for naught
    for (int i = 0; i < ndim; ++i) {
        if (*second_sister_) {
            ueff_[i] = -u_[i];
        } else {
            ueff_[i] = u_[i]; 
        }
    }
    //std::cout << "ueff(" << ueff_[0] << ", "
    //                     << ueff_[1] << ", "
    //                     << ueff_[2] << ")\n";
    // Torque factor
    double uA_dot_rhat = dot_product(ndim, ueff_, rhat);
    //std::cout << "uA_dot_rhat: " << uA_dot_rhat << std::endl;
    double rhat_cross_uA[3] = {0.0};
    double rhat_cross_rhat_cross_uA[3] = {0.0};
    cross_product(rhat, ueff_, rhat_cross_uA, ndim);
    cross_product(rhat, rhat_cross_uA, rhat_cross_rhat_cross_uA, ndim);
    double tfactor = 0.0;
    if (rmag > 0.0) {
        tfactor = kr / rmag * (uA_dot_rhat - 1.0);
    }
    //std::cout << "tfactor: " << tfactor << std::endl;

    for (int i = 0; i < ndim; ++i) {
        f[i] = linearfactor * rhat[i] - tfactor * rhat_cross_rhat_cross_uA[i];
        t[i] = rmag * tfactor * rhat_cross_uA[i];
        flink_[isite][i] = f[i];
    }
    feff_[isite] = linearfactor;
    fangular_[isite] = -tfactor*rmag;
    upot_[isite] = 0.5 * linearfactor * linearfactor / k;
    if (kr > 0.0) {
        upot_[isite] += 0.5 * tfactor * tfactor * rmag * rmag / kr;
    }

    return upot_[isite];
}

// Calculate the force due to a quartic potential
double Kinetochore::CalcForceQuartic(int ndim,
                                     int nperiodic,
                                     double **h,
                                     int isite,
                                     double *rbond,
                                     double *sbond,
                                     double *ubond,
                                     double lbond,
                                     double *f,
                                     double *t) {
    UpdateSiteVector(ndim,
                     nperiodic,
                     h,
                     isite,
                     rbond,
                     sbond,
                     ubond,
                     lbond);
    upot_[isite] = 0.0;

    //std::cout << "rcross(" << r_cross_[isite][0] << ", "
    //                       << r_cross_[isite][1] << ", "
    //                       << r_cross_[isite][2] << ")\n";

    double k = parent_->af_k_;
    double linearfactor = k;
    double kr = 0.0;
    if (parent_->prog_lock_ == 0) {
        kr = parent_->af_kr_[0];
    } else {
        kr = parent_->af_kr_[parent_->n_bound_[idx_]];
    }
    double rmag = sqrt(dot_product(ndim, r_cross_[isite], r_cross_[isite]));
    //std::cout << "rmag: " << rmag << std::endl;
    double rhat[3] = {0.0};
    for (int i = 0; i < ndim; ++i) {
        rhat[i] = r_cross_[isite][i] / rmag;
    }
    // Linear factor
    if (parent_->af_r0_ == 0.0) {
        linearfactor = k;
    } else {
        linearfactor = k * (rmag - parent_->af_r0_);
    }
    //std::cout << "linearfactor: " << linearfactor << std::endl;
    //std::cout << "rhat(" << rhat[0] << ", "
    //                     << rhat[1] << ", "
    //                     << rhat[2] << ")\n";
    // Update the effective U, otherwise this is all for naught
    for (int i = 0; i < ndim; ++i) {
        if (*second_sister_) {
            ueff_[i] = -u_[i];
        } else {
            ueff_[i] = u_[i]; 
        }
    }
    //std::cout << "ueff(" << ueff_[0] << ", "
    //                     << ueff_[1] << ", "
    //                     << ueff_[2] << ")\n";
    // Torque factor
    double uA_dot_rhat = dot_product(ndim, ueff_, rhat);
    //std::cout << "uA_dot_rhat: " << uA_dot_rhat << std::endl;
    double rhat_cross_uA[3] = {0.0};
    double rhat_cross_rhat_cross_uA[3] = {0.0};
    cross_product(rhat, ueff_, rhat_cross_uA, ndim);
    cross_product(rhat, rhat_cross_uA, rhat_cross_rhat_cross_uA, ndim);
    double tfactor = 0.0;
    if (rmag > 0.0) {
        tfactor = 2.0 * kr / rmag * pow((uA_dot_rhat - 1.0), 3.0);
    }
    //std::cout << "tfactor: " << tfactor << std::endl;

    for (int i = 0; i < ndim; ++i) {
        f[i] = linearfactor * rhat[i] - tfactor * rhat_cross_rhat_cross_uA[i];
        t[i] = rmag * tfactor * rhat_cross_uA[i];
        flink_[isite][i] = f[i];
    }
    feff_[isite] = linearfactor;
    fangular_[isite] = -tfactor*rmag;
    upot_[isite] = 0.5 * linearfactor * linearfactor / k;
    if (kr > 0.0) {
        upot_[isite] += 0.5 * tfactor * tfactor * rmag * rmag / kr;
    }

    return upot_[isite];
}

void Kinetochore::UpdateSiteVector(int ndim,
                                   int nperiodic,
                                   double **h,
                                   int isite,
                                   double *rbond,
                                   double *sbond,
                                   double *ubond,
                                   double lbond) {
    if (nperiodic != 0) {
        std::cout << "Periodic space for kinetochores not allowed, exiting\n";
        exit(1);
    }

    double mu = cross_pos_[isite] - 0.5 * lbond;
    for (int i = 0; i < ndim; ++i) {
        r_cross_[isite][i] = rbond[i] + mu * ubond[i] - (*anchors_)[isite].pos[i];
    }

    //std::cout << "Debug update site vector\n";
    //std::cout << "rcross: (" << r_cross_[isite][0] << ", " << r_cross_[isite][1] << ", " << r_cross_[isite][2] << ")\n";
}

void Kinetochore::Step(int ndim,
                       int nperiodic,
                       int nbonds,
                       double **h,
                       double **rbond,
                       double **sbond,
                       double **ubond,
                       double *lbond,
                       double delta,
                       gsl_rng *r) {
    // Check if we need to update the sites
    bool update_sites = false;
    for (int isite = 0; isite < nsites_; ++isite) {
        if (dot_product(ndim, dr_tot_[isite], dr_tot_[isite]) > 0.25 * SQR(SKIN_)) {
            update_sites = true;
        }
    }

    if (update_sites) {
        //std::cout << "Tripped update on sites!\n";
        double rcutoff_2 = SQR(parent_->rcutoff_ + SKIN_);
        UpdateNeighbors(ndim,
                        nperiodic,
                        nbonds,
                        h,
                        rbond,
                        sbond,
                        ubond,
                        lbond,
                        rcutoff_2);
    }

    // Now we walk and diffuse
    // Here is the walking
    if (parent_->af_velocity_ != 0.0) {
        for (int isite = 0; isite < nsites_; ++isite) {
            if (attach_[isite] == -1)
                continue;
            int ibond = attach_[isite];
            double ui_dot_f = dot_product(ndim, ubond[ibond], flink_[isite]);
            double f_mag_j = -ui_dot_f; // following convention from croslsinks
            double velocity = parent_->af_velocity_;
            if (f_mag_j*velocity > 0.0)
                f_mag_j = 0.0;
            else {
                // Assume parallel stall type
                f_mag_j = ABS(f_mag_j);
            }

            if (f_mag_j < parent_->af_stall_force_)
                velocity *= 1.0 - f_mag_j/parent_->af_stall_force_;
            else
                velocity = 0.0;

            //std::cout << "site[" << isite << "] adj vel: " << velocity << std::endl;
            cross_pos_[isite] += delta * velocity;
        }
    }
    // Here is the diffusion
    if (parent_->af_side_diffusion_ > 0.0 || parent_->af_tip_diffusion_ > 0.0) {
        for (int isite = 0; isite < nsites_; ++isite) {
            if (attach_[isite] == -1)
                continue;
            int ibond = attach_[isite];
            //std::cout << "initial crosspos[" << isite << "]: " << cross_pos_[isite] << std::endl;
            // Check the tip enhancement and reduce the set size accordingly
            double dstab = lbond[ibond] - cross_pos_[isite];
            double diffusion_const = 0.0;
            if (dstab < parent_->af_tip_distance_) {
                diffusion_const = parent_->af_tip_diffusion_;
            } else {
                diffusion_const = parent_->af_side_diffusion_;
            }
            // random 1d diffusion
            cross_pos_[isite] += sqrt(2.0 * diffusion_const * delta) *
                gsl_ran_gaussian_ziggurat(r, 1.0);
            // Force dependent diffusion
            double ui_dot_f = -dot_product(ndim, ubond[ibond], flink_[isite]);
            //std::cout << "force diffusion[" << isite << "] with bond[" << ibond << "]\n";
            //std::cout << "  ubond(" << ubond[ibond][0] << ", " << ubond[ibond][1] << ", " << ubond[ibond][2] << ")\n";
            //std::cout << "  flink(" << flink_[isite][0] << ", " << flink_[isite][1] << ", " << flink_[isite][2] << ")\n";
            //std::cout << "  ui_dot_f: " << ui_dot_f << std::endl;
            cross_pos_[isite] += ui_dot_f * diffusion_const * delta; // Force in the direction of diffusion
            // Tip enhancement diffusion
        }
    }

    // Here is where we do tip tracking. If the bond is growing, and we are tip-attached, then zip with the tip
    if (parent_->af_tip_tracking_ > 0.0) {
        for (int isite = 0; isite < nsites_; ++isite) {
            if (attach_[isite] == -1) {
                lbond_last_[isite] = -1.0; //reset the lbond_last to unknown
                continue;
            }
            int ibond = attach_[isite];
            double dstab = lbond[ibond] - cross_pos_[isite];

            // IF the KC is tip attached AND is growing, do the zipping, otherwise, reset the last known
            // bond length
            if ((dstab < parent_->af_tip_distance_) && (parent_->properties_->bonds.poly_state[ibond] == GROW)) {
                //std::cout << "bond: " << ibond << " starting tip tracking\n";
                // Check if this is the first growing time we know about and set
                if (lbond_last_[isite] < 0.0) {
                    lbond_last_[isite] = lbond[ibond];
                }
                //std::cout << "lbond_last to start: " << lbond_last_[isite] << std::endl;

                double delta_L = lbond[ibond] - lbond_last_[isite];
                //std::cout << "delta_L: " << delta_L << std::endl;

                // Zip this amount times the tip tracking value toward the MT tip
                //std::cout << "Tip tracking, bond: " << ibond << ", delta_L: " << delta_L << ", zip: " << delta_L * parent_->af_tip_tracking_ << std::endl;
                cross_pos_[isite] += delta_L * parent_->af_tip_tracking_;
                lbond_last_[isite] = lbond[ibond];
                //std::cout << "lbond_last to end: " << lbond_last_[isite] << std::endl;
            } else {
                lbond_last_[isite] = -1.0;
            }
        }
    }

    // No matter what, check if we've gone off the end and reset this
    for (int isite = 0; isite < nsites_; ++isite) {
        int ibond = attach_[isite];
        if (ibond == -1)
            continue;
        
        if (cross_pos_[isite] > lbond[ibond]) {
            cross_pos_[isite] = lbond[ibond];
        }
        if (cross_pos_[isite] < 0.0) {
            cross_pos_[isite] = 0.0;
        }
    }
}

// Determine the attachment status
void Kinetochore::DetermineAttachmentStatus(system_parameters *parameters,
                                            system_properties *properties) {
    //std::cout << "Kinetochore[" << idx_ << "] determining attachment status\n";
    int pole[nsites_];
    *attach_status_ = -1; // Default to unattached
    bool unattached = true;
    // Check to see if we even have any MTs per SPB
    if (nmt_per_spb_ <= 0) {
        return;
    }
    for (int isite = 0; isite < nsites_; ++isite) {
        if (attach_[isite] != -1) {
            pole[isite] = attach_[isite] / nmt_per_spb_;
            unattached = false;
        } else {
            pole[isite] = -1;
        }
    }

    // Check first to see if unattached (all are equal to -1)
    if (unattached) {
        return;
    }

    // Now, figure out the attachment status
    int first_pole = -1;
    for (int isite = 0; isite < nsites_; ++isite) {
        // Check if this pole isn't unattached
        if (pole[isite] >= 0) {
            // If the first_pole is unattached, then grab the pole we are attached to
            if (first_pole == -1) {
                first_pole = pole[isite];
            } else if (first_pole != pole[isite]) {
                // First pole does't match current pole, merotelic attachment
                *attach_status_ = 2;
                return;
            }
        }
    }
    *attach_status_ = first_pole;

    //std::cout << "   pole[" << pole[0] << ", " << pole[1] << ", " << pole[2] << "]\n";
    //std::cout << "   attach status: " << *attach_status_ << std::endl;
}
