// Kinetochore Microtubule repulsive harmonic potential

#include "bob.h"

#include "kinetochore.h"
#include "minimum_distance.h"
#include "triangle_mesh.h"

#include <iostream>

double kinetochoremesh_mt_wca_potential_allpairs(system_parameters *parameters, system_properties *properties,
                                                 double **f_bond, double **virial, double **t_bond, int *calc_matrix) {
    double u = 0.0;

    const double r_cutoff = pow(2, 1.0/6.0) * 0.5; // infinite thinness, half of the microtubule
    const double r_cutoff2 = SQR(r_cutoff);
    const double sigma2 = 0.25; // 0.5 ** 2
    const double four_epsilon = 4.0;
    const double rho2 = sigma2 / r_cutoff2;
    const double rho6 = CUBE(rho2);
    const double rho12 = SQR(rho6);
    const double u_shift = -four_epsilon * (rho12 - rho6);

    // Set up shortcuts
    int ndim = parameters->n_dim;
    int nperiodic = parameters->n_periodic;
    int n_bonds = properties->bonds.n_bonds;
    int nchromosomes = properties->chromosomes.nchromosomes_;
    int nkcs = properties->chromosomes.nkcs_;
    double radiuskc = properties->chromosomes.kc_diameter_ * 0.5;
    double **h = properties->unit_cell.h;
    double **r_bond = properties->bonds.r_bond;
    double **s_bond = properties->bonds.s_bond;
    double **u_bond = properties->bonds.u_bond;
    double *l_bond = properties->bonds.length;
    double **f_kc = properties->chromosomes.f_;
    double **t_kc = properties->chromosomes.t_;
    double **ftip = properties->chromosomes.fkcmttip_;

    ChromosomeManagement* chromosomes = &properties->chromosomes;

    if (properties->control.virial_flag)
        memset(virial[0], 0, ndim * ndim * sizeof(double));
    memset(f_bond[0], 0, n_bonds * ndim * sizeof(double));
    memset(t_bond[0], 0, n_bonds * 3 * sizeof(double));

    // Really this loops over kinetochores (and chromatids) which are the interacting partners
    for (int ikc = 0; ikc < nkcs; ++ikc) {
        //Kinetochore *kc_iter = &(chromosomes->kinetochores_[ikc]);
        //double flipsister = kc_iter->second_sister_ ? -1.0 : 1.0; //needed to get offset of chromatid

        // Brute force this for now
        for (int ibond = 0; ibond < n_bonds; ++ibond) {

            //if (properties->i_current_step >= 2661830) {
            //    std::cout << "rbond (" << r_bond[ibond][0] << ", "
            //                           << r_bond[ibond][1] << ", "
            //                           << r_bond[ibond][2] << ")\n";
            //    std::cout << "ubond (" << u_bond[ibond][0] << ", "
            //                           << u_bond[ibond][1] << ", "
            //                           << u_bond[ibond][2] << ")\n";
            //    std::cout << "lbond " << l_bond[ibond] << std::endl;
            //}
            double rmin[3], r_min_mag2, rcontact[3], mu;
            min_distance_sphero_polygon(parameters->n_dim,
                                        parameters->n_periodic,
                                        properties->unit_cell.h,
                                        r_bond[ibond],
                                        s_bond[ibond],
                                        u_bond[ibond],
                                        l_bond[ibond],
                                        &(chromosomes->tris_[ikc]),
                                        rmin, &r_min_mag2, rcontact, &mu);

            //if (properties->i_current_step >= 2661830) {
            //    std::cout << "KC[" << ikc << "]bond[" << ibond << "]\n";
            //    std::cout << "rminmag2: " << r_min_mag2 << std::endl;
            //    std::cout << "rmin (" << rmin[0] << ", "
            //                          << rmin[1] << ", "
            //                          << rmin[2] << ")\n";
            //    std::cout << "rcontact (" << r_contact[0] << ", "
            //                              << r_contact[1] << ", "
            //                              << r_contact[2] << ")\n";
            //}
            if (r_min_mag2 < r_cutoff2) {
                //std::cout << "Firing off WCA calc\n";
                // Calculate WCA potential and forces
                double rho2 = sigma2 / r_min_mag2;
                double rho6 = CUBE(rho2);
                double rho12 = SQR(rho6);

                u += four_epsilon * (rho12 - rho6) + u_shift;
                double factor = 6.0 * four_epsilon * (2.0 * rho12 - rho6) / r_min_mag2;
                double f_cutoff = 0.1 / parameters->delta * 
                    MIN(properties->bonds.gamma_par[ibond],
                        chromosomes->gamma_t_);

                // Truncate the forces if necessary
                double r_min_mag = sqrt(r_min_mag2);
                if (factor * r_min_mag > f_cutoff) {
                    //std::cout << "NOTE tripping fcutoff in kcmt, resetting force factor, original " << factor << " to ";
                    factor = f_cutoff / r_min_mag;
                    //std::cout << factor << std::endl;
                    printf(" *** Force exceeded f_cutoff kinetochoremesh_mt_wca_potential_allpairs ***\n");
                }
                double f_lj[3] = {0.0};
                for (int i = 0; i < ndim; ++i) {
                    f_lj[i] = factor * rmin[i];
                }
                //std::cout << "KC MT Current forces Steric:\n";
                //std::cout << "  KC[" << ikc << "] bond[" << ibond << "]\n";
                //std::cout << "  f_lj: (" << f_lj[0] << ", " << f_lj[1] << ", " << f_lj[2] << ")\n";
                //std::cout << "  fkc:   (" << f_kc[ikc][0] << ", " << f_kc[ikc][1] << ", " << f_kc[ikc][2] << ")\n";
                //std::cout << "  fbond: (" << f_bond[ibond][0] << ", " << f_bond[ibond][1] << ", " << f_bond[ibond][2] << ")\n";

                // Add to accumulators
                for (int i = 0; i < ndim; ++i) {
                    f_kc[ikc][i] += f_lj[i];
                    f_bond[ibond][i] -= f_lj[i];
                    //ftip[ibond][i] -= f_lj[i]; // we are definitely talking to the tip
                }
                // We need to do a special check to see if we're within the tip distance for the KC-MT interaction
                // for force dependent catastrophe
                if (properties->bonds.length[ibond] - (mu - 0.5)*properties->bonds.length[ibond] < chromosomes->chromatid_mt_fc_distance_) {
                    for (int i = 0; i < ndim; ++i) {
                        ftip[ibond][i] -= f_lj[i];
                    }
                }

                // Calculate the virial contribution
                if (properties->control.virial_flag == 1) {
                    double r_cm[3] = {0.0};
                    for (int i = 0; i < parameters->n_dim; ++i) {
                        r_cm[i] = chromosomes->kinetochores_[ikc].r_[i] - r_bond[ibond][i];
                    }
                    for (int i = 0; i < parameters->n_dim; ++i) {
                        for (int j = 0; j < parameters->n_dim; ++j) {
                            virial[i][j] += r_cm[i] * f_lj[j];
                        }
                    }
                }

                // Calculate torques
                double rcontact_kc[3] = {0.0};
                double rcontact_mt[3] = {0.0};
                for (int i = 0; i < ndim; ++i) {
                    //rcontact_kc[i] = kc_iter->r_[i] - rcontact[i]; 
                    rcontact_kc[i] = chromosomes->r_[ikc][i] - rcontact[i];
                    rcontact_mt[i] = mu * u_bond[ibond][i];
                }
                //std::cout << "rcontact_kc(" << rcontact_kc[0] << ", " << rcontact_kc[1] << ", " << rcontact_kc[2] << ")\n";
                //std::cout << "rcontact_mt(" << rcontact_mt[0] << ", " << rcontact_mt[1] << ", " << rcontact_mt[2] << ")\n";
                double tau[3] = {0.0};
                cross_product(rcontact_kc, f_lj, tau, 3);
                for (int i = 0; i < 3; ++i) {
                    t_kc[ikc][i] -= tau[i];
                }
                cross_product(rcontact_mt, f_lj, tau, 3);
                for (int i = 0; i < 3; ++i) {
                    t_bond[ibond][i] -= tau[i];
                }

                //// FIXME XXX
                //// Make sure we are never, ever, ever attractive
                //double rdifforig[3] = {0.0};
                //for (int i = 0; i < ndim; ++i) {
                //    rdifforig[i] = properties->bonds.r_bond[ibond][i] - chromosomes->r_[ikc][i];
                //}
                //double rdiffnew[3] = {0.0};
                //for (int i = 0; i < ndim; ++i) {
                //    rdiffnew[i] = properties->bonds.r_bond[ibond][i] - (chromosomes->r_[ikc][i] + f_lj[i] * parameters->delta / properties->chromosomes.gamma_t_);
                //}

                //if ((sqrt(dot_product(ndim, rdifforig, rdifforig)) > sqrt(dot_product(ndim, rdiffnew, rdiffnew))) ||
                //    (dot_product(ndim, rmin, f_lj) < 0.0)) {
                ////if (dot_product(ndim, rmin, f_lj) < 0.0) {
                //    std::cout << "WARNING, attractive interaction KC-MT steric\n";
                //    std::cout << "KC[" << ikc << "] bond[" << ibond << "]\n";
                //    std::cout << "  rkc  ("
                //        << chromosomes->r_[ikc][0] << ", "
                //        << chromosomes->r_[ikc][1] << ", "
                //        << chromosomes->r_[ikc][2] << ")\n";
                //    std::cout << "  rb   ("
                //        << properties->bonds.r_bond[ibond][0] << ", "
                //        << properties->bonds.r_bond[ibond][1] << ", "
                //        << properties->bonds.r_bond[ibond][2] << ")\n";
                //    std::cout << "  f_lj (" << f_lj[0] << ", "
                //                            << f_lj[1] << ", "
                //                            << f_lj[2] << ")\n";
                //    std::cout << "  rmin (" << rmin[0] << ", "
                //                            << rmin[1] << ", "
                //                            << rmin[2] << ")\n";
                //    exit(1);
                //}
                //// FIXME XXX
            }

            if (u != u) {
                std::cerr << "NaN encountered in chromosome_mt_soft_gaussian_potential\n";
                std::cerr << "Step: " << properties->i_current_step << std::endl;
                std::cerr << "Kinetochore[" << ikc << "], bond[" << ibond << "]\n";
            }
        }
    }

    return u;
}

// Neighbor list based kinetochore mesh potential check
// Shoudl be faster than the allpairs version
double kinetochoremesh_mt_wca_potential_neighbors(system_parameters *parameters, system_properties *properties,
                                                  double **f_bond, double **virial, double **t_bond, int *calc_matrix) {
    double u = 0.0;

    const double r_cutoff = pow(2, 1.0/6.0) * 0.5; // infinite thinness, half of the microtubule
    const double r_cutoff2 = SQR(r_cutoff);
    const double sigma2 = 0.25; // 0.5 ** 2
    const double four_epsilon = 4.0;
    const double rho2 = sigma2 / r_cutoff2;
    const double rho6 = CUBE(rho2);
    const double rho12 = SQR(rho6);
    const double u_shift = -four_epsilon * (rho12 - rho6);

    // Set up shortcuts
    int ndim = parameters->n_dim;
    int nperiodic = parameters->n_periodic;
    int n_bonds = properties->bonds.n_bonds;
    int nchromosomes = properties->chromosomes.nchromosomes_;
    int nkcs = properties->chromosomes.nkcs_;
    double radiuskc = properties->chromosomes.kc_diameter_ * 0.5;
    double **h = properties->unit_cell.h;
    double **r_bond = properties->bonds.r_bond;
    double **s_bond = properties->bonds.s_bond;
    double **u_bond = properties->bonds.u_bond;
    double *l_bond = properties->bonds.length;
    double **f_kc = properties->chromosomes.f_;
    double **t_kc = properties->chromosomes.t_;
    double **ftip = properties->chromosomes.fkcmttip_;

    ChromosomeManagement* chromosomes = &properties->chromosomes;

    if (properties->control.virial_flag)
        memset(virial[0], 0, ndim * ndim * sizeof(double));
    memset(f_bond[0], 0, n_bonds * ndim * sizeof(double));
    memset(t_bond[0], 0, n_bonds * 3 * sizeof(double));

    // Force the chromosomes to check the neighbor lists for validity
    chromosomes->CheckKCmeshNL();

    // Loop over the kinetochores and check the neighbor lists
    for (int ikc = 0; ikc < nkcs; ++ikc) {
        // Check the neighbor list of this kc
        for (nl_list::iterator p  = chromosomes->neighbors_[ikc].begin();
                               p != chromosomes->neighbors_[ikc].end();
                               p++) {
            int ibond = p->label;
            //std::cout << "KC " << ikc << " WCA with bond " << ibond << std::endl;
            //if (properties->i_current_step >= 2661830) {
            //    std::cout << "rbond (" << r_bond[ibond][0] << ", "
            //                           << r_bond[ibond][1] << ", "
            //                           << r_bond[ibond][2] << ")\n";
            //    std::cout << "ubond (" << u_bond[ibond][0] << ", "
            //                           << u_bond[ibond][1] << ", "
            //                           << u_bond[ibond][2] << ")\n";
            //    std::cout << "lbond " << l_bond[ibond] << std::endl;
            //}
            double rmin[3], r_min_mag2, rcontact[3], mu;
            min_distance_sphero_polygon(parameters->n_dim,
                                        parameters->n_periodic,
                                        properties->unit_cell.h,
                                        r_bond[ibond],
                                        s_bond[ibond],
                                        u_bond[ibond],
                                        l_bond[ibond],
                                        &(chromosomes->tris_[ikc]),
                                        rmin, &r_min_mag2, rcontact, &mu);

            //if (properties->i_current_step >= 2661830) {
            //    std::cout << "KC[" << ikc << "]bond[" << ibond << "]\n";
            //    std::cout << "rminmag2: " << r_min_mag2 << std::endl;
            //    std::cout << "rmin (" << rmin[0] << ", "
            //                          << rmin[1] << ", "
            //                          << rmin[2] << ")\n";
            //    std::cout << "rcontact (" << r_contact[0] << ", "
            //                              << r_contact[1] << ", "
            //                              << r_contact[2] << ")\n";
            //}
            if (r_min_mag2 < r_cutoff2) {
                //std::cout << "Firing off WCA calc\n";
                // Calculate WCA potential and forces
                double rho2 = sigma2 / r_min_mag2;
                double rho6 = CUBE(rho2);
                double rho12 = SQR(rho6);

                u += four_epsilon * (rho12 - rho6) + u_shift;
                double factor = 6.0 * four_epsilon * (2.0 * rho12 - rho6) / r_min_mag2;
                double f_cutoff = 0.1 / parameters->delta * 
                    MIN(properties->bonds.gamma_par[ibond],
                        chromosomes->gamma_t_);

                // Truncate the forces if necessary
                double r_min_mag = sqrt(r_min_mag2);
                if (factor * r_min_mag > f_cutoff) {
                    //std::cout << "NOTE tripping fcutoff in kcmt, resetting force factor, original " << factor << " to ";
                    factor = f_cutoff / r_min_mag;
                    //std::cout << factor << std::endl;
                    printf(" *** Force exceeded f_cutoff kinetochoremesh_mt_wca_potential_neighbors ***\n");
                }
                double f_lj[3] = {0.0};
                for (int i = 0; i < ndim; ++i) {
                    f_lj[i] = factor * rmin[i];
                }
                //std::cout << "KC MT Current forces Steric:\n";
                //std::cout << "  KC[" << ikc << "] bond[" << ibond << "]\n";
                //std::cout << "  f_lj: (" << f_lj[0] << ", " << f_lj[1] << ", " << f_lj[2] << ")\n";
                //std::cout << "  fkc:   (" << f_kc[ikc][0] << ", " << f_kc[ikc][1] << ", " << f_kc[ikc][2] << ")\n";
                //std::cout << "  fbond: (" << f_bond[ibond][0] << ", " << f_bond[ibond][1] << ", " << f_bond[ibond][2] << ")\n";

                // Add to accumulators
                for (int i = 0; i < ndim; ++i) {
                    f_kc[ikc][i] += f_lj[i];
                    f_bond[ibond][i] -= f_lj[i];
                    //ftip[ibond][i] -= f_lj[i]; // we are definitely talking to the tip
                }
                // We need to do a special check to see if we're within the tip distance for the KC-MT interaction
                // for force dependent catastrophe
                if (properties->bonds.length[ibond] - (mu - 0.5)*properties->bonds.length[ibond] < chromosomes->chromatid_mt_fc_distance_) {
                    for (int i = 0; i < ndim; ++i) {
                        ftip[ibond][i] -= f_lj[i];
                    }
                }

                // Calculate the virial contribution
                if (properties->control.virial_flag == 1) {
                    for (int i = 0; i < parameters->n_dim; ++i) {
                        for (int j = 0; j < parameters->n_dim; ++j) {
                            virial[i][j] += rmin[i] * f_lj[j];
                        }
                    }
                }

                // Calculate torques
                double rcontact_kc[3] = {0.0};
                double rcontact_mt[3] = {0.0};
                for (int i = 0; i < ndim; ++i) {
                    //rcontact_kc[i] = kc_iter->r_[i] - rcontact[i]; 
                    rcontact_kc[i] = chromosomes->r_[ikc][i] - rcontact[i];
                    rcontact_mt[i] = mu * u_bond[ibond][i];
                }
                //std::cout << "rcontact_kc(" << rcontact_kc[0] << ", " << rcontact_kc[1] << ", " << rcontact_kc[2] << ")\n";
                //std::cout << "rcontact_mt(" << rcontact_mt[0] << ", " << rcontact_mt[1] << ", " << rcontact_mt[2] << ")\n";
                double tau[3] = {0.0};
                cross_product(rcontact_kc, f_lj, tau, 3);
                for (int i = 0; i < 3; ++i) {
                    t_kc[ikc][i] -= tau[i];
                }
                cross_product(rcontact_mt, f_lj, tau, 3);
                for (int i = 0; i < 3; ++i) {
                    t_bond[ibond][i] -= tau[i];
                }

                //// FIXME XXX
                //// Make sure we are never, ever, ever attractive
                //double rdifforig[3] = {0.0};
                //for (int i = 0; i < ndim; ++i) {
                //    rdifforig[i] = properties->bonds.r_bond[ibond][i] - chromosomes->r_[ikc][i];
                //}
                //double rdiffnew[3] = {0.0};
                //for (int i = 0; i < ndim; ++i) {
                //    rdiffnew[i] = properties->bonds.r_bond[ibond][i] - (chromosomes->r_[ikc][i] + f_lj[i] * parameters->delta / properties->chromosomes.gamma_t_);
                //}

                //if ((sqrt(dot_product(ndim, rdifforig, rdifforig)) > sqrt(dot_product(ndim, rdiffnew, rdiffnew))) ||
                //    (dot_product(ndim, rmin, f_lj) < 0.0)) {
                ////if (dot_product(ndim, rmin, f_lj) < 0.0) {
                //    std::cout << "WARNING, attractive interaction KC-MT steric\n";
                //    std::cout << "KC[" << ikc << "] bond[" << ibond << "]\n";
                //    std::cout << "  rkc  ("
                //        << chromosomes->r_[ikc][0] << ", "
                //        << chromosomes->r_[ikc][1] << ", "
                //        << chromosomes->r_[ikc][2] << ")\n";
                //    std::cout << "  rb   ("
                //        << properties->bonds.r_bond[ibond][0] << ", "
                //        << properties->bonds.r_bond[ibond][1] << ", "
                //        << properties->bonds.r_bond[ibond][2] << ")\n";
                //    std::cout << "  f_lj (" << f_lj[0] << ", "
                //                            << f_lj[1] << ", "
                //                            << f_lj[2] << ")\n";
                //    std::cout << "  rmin (" << rmin[0] << ", "
                //                            << rmin[1] << ", "
                //                            << rmin[2] << ")\n";
                //    exit(1);
                //}
                //// FIXME XXX
            }

            if (u != u) {
                std::cerr << "NaN encountered in kinetochoremesh_mt_wca_neighbors\n";
                std::cerr << "Step: " << properties->i_current_step << std::endl;
                std::cerr << "Kinetochore[" << ikc << "], bond[" << ibond << "]\n";
            }
        }
    }

    return u;
}

