// Implementation of advanced (multi-dimensional) lookup table

#include "lookup_table_advanced.h"
#include <algorithm>
#include <chrono>
#include <cmath>
#include <iostream>

LookupTableAdvanced::~LookupTableAdvanced() {
    ngrid_.clear();
    table_.clear();

    // x_ is an array of vectors, that's what you use delete[] (as opposed to delete)
    delete[] x_;
}

void LookupTableAdvanced::Init(int ndim,
                               std::vector<double> *x,
                               double (*func) (std::vector<double> &x, void *params),
                               void *params) {
    ndim_ = ndim;

    x_ = new std::vector<double>[ndim_];
    ngrid_.resize(ndim_);
    std::vector<double> minx;
    minx.resize(ndim_);
    xshift_.resize(ndim_);
    for (int i = 0; i < ndim_; ++i) {
        x_[i] = x[i];
        minx[i] = *std::min_element(x_[i].begin(), x_[i].end());
        xshift_[i] = minx[i];
        ngrid_[i] = x_[i].size();
    }

    //if (ndim_ == 2) {
    //    std::cout << "ngrid(" << ngrid_[0] << ", " << ngrid_[1] << ")\n";
    //} else if (ndim_ == 3) {
    //    std::cout << "ngrid(" << ngrid_[0] << ", " << ngrid_[1] << ", " << ngrid_[2] << ")\n";
    //} else if (ndim_ == 4) {
    //    std::cout << "ngrid(" << ngrid_[0] << ", " << ngrid_[1] << ", " << ngrid_[2] << ", " << ngrid_[3] << ")\n";
    //}

    // FIXME Hardcode 2,3,4 dimensions
    std::vector<double> xvec;
    xvec.resize(4);
    if (ndim_ == 2) {
        table_.resize(ngrid_[0] * ngrid_[1]);
        for (int i = 0; i < ngrid_[0]; ++i){
            for (int j = 0; j < ngrid_[1]; ++j) {
                xvec[0] = x_[0][i];
                xvec[1] = x_[1][j];
                xvec[2] = 0.0;
                xvec[3] = 0.0;
                //table_[LinearIndex2(i, j, ngrid_[0], ngrid_[1])] = func(xvec, params);
                table_[LinearIndex2(i, j)] = func(xvec, params);
            }
        }
    } else if (ndim_ == 3) {
        table_.resize(ngrid_[0] * ngrid_[1] * ngrid_[2]);
        for (int i = 0; i < ngrid_[0]; ++i) {
            for (int j = 0; j < ngrid_[1]; ++j) {
                for (int k = 0; k < ngrid_[2]; ++k) {
                    xvec[0] = x_[0][i];
                    xvec[1] = x_[1][j];
                    xvec[2] = x_[2][k];
                    xvec[3] = 0.0;
                    table_[LinearIndex3(i, j, k)] = func(xvec, params);
                }
            }
        }
    } else if (ndim_ == 4) {
        table_.resize(ngrid_[0] * ngrid_[1] * ngrid_[2] * ngrid_[3]);
        for (int i = 0; i < ngrid_[0]; ++i){
            for (int j = 0; j < ngrid_[1]; ++j) {
                for (int k = 0; k < ngrid_[2]; ++k) {
                    for (int l = 0; l < ngrid_[3]; ++l) {
                        xvec[0] = x_[0][i];
                        xvec[1] = x_[1][j];
                        xvec[2] = x_[2][k];
                        xvec[3] = x_[3][l];
                        table_[LinearIndex4(i, j, k, l)] = func(xvec, params);
                    }
                }
            }
        }
    } else {
        std::cerr << "Dimension " << ndim_ << " not in advanced lookup table yet, exiting!\n";
        exit(1);
    }

    //std::cout << "Finished building lookup table\n";
}

void LookupTableAdvanced::InitAsymmetric(int ndim,
                                         std::vector<double> *x,
                                         double (*func_upper) (std::vector<double> &x, void *params),
                                         double (*func_lower) (std::vector<double> &x, void *params),
                                         void *params) {
    auto start_time = std::chrono::high_resolution_clock::now();
    ndim_ = ndim;

    x_ = new std::vector<double>[ndim_];
    ngrid_.resize(ndim_);
    std::vector<double> minx;
    minx.resize(ndim_);
    xshift_.resize(ndim_);
    for (int i = 0; i < ndim_; ++i) {
        x_[i] = x[i];
        minx[i] = *std::min_element(x_[i].begin(), x_[i].end());
        xshift_[i] = minx[i];
        ngrid_[i] = x_[i].size();
    }

    //if (ndim_ == 2) {
    //    std::cout << "ngrid(" << ngrid_[0] << ", " << ngrid_[1] << ")\n";
    //} else if (ndim_ == 3) {
    //    std::cout << "ngrid(" << ngrid_[0] << ", " << ngrid_[1] << ", " << ngrid_[2] << ")\n";
    //} else if (ndim_ == 4) {
    //    std::cout << "ngrid(" << ngrid_[0] << ", " << ngrid_[1] << ", " << ngrid_[2] << ", " << ngrid_[3] << ")\n";
    //    std::cout << "xshift(" << xshift_[0] << ", " << xshift_[1] << ", " << xshift_[2] << ", " << xshift_[3] << ")\n";
    //}

    // FIXME Hardcode 4 dimensions
    std::vector<double> xvec;
    xvec.resize(4);
    if (ndim_ == 4) {
        table_.resize(ngrid_[0] * ngrid_[1] * ngrid_[2] * ngrid_[3]);
        for (int i = 0; i < ngrid_[0]; ++i){
            for (int j = 0; j < ngrid_[1]; ++j) {
                for (int k = 0; k < ngrid_[2]; ++k) {
                    for (int l = 0; l < ngrid_[3]; ++l) {
                        xvec[0] = x_[0][i];
                        xvec[1] = x_[1][j];
                        xvec[2] = x_[2][k];
                        xvec[3] = x_[3][l];
                        // Designed for asymmetric potentials, so load appropriately based on if
                        // xvec[0] < 0 or not
                        if (xvec[0] <= 0.0) {
                            table_[LinearIndex4(i, j, k, l)] = func_lower(xvec, params);
                        } else {
                            table_[LinearIndex4(i, j, k, l)] = func_upper(xvec, params);
                        }
                    }
                }
            }
        }
    } else {
        std::cerr << "Dimension " << ndim_ << " not in advanced lookup table yet for asymmetric potentials, exiting!\n";
        exit(1);
    }

    auto end_time = std::chrono::high_resolution_clock::now();
    auto time_span = std::chrono::duration_cast<std::chrono::duration<double>>(end_time - start_time);
    std::cout << "Lookup table elements: " << table_.size() << ", buildtime: " << time_span.count() << "s\n";

    //std::cout << "Finished building lookup table\n";
}

// Lookup the value that corresponds to x
// Do this via bilinear interpolation
// See: Numerical Recipes - Bilinear interpolation
double LookupTableAdvanced::Lookup(double *x) {
    int lowindex[ndim_];
    double t[ndim_];

    //std::cout << "Lookup x(" << x[0] << ", " << x[1] << ", " << x[2] << ", " << x[3] << ")\n";
    // Find the index points
    for (int i = 0; i < ndim_; ++i) {
        double L = x_[i].back() - x_[i].front();
        int nbin = ngrid_[i] - 1;
        lowindex[i] = (int) ((x[i] - xshift_[i])/L * nbin);
        double pos = x[i];
        if (lowindex[i] >= nbin) {
            lowindex[i] = nbin - 1;
            pos = x_[i].back();
        } else if (lowindex[i] < 0) {
            lowindex[i] = 0;
            pos = x_[i].front();
        }

        // Grab the value of the midpoint along this line
        // Between 0 and 1 describing location between lattice points
        t[i] = (pos - x_[i][lowindex[i]])/(x_[i][lowindex[i]+1] - x_[i][lowindex[i]]);
        //std::cout << "  dim[" << i << "] L = " << L << ", nbin = " << nbin << " lowindex = " << lowindex[i] << ", t = " << t[i] << std::endl;
    }

    // Do the interpolation based on numer of dimensions (hardcoded for now)
    double returnval = 0.0;
    if (ndim_ == 2) {
        // Based on old lookup table bilinear interpolation scheme
        int ny = 2<<(ndim_-1); // bitshift to take 2 to some power
        double y[ny];

        // All combinatorics of the y functions
        y[0] = table_[LinearIndex2(lowindex[0],   lowindex[1]  )];
        y[1] = table_[LinearIndex2(lowindex[0]+1, lowindex[1]  )];
        y[2] = table_[LinearIndex2(lowindex[0]+1, lowindex[1]+1)];
        y[3] = table_[LinearIndex2(lowindex[0]  , lowindex[1]+1)];

        // Calculation of the bilinear interpolation
        returnval = (1 - t[0]) * (1 - t[1]) * y[0]
                  + (    t[0]) * (1 - t[1]) * y[1]
                  + (    t[0]) * (    t[1]) * y[2]
                  + (1 - t[0]) * (    t[1]) * y[3];
        //returnval = RecursiveInterpolation(lowindex, t);
    } else if (ndim_ == 3) {
        // Use what is known from wikipedia on trilinear interpolation to do this

        // Do the interpolation along x first
        double c00 = table_[LinearIndex3(lowindex[0]  , lowindex[1]  , lowindex[2]  )] * (1 - t[0]) +
                     table_[LinearIndex3(lowindex[0]+1, lowindex[1]  , lowindex[2]  )] * (    t[0]);
        double c01 = table_[LinearIndex3(lowindex[0]  , lowindex[1]  , lowindex[2]+1)] * (1 - t[0]) +
                     table_[LinearIndex3(lowindex[0]+1, lowindex[1]  , lowindex[2]+1)] * (    t[0]);
        double c10 = table_[LinearIndex3(lowindex[0]  , lowindex[1]+1, lowindex[2]  )] * (1 - t[0]) +
                     table_[LinearIndex3(lowindex[0]+1, lowindex[1]+1, lowindex[2]  )] * (    t[0]);
        double c11 = table_[LinearIndex3(lowindex[0]  , lowindex[1]+1, lowindex[2]+1)] * (1 - t[0]) +
                     table_[LinearIndex3(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1)] * (    t[0]);

        // Then do the interpolation along y
        double c0 = c00*(1 - t[1]) + c10*(    t[1]);
        double c1 = c01*(1 - t[1]) + c11*(    t[1]);

        // Finally do interpolation along Z
        returnval = c0*(1 - t[2]) + c1*(t[2]);
    } else if (ndim_ == 4) {
        // Pass interpolation off to other function
        returnval = Quadrilinear(lowindex, t);

        // Try the recursive version of this
        //returnval = RecursiveInterpolation(lowindex, t);
    }

    return returnval;
}

double LookupTableAdvanced::Invert(int dim,
                                   double u,
                                   double *val) {
    int lowindex[ndim_];
    double t[ndim_];

    // Find the index points
    for (int i = 0; i < ndim_; ++i) {
        // Exclude the dimension we're looking in
        if (i == dim) continue;
        double L = x_[i].back() - x_[i].front();
        int nbin = ngrid_[i] - 1;
        lowindex[i] = (int) ((val[i] - xshift_[i])/L * nbin);
        double pos = val[i];
        if (lowindex[i] >= nbin) {
            lowindex[i] = nbin - 1;
            pos = x_[i].back();
        } else if (lowindex[i] < 0) {
            lowindex[i] = 0;
            pos = x_[i].front();
        }

        // Grab the value of the midpoint along this line
        // Between 0 and 1 describing location between lattice points
        t[i] = (pos - x_[i][lowindex[i]])/(x_[i][lowindex[i]+1] - x_[i][lowindex[i]]);
    }

    // Set the value and lookup
    val[dim] = x_[dim][ngrid_[dim]-1];
    double amax = Lookup(val);
    // Get the other lookup value
    val[dim] = x_[dim][0];
    amax += Lookup(val);

    // Bin the result to know which one we're looking in for the solution
    int lowi = 0;
    int highi = ngrid_[dim]-1;

    double f_low, f_high;
    std::cout << "Starting bin search, u: " << u << std::endl;
    std::cout << "   amax: " << amax << std::endl;
    int maxiter = 10;
    int iiter = 0;
    while (lowi <= highi) {
        std::cout << "   lowi: " << lowi << ", highi: " << highi << std::endl;
        lowindex[dim] = (highi+lowi)/2;
        std::cout << "   lowindex = " << lowindex[dim] << std::endl;
        val[dim] = x_[dim][lowindex[dim]];
        f_low = Lookup(val)/amax;
        std::cout << "   f_low = " << f_low << std::endl;
        val[dim] = x_[dim][lowindex[dim]+1];
        f_high = Lookup(val)/amax;
        std::cout << "   f_high= " << f_high << std::endl;
        // See if we fall into this bin
        if (f_low < u) {
            if (f_high > u)
                break;
            lowi = lowindex[dim];
        } else if (f_high > u) {
            highi = lowindex[dim];
        }
        iiter++;
        if (iiter > maxiter) {
            std::cout << "over max number iterations!\n";
            exit(1);
        }
    }
    std::cout << "u: " << u << ", target lowindex = " << lowindex[dim] << std::endl;

    // Now comes some really, really awful arithmatic
    double a;
    double b;
    double newval = InverseQuadrilinear(lowindex, t, u, amax, &a, &b);
    val[dim] = (u - b) * (x_[dim][lowindex[dim]+1] - x_[dim][lowindex[dim]]) / a +
        x_[dim][lowindex[dim]];
    return val[dim];
}

double LookupTableAdvanced::Quadrilinear(int *lowindex,
                                         double *t) {

    // After interpreting how the trilinear version works, let's try a naive version of the quadrilinear interpolation
    // First, along x
    // c000 in binary essentially, meaning the other 3 indicies, should be 8 total
    // c000 means from x0 to x1, no others change
    // c010 means from x0 to x1 and z0 to z1
    double c000 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]  , lowindex[3]  )] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]  , lowindex[3]  )] * (    t[0]);
    double c001 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]  , lowindex[3]+1)] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]  , lowindex[3]+1)] * (    t[0]);
    double c010 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]+1, lowindex[3]  )] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]+1, lowindex[3]  )] * (    t[0]);
    double c011 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]+1, lowindex[3]+1)] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]+1, lowindex[3]+1)] * (    t[0]);
    double c100 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]  , lowindex[3]  )] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]  , lowindex[3]  )] * (    t[0]);
    double c101 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]  , lowindex[3]+1)] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]  , lowindex[3]+1)] * (    t[0]);
    double c110 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]+1, lowindex[3]  )] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1, lowindex[3]  )] * (    t[0]);
    double c111 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]+1, lowindex[3]+1)] * (1 - t[0]) +
                  table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1, lowindex[3]+1)] * (    t[0]);
    //std::cout
    //    << "c000: " << c000 << "\n"
    //    << "c001: " << c001 << "\n"
    //    << "c010: " << c010 << "\n"
    //    << "c011: " << c011 << "\n"
    //    << "c100: " << c100 << "\n"
    //    << "c101: " << c101 << "\n"
    //    << "c110: " << c110 << "\n"
    //    << "c111: " << c111 << "\n";

    // Do the next level of interpolation
    // Second, along y
    double c00 = c000 * (1 - t[1]) + c100 * (    t[1]);
    double c01 = c001 * (1 - t[1]) + c101 * (    t[1]);
    double c10 = c010 * (1 - t[1]) + c110 * (    t[1]);
    double c11 = c011 * (1 - t[1]) + c111 * (    t[1]);

    //std::cout
    //    << "c00: " << c00 << "\n"
    //    << "c01: " << c01 << "\n"
    //    << "c10: " << c10 << "\n"
    //    << "c11: " << c11 << "\n";

    // Next level, along z
    double c0 = c00 * (1 - t[2]) + c10 * (   t[2]);
    double c1 = c01 * (1 - t[2]) + c11 * (   t[2]);

    //std::cout
    //    << "c0: " << c0 << "\n"
    //    << "c1: " << c1 << "\n";

    // Final interpolation
    double returnval = c0 * (1 - t[3]) + c1 * (t[3]);
    return returnval;
}

// Inverse quadrilinear lookup
double LookupTableAdvanced::InverseQuadrilinear(int *lowindex,
                                                double *t,
                                                double s,
                                                double amax,
                                                double *a,
                                                double *b) {
    // Get aliases to all the points in the table that we are next to
    double y0000 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]  , lowindex[3]  )];
    double y0001 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]  , lowindex[3]+1)];
    double y0010 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]+1, lowindex[3]  )];
    double y0011 = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]+1, lowindex[3]+1)];

    double y0100 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]  , lowindex[3]  )];
    double y0101 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]  , lowindex[3]+1)];
    double y0110 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]+1, lowindex[3]  )];
    double y0111 = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]+1, lowindex[3]+1)];

    double y1000 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]  , lowindex[3]  )];
    double y1001 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]  , lowindex[3]+1)];
    double y1010 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]+1, lowindex[3]  )];
    double y1011 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]+1, lowindex[3]+1)];

    double y1100 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]  , lowindex[3]  )];
    double y1101 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]  , lowindex[3]+1)];
    double y1110 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1, lowindex[3]  )];
    double y1111 = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1, lowindex[3]+1)];

    // Generate a term for the top
    double u = t[1];
    double v = t[2];
    double w = t[3];
    double aval = ((1 - u)*(1 - v)*(1 - w) * y0000 +
                   (1 - u)*(1 - v)*(    w) * y0001 +
                   (1 - u)*(    v)*(1 - w) * y0010 +
                   (1 - u)*(    v)*(    w) * y0011 +
                   (    u)*(1 - v)*(1 - w) * y0100 +
                   (    u)*(1 - v)*(    w) * y0101 +
                   (    u)*(    v)*(1 - w) * y0110 +
                   (    u)*(    v)*(    w) * y0111);
    double bval = ((1 - u)*(1 - v)*(1 - w) * y1000 +
                   (1 - u)*(1 - v)*(    w) * y1001 +
                   (1 - u)*(    v)*(1 - w) * y1010 +
                   (1 - u)*(    v)*(    w) * y1011 +
                   (    u)*(1 - v)*(1 - w) * y1100 +
                   (    u)*(1 - v)*(    w) * y1101 +
                   (    u)*(    v)*(1 - w) * y1110 +
                   (    u)*(    v)*(    w) * y1111);

    // Return the answer, shifted to match the old lookup table version
    //return (-s + a)/(a - b);
    *a = (bval - aval)/amax;
    *b = aval/amax;
    return (-s + aval)/(aval - bval);
}

double rInterpolate(int n,
                    int i,
                    double *a,
                    double *t) {
    double returnval = 0.0;
    std::cout << "Recursion\n";
    std::cout << "  n,i: (" << n << ", " << i << ")\n";
    // Oh boy, what fun!
    if (n == 1) {
        // Terminate interpolation and return values
        returnval = (1.0 - t[0])*a[i] + (t[0])*a[i+1];
    } else {
        // Double interpolation recursion!
        returnval = (1.0 - t[n-1])*rInterpolate(n-1, i, a, t) + (t[n-1])*rInterpolate(n-1, i+(1<<(n-1)), a, t);
    }
    std::cout << "  val: " << returnval << std::endl;
    return returnval;
}

// Interpolate recursively, ugh
double LookupTableAdvanced::RecursiveInterpolation(int *lowindex,
                                                   double *t) {
    std::cout << "lowindex(" << lowindex[0] << ", " << lowindex[1] << ", " << lowindex[2] << ", " << lowindex[3] << ")\n";
    std::cout << "t(" << t[0] << ", " << t[1] << ", " << t[2] << ", " << t[3] << ")\n";
    // Try the recursive version of this
    double a[2<<ndim_];
    if (ndim_ == 2) {
        a[0] = table_[LinearIndex2(lowindex[0]  , lowindex[1]  )];
        a[1] = table_[LinearIndex2(lowindex[0]+1, lowindex[1]  )];
        a[2] = table_[LinearIndex2(lowindex[0]  , lowindex[1]+1)];
        a[3] = table_[LinearIndex2(lowindex[0]+1, lowindex[1]+1)];
    } else if (ndim_ == 4) {
        // Assign values of the points around a
        // Increment x and y                                                                      // azyx
        a[0 ] = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]  , lowindex[3]  )]; // 0000
        a[1 ] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]  , lowindex[3]  )]; // 0001
        a[2 ] = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]  , lowindex[3]  )]; // 0010
        a[3 ] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]  , lowindex[3]  )]; // 0011

        // Increment z, then xy
        a[4 ] = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]+1, lowindex[3]  )]; // 0100
        a[5 ] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]+1, lowindex[3]  )]; // 0101
        a[6 ] = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]+1, lowindex[3]  )]; // 0110
        a[7 ] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1, lowindex[3]  )]; // 0111

        // Increment a, then xy
        a[8 ] = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]  , lowindex[3]+1)]; // 1000
        a[9 ] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]  , lowindex[3]+1)]; // 1001
        a[10] = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]  , lowindex[3]+1)]; // 1010
        a[11] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]  , lowindex[3]+1)]; // 1011

        // Increment a, z, xy
        a[12] = table_[LinearIndex4(lowindex[0]  , lowindex[1]  , lowindex[2]+1, lowindex[3]+1)]; // 1100
        a[13] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]  , lowindex[2]+1, lowindex[3]+1)]; // 1101
        a[14] = table_[LinearIndex4(lowindex[0]  , lowindex[1]+1, lowindex[2]+1, lowindex[3]+1)]; // 1110
        a[15] = table_[LinearIndex4(lowindex[0]+1, lowindex[1]+1, lowindex[2]+1, lowindex[3]+1)]; // 1111
    }

    double returnval = rInterpolate(ndim_, 0, a, t);

    return returnval;
}
