#ifndef _LOOKUP_TABLE_ADVANCED_H_
#define _LOOKUP_TABLE_ADVANCED_H_

#include <vector>

/*
 * Class LookupTableAdvanced
 *      Creates an advanced lookup table for the evaluation of integrals for
 *      some number of dimensions. Supports interpolated lookup of points
 *      on the lattice, and inversion for probability density supports.
 */
class LookupTableAdvanced {
  public:
    int ndim_;

    std::vector<int> ngrid_; // [dim]: number of grid points along dimension dim
                             // EX: ngrid_[0] = 10 -> 10 grid points along x dimension

    std::vector<double> *x_; // [dim][pointnumber]: input values at lattice points {x,y,z,a,....}
                             // EX: x_[0][1] -> x dimension, lattice point 1

    std::vector<double> table_; // [linearindex]: function value at linear index
                                // EX: table_[10] -> function value at linear index 10
    
    std::vector<double> xshift_; // [dim] shift value to bring minimum value along dimension to zero

  public:

    LookupTableAdvanced() {}
    ~LookupTableAdvanced();

    /*
     * Inputs:
     *      ndim: number of dimensions in the lookup table
     *      x: lattice points to evaluate function on
     *      func: funciton to evaluate
     *      params: parameters for function
     * Outputs:
     *      None
     * Purpose:
     *      Initializes the lookup table by evalulating the function
     *      on the lattice points.
     */
    void Init(int ndim,
              std::vector<double> *x,
              double (*func) (std::vector<double> &x, void *params),
              void *params);
    void InitAsymmetric(int ndim,
                        std::vector<double> *x,
                        double (*func_upper) (std::vector<double> &x, void *params),
                        double (*func_lower) (std::vector<double> &x, void *params),
                        void *params);
    double Lookup(double *x);
    double Invert(int dim,
                  double u,
                  double *val);

    // Linear indexing functions
    inline int LinearIndex2(int ix, int iy) {
        return ix * ngrid_[1] + iy;
    }
    inline int LinearIndex3(int ix, int iy, int iz) {
        return ix * ngrid_[1] * ngrid_[2] +
               iy * ngrid_[2] +
               iz;
    }
    inline int LinearIndex4(int ix, int iy, int iz, int ia) {
        return ix * ngrid_[1] * ngrid_[2] * ngrid_[3] +
               iy * ngrid_[2] * ngrid_[3] +
               iz * ngrid_[3] +
               ia;
    }

    // 4D interpolation functions
    double Quadrilinear(int *lowindex,
                        double *t);
    double RecursiveInterpolation(int *lowindex,
                                  double *t);
    double InverseQuadrilinear(int *lowindex,
                               double *t,
                               double s,
                               double amax,
                               double *a,
                               double *b);
};
    
#endif
