/* Routines from Numerical Recipes (slightly modified). */

#include "bob.h"

#define ROTATE(a,i,j,k,l) g=a[i][j];h=a[k][l];a[i][j]=g-s*(h+g*tau);\
	a[k][l]=h+s*(g-h*tau);

void jacobi(double **a, int n, double d[], double **v, int *nrot)
{
    int j, iq, ip, i;
    double tresh, theta, tau, t, sm, s, h, g, c, *b, *z;

    b = dvector(1, n);
    z = dvector(1, n);
    for (ip = 1; ip <= n; ip++) {
        for (iq = 1; iq <= n; iq++)
            v[ip][iq] = 0.0;
        v[ip][ip] = 1.0;
    }
    for (ip = 1; ip <= n; ip++) {
        b[ip] = d[ip] = a[ip][ip];
        z[ip] = 0.0;
    }
    *nrot = 0;
    for (i = 1; i <= 50; i++) {
        sm = 0.0;
        for (ip = 1; ip <= n - 1; ip++) {
            for (iq = ip + 1; iq <= n; iq++)
                sm += fabs(a[ip][iq]);
        }
        if (sm == 0.0) {
            free_dvector(z, 1, n);
            free_dvector(b, 1, n);
            return;
        }
        if (i < 4)
            tresh = 0.2 * sm / (n * n);
        else
            tresh = 0.0;
        for (ip = 1; ip <= n - 1; ip++) {
            for (iq = ip + 1; iq <= n; iq++) {
                g = 100.0 * fabs(a[ip][iq]);
                if (i > 4 && (double) (fabs(d[ip]) + g) == (double) fabs(d[ip])
                    && (double) (fabs(d[iq]) + g) == (double) fabs(d[iq]))
                    a[ip][iq] = 0.0;
                else if (fabs(a[ip][iq]) > tresh) {
                    h = d[iq] - d[ip];
                    if ((double) (fabs(h) + g) == (double) fabs(h))
                        t = (a[ip][iq]) / h;
                    else {
                        theta = 0.5 * h / (a[ip][iq]);
                        t = 1.0 / (fabs(theta) + sqrt(1.0 + theta * theta));
                        if (theta < 0.0)
                            t = -t;
                    }
                    c = 1.0 / sqrt(1 + t * t);
                    s = t * c;
                    tau = s / (1.0 + c);
                    h = t * a[ip][iq];
                    z[ip] -= h;
                    z[iq] += h;
                    d[ip] -= h;
                    d[iq] += h;
                    a[ip][iq] = 0.0;
                    for (j = 1; j <= ip - 1; j++) {
                        ROTATE(a, j, ip, j, iq)
                    }
                    for (j = ip + 1; j <= iq - 1; j++) {
                        ROTATE(a, ip, j, j, iq)
                    }
                    for (j = iq + 1; j <= n; j++) {
                        ROTATE(a, ip, j, iq, j)
                    }
                    for (j = 1; j <= n; j++) {
                        ROTATE(v, j, ip, j, iq)
                    }
                    ++(*nrot);
                }
            }
        }
        for (ip = 1; ip <= n; ip++) {
            b[ip] += z[ip];
            d[ip] = b[ip];
            z[ip] = 0.0;
        }
    }
    nrerror("Too many iterations in routine jacobi");
}

#undef ROTATE

void eigsrt(double d[], double **v, int n)
{
    int k, j, i;
    double p;

    for (i = 1; i < n; i++) {
        p = d[k = i];
        for (j = i + 1; j <= n; j++)
            if (d[j] >= p)
                p = d[k = j];
        if (k != i) {
            d[k] = d[i];
            d[i] = p;
            for (j = 1; j <= n; j++) {
                p = v[j][i];
                v[j][i] = v[j][k];
                v[j][k] = p;
            }
        }
    }
}

#define TINY 1.0e-20;

void ludcmp(double **a, int n, int *indx, double *d)
{
    int i, imax, j, k;
    double big, dum, sum, temp;
    double *vv;

    vv = dvector(1, n);
    *d = 1.0;
    for (i = 1; i <= n; i++) {
        big = 0.0;
        for (j = 1; j <= n; j++)
            if ((temp = fabs(a[i][j])) > big)
                big = temp;
        if (big == 0.0)
            nrerror("Singular matrix in routine ludcmp");
        vv[i] = 1.0 / big;
    }
    for (j = 1; j <= n; j++) {
        for (i = 1; i < j; i++) {
            sum = a[i][j];
            for (k = 1; k < i; k++)
                sum -= a[i][k] * a[k][j];
            a[i][j] = sum;
        }
        big = 0.0;
        for (i = j; i <= n; i++) {
            sum = a[i][j];
            for (k = 1; k < j; k++)
                sum -= a[i][k] * a[k][j];
            a[i][j] = sum;
            if ((dum = vv[i] * fabs(sum)) >= big) {
                big = dum;
                imax = i;
            }
        }
        if (j != imax) {
            for (k = 1; k <= n; k++) {
                dum = a[imax][k];
                a[imax][k] = a[j][k];
                a[j][k] = dum;
            }
            *d = -(*d);
            vv[imax] = vv[j];
        }
        indx[j] = imax;
        if (a[j][j] == 0.0)
            a[j][j] = TINY;
        if (j != n) {
            dum = 1.0 / (a[j][j]);
            for (i = j + 1; i <= n; i++)
                a[i][j] *= dum;
        }
    }
    free_dvector(vv, 1, n);
}

#undef TINY

void lubksb(double **a, int n, int *indx, double b[])
{
    int i, ii = 0, ip, j;
    double sum;

    for (i = 1; i <= n; i++) {
        ip = indx[i];
        sum = b[ip];
        b[ip] = b[i];
        if (ii)
            for (j = ii; j <= i - 1; j++)
                sum -= a[i][j] * b[j];
        else if (sum)
            ii = i;
        b[i] = sum;
    }
    for (i = n; i >= 1; i--) {
        sum = b[i];
        for (j = i + 1; j <= n; j++)
            sum -= a[i][j] * b[j];
        b[i] = sum / a[i][i];
    }
}

double gammq(double a, double x)
{
    void gcf(double *gammcf, double a, double x, double *gln);
    void gser(double *gamser, double a, double x, double *gln);
    void nrerror(const char* error_text);
    double gamser, gammcf, gln;

    if (x < 0.0 || a <= 0.0)
        nrerror("Invalid arguments in routine gammq");
    if (x < (a + 1.0)) {
        gser(&gamser, a, x, &gln);
        return 1.0 - gamser;
    } else {
        gcf(&gammcf, a, x, &gln);
        return gammcf;
    }
}

double gammln(double xx)
{
    double x, y, tmp, ser;
    static double cof[6] = { 76.18009172947146, -86.50532032941677,
        24.01409824083091, -1.231739572450155,
        0.1208650973866179e-2, -0.5395239384953e-5
    };
    int j;

    y = x = xx;
    tmp = x + 5.5;
    tmp -= (x + 0.5) * log(tmp);
    ser = 1.000000000190015;
    for (j = 0; j <= 5; j++)
        ser += cof[j] / ++y;
    return -tmp + log(2.5066282746310005 * ser / x);
}

#define ITMAX 100
#define EPS 3.0e-7

void gser(double *gamser, double a, double x, double *gln)
{
    double gammln(double xx);
    void nrerror(const char* error_text);
    int n;
    double sum, del, ap;

    *gln = gammln(a);
    if (x <= 0.0) {
        if (x < 0.0)
            nrerror("x less than 0 in routine gser");
        *gamser = 0.0;
        return;
    } else {
        ap = a;
        del = sum = 1.0 / a;
        for (n = 1; n <= ITMAX; n++) {
            ++ap;
            del *= x / ap;
            sum += del;
            if (fabs(del) < fabs(sum) * EPS) {
                *gamser = sum * exp(-x + a * log(x) - (*gln));
                return;
            }
        }
        nrerror("a too large, ITMAX too small in routine gser");
        return;
    }
}

#undef ITMAX
#undef EPS

#define ITMAX 100
#define EPS 3.0e-7
#define FPMIN 1.0e-30

void gcf(double *gammcf, double a, double x, double *gln)
{
    double gammln(double xx);
    void nrerror(const char* error_text);
    int i;
    double an, b, c, d, del, h;

    *gln = gammln(a);
    b = x + 1.0 - a;
    c = 1.0 / FPMIN;
    d = 1.0 / b;
    h = d;
    for (i = 1; i <= ITMAX; i++) {
        an = -i * (i - a);
        b += 2.0;
        d = an * d + b;
        if (fabs(d) < FPMIN)
            d = FPMIN;
        c = b + an / c;
        if (fabs(c) < FPMIN)
            c = FPMIN;
        d = 1.0 / d;
        del = d * c;
        h *= del;
        if (fabs(del - 1.0) < EPS)
            break;
    }
    if (i > ITMAX)
        nrerror("a too large, ITMAX too small in gcf");
    *gammcf = exp(-x + a * log(x) - (*gln)) * h;
}

#undef ITMAX
#undef EPS
#undef FPMIN

void fit(double x[], double y[], int ndata, double sig[], int mwt, double *a,
         double *b, double *siga, double *sigb, double *chi2, double *q)
{
    double gammq(double a, double x);
    int i;
    double wt, t, sxoss, sx = 0.0, sy = 0.0, st2 = 0.0, ss, sigdat;

    *b = 0.0;
    if (mwt) {
        ss = 0.0;
        for (i = 1; i <= ndata; i++) {
            wt = 1.0 / SQR(sig[i]);
            ss += wt;
            sx += x[i] * wt;
            sy += y[i] * wt;
        }
    } else {
        for (i = 1; i <= ndata; i++) {
            sx += x[i];
            sy += y[i];
        }
        ss = ndata;
    }
    sxoss = sx / ss;
    if (mwt) {
        for (i = 1; i <= ndata; i++) {
            t = (x[i] - sxoss) / sig[i];
            st2 += t * t;
            *b += t * y[i] / sig[i];
        }
    } else {
        for (i = 1; i <= ndata; i++) {
            t = x[i] - sxoss;
            st2 += t * t;
            *b += t * y[i];
        }
    }
    *b /= st2;
    *a = (sy - sx * (*b)) / ss;
    *siga = sqrt((1.0 + sx * sx / (ss * st2)) / ss);
    *sigb = sqrt(1.0 / st2);
    *chi2 = 0.0;
    if (mwt == 0) {
        for (i = 1; i <= ndata; i++)
            *chi2 += SQR(y[i] - (*a) - (*b) * x[i]);
        *q = 1.0;
        sigdat = sqrt((*chi2) / (ndata - 2));
        *siga *= sigdat;
        *sigb *= sigdat;
    } else {
        for (i = 1; i <= ndata; i++)
            *chi2 += SQR((y[i] - (*a) - (*b) * x[i]) / sig[i]);
        *q = gammq(0.5 * (ndata - 2), 0.5 * (*chi2));
    }
}

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr

void fourn(double data[], unsigned int nn[], int ndim, int isign)
{
    int idim;
    unsigned long i1, i2, i3, i2rev, i3rev, ip1, ip2, ip3, ifp1, ifp2;
    unsigned long ibit, k1, k2, n, nprev, nrem, ntot;
    double tempi, tempr;
    double theta, wi, wpi, wpr, wr, wtemp;

    for (ntot = 1, idim = 1; idim <= ndim; idim++)
        ntot *= nn[idim];
    nprev = 1;
    for (idim = ndim; idim >= 1; idim--) {
        n = nn[idim];
        nrem = ntot / (n * nprev);
        ip1 = nprev << 1;
        ip2 = ip1 * n;
        ip3 = ip2 * nrem;
        i2rev = 1;
        for (i2 = 1; i2 <= ip2; i2 += ip1) {
            if (i2 < i2rev) {
                for (i1 = i2; i1 <= i2 + ip1 - 2; i1 += 2) {
                    for (i3 = i1; i3 <= ip3; i3 += ip2) {
                        i3rev = i2rev + i3 - i2;
                        SWAP(data[i3], data[i3rev]);
                        SWAP(data[i3 + 1], data[i3rev + 1]);
                    }
                }
            }
            ibit = ip2 >> 1;
            while (ibit >= ip1 && i2rev > ibit) {
                i2rev -= ibit;
                ibit >>= 1;
            }
            i2rev += ibit;
        }
        ifp1 = ip1;
        while (ifp1 < ip2) {
            ifp2 = ifp1 << 1;
            theta = isign * 6.28318530717959 / (ifp2 / ip1);
            wtemp = sin(0.5 * theta);
            wpr = -2.0 * wtemp * wtemp;
            wpi = sin(theta);
            wr = 1.0;
            wi = 0.0;
            for (i3 = 1; i3 <= ifp1; i3 += ip1) {
                for (i1 = i3; i1 <= i3 + ip1 - 2; i1 += 2) {
                    for (i2 = i1; i2 <= ip3; i2 += ifp2) {
                        k1 = i2;
                        k2 = k1 + ifp1;
                        tempr = (double) wr *data[k2] - (double) wi *data[k2 + 1];
                        tempi = (double) wr *data[k2 + 1] + (double) wi *data[k2];
                        data[k2] = data[k1] - tempr;
                        data[k2 + 1] = data[k1 + 1] - tempi;
                        data[k1] += tempr;
                        data[k1 + 1] += tempi;
                    }
                }
                wr = (wtemp = wr) * wpr - wi * wpi + wr;
                wi = wi * wpr + wtemp * wpi + wi;
            }
            ifp1 = ifp2;
        }
        nprev *= n;
    }
}

#undef SWAP

/*
#define EPS 1.0e-6
#define JMAX 30
*/
#define EPS 1.0e-5
#define JMAX 20

double qsimp(double (*func) (double), double a, double b)
{
    double trapzd(double (*func) (double), double a, double b, int n, double s);
    void nrerror(const char* error_text);
    int j;
    double s, st, ost, os;

    ost = os = -1.0e30;
    for (j = 1; j <= JMAX; j++) {
        st = trapzd(func, a, b, j, st);
        s = (4.0 * st - ost) / 3.0;
/*
		if (fabs(s-os) <= EPS*fabs(os)) return s;
*/
        if ((fabs(s - os) <= EPS * fabs(os) || j == JMAX) && (fabs(s) > 0.0 || j >= 8))
            return s;
        os = s;
        ost = st;
    }
    nrerror("Too many steps in routine qsimp");
    return 0.0;
}

#undef EPS
#undef JMAX

#define FUNC(x) ((*func)(x))

double trapzd(double (*func) (double), double a, double b, int n, double s)
{
    double x, tnm, sum, del;
    int it, j;

    if (n == 1) {
        return (s = 0.5 * (b - a) * (FUNC(a) + FUNC(b)));
    } else {
        for (it = 1, j = 1; j < n - 1; j++)
            it <<= 1;
        tnm = it;
        del = (b - a) / tnm;
        x = a + 0.5 * del;
        for (sum = 0.0, j = 1; j <= it; j++, x += del)
            sum += FUNC(x);
        s = 0.5 * (s + (b - a) * sum / tnm);
        return s;
    }
}

#undef FUNC

#define EPS 3.0e-11

void gauleg(double x1, double x2, double x[], double w[], int n)
{
    int m, j, i;
    double z1, z, xm, xl, pp, p3, p2, p1;

    m = (n + 1) / 2;
    xm = 0.5 * (x2 + x1);
    xl = 0.5 * (x2 - x1);
    for (i = 1; i <= m; i++) {
        z = cos(3.141592654 * (i - 0.25) / (n + 0.5));
        do {
            p1 = 1.0;
            p2 = 0.0;
            for (j = 1; j <= n; j++) {
                p3 = p2;
                p2 = p1;
                p1 = ((2.0 * j - 1.0) * z * p2 - (j - 1.0) * p3) / j;
            }
            pp = n * (z * p1 - p2) / (z * z - 1.0);
            z1 = z;
            z = z1 - p1 / pp;
        } while (fabs(z - z1) > EPS);
        x[i] = xm - xl * z;
        x[n + 1 - i] = xm + xl * z;
        w[i] = 2.0 * xl / ((1.0 - z * z) * pp * pp);
        w[n + 1 - i] = w[i];
    }
}

#undef EPS

void mrqmin(double x[], double y[], double sig[], int ndata, double a[], int ia[],
            int ma, double **covar, double **alpha, double *chisq,
            void (*funcs) (double, double[], double *, double[], int), double *alamda)
{
    void covsrt(double **covar, int ma, int ia[], int mfit);
    void gaussj(double **a, int n, double **b, int m);
    void mrqcof(double x[], double y[], double sig[], int ndata, double a[],
                int ia[], int ma, double **alpha, double beta[], double *chisq,
                void (*funcs) (double, double[], double *, double[], int));
    int j, k, l, m;
    static int mfit;
    static double ochisq, *atry, *beta, *da, **oneda;

    if (*alamda < 0.0) {
        atry = dvector(1, ma);
        beta = dvector(1, ma);
        da = dvector(1, ma);
        for (mfit = 0, j = 1; j <= ma; j++)
            if (ia[j])
                mfit++;
        oneda = dmatrix(1, mfit, 1, 1);
        *alamda = 0.001;
        mrqcof(x, y, sig, ndata, a, ia, ma, alpha, beta, chisq, funcs);
        ochisq = (*chisq);
        for (j = 1; j <= ma; j++)
            atry[j] = a[j];
    }
    for (j = 0, l = 1; l <= ma; l++) {
        if (ia[l]) {
            for (j++, k = 0, m = 1; m <= ma; m++) {
                if (ia[m]) {
                    k++;
                    covar[j][k] = alpha[j][k];
                }
            }
            covar[j][j] = alpha[j][j] * (1.0 + (*alamda));
            oneda[j][1] = beta[j];
        }
    }
    gaussj(covar, mfit, oneda, 1);
    for (j = 1; j <= mfit; j++)
        da[j] = oneda[j][1];
    if (*alamda == 0.0) {
        covsrt(covar, ma, ia, mfit);
        free_dmatrix(oneda, 1, mfit, 1, 1);
        free_dvector(da, 1, ma);
        free_dvector(beta, 1, ma);
        free_dvector(atry, 1, ma);
        return;
    }
    for (j = 0, l = 1; l <= ma; l++)
        if (ia[l])
            atry[l] = a[l] + da[++j];
    mrqcof(x, y, sig, ndata, atry, ia, ma, covar, da, chisq, funcs);
    if (*chisq < ochisq) {
        *alamda *= 0.1;
        ochisq = (*chisq);
        for (j = 0, l = 1; l <= ma; l++) {
            if (ia[l]) {
                for (j++, k = 0, m = 1; m <= ma; m++) {
                    if (ia[m]) {
                        k++;
                        alpha[j][k] = covar[j][k];
                    }
                }
                beta[j] = da[j];
                a[l] = atry[l];
            }
        }
    } else {
        *alamda *= 10.0;
        *chisq = ochisq;
    }
}

#define SWAP(a,b) {swap=(a);(a)=(b);(b)=swap;}

void covsrt(double **covar, int ma, int ia[], int mfit)
{
    int i, j, k;
    double swap;

    for (i = mfit + 1; i <= ma; i++)
        for (j = 1; j <= i; j++)
            covar[i][j] = covar[j][i] = 0.0;
    k = mfit;
    for (j = ma; j >= 1; j--) {
        if (ia[j]) {
            for (i = 1; i <= ma; i++)
                SWAP(covar[i][k], covar[i][j])
                    for (i = 1; i <= ma; i++)
                    SWAP(covar[k][i], covar[j][i])
                        k--;
        }
    }
}

#undef SWAP

#define SWAP(a,b) {temp=(a);(a)=(b);(b)=temp;}

void gaussj(double **a, int n, double **b, int m)
{
    int *indxc, *indxr, *ipiv;
    int i, icol, irow, j, k, l, ll;
    double big, dum, pivinv, temp;

    indxc = ivector(1, n);
    indxr = ivector(1, n);
    ipiv = ivector(1, n);
    for (j = 1; j <= n; j++)
        ipiv[j] = 0;
    for (i = 1; i <= n; i++) {
        big = 0.0;
        for (j = 1; j <= n; j++)
            if (ipiv[j] != 1)
                for (k = 1; k <= n; k++) {
                    if (ipiv[k] == 0) {
                        if (fabs(a[j][k]) >= big) {
                            big = fabs(a[j][k]);
                            irow = j;
                            icol = k;
                        }
                    } else if (ipiv[k] > 1)
                        nrerror("gaussj: Singular Matrix-1");
                }
        ++(ipiv[icol]);
        if (irow != icol) {
            for (l = 1; l <= n; l++)
                SWAP(a[irow][l], a[icol][l])
                    for (l = 1; l <= m; l++)
                    SWAP(b[irow][l], b[icol][l])
                    }
                    indxr[i] = irow;
            indxc[i] = icol;
            if (a[icol][icol] == 0.0)
                nrerror("gaussj: Singular Matrix-2");
            pivinv = 1.0 / a[icol][icol];
            a[icol][icol] = 1.0;
            for (l = 1; l <= n; l++)
                a[icol][l] *= pivinv;
            for (l = 1; l <= m; l++)
                b[icol][l] *= pivinv;
            for (ll = 1; ll <= n; ll++)
                if (ll != icol) {
                    dum = a[ll][icol];
                    a[ll][icol] = 0.0;
                    for (l = 1; l <= n; l++)
                        a[ll][l] -= a[icol][l] * dum;
                    for (l = 1; l <= m; l++)
                        b[ll][l] -= b[icol][l] * dum;
                }
        }
        for (l = n; l >= 1; l--) {
            if (indxr[l] != indxc[l])
                for (k = 1; k <= n; k++)
                    SWAP(a[k][indxr[l]], a[k][indxc[l]]);
        }
        free_ivector(ipiv, 1, n);
        free_ivector(indxr, 1, n);
        free_ivector(indxc, 1, n);
    }
#undef SWAP

    void mrqcof(double x[], double y[], double sig[], int ndata, double a[], int ia[],
                int ma, double **alpha, double beta[], double *chisq,
                void (*funcs) (double, double[], double *, double[], int)) {
        int i, j, k, l, m, mfit = 0;
        double ymod, wt, sig2i, dy, *dyda;

        dyda = dvector(1, ma);
        for (j = 1; j <= ma; j++)
            if (ia[j])
                mfit++;
        for (j = 1; j <= mfit; j++) {
            for (k = 1; k <= j; k++)
                alpha[j][k] = 0.0;
            beta[j] = 0.0;
        }
        *chisq = 0.0;
        for (i = 1; i <= ndata; i++) {
            (*funcs) (x[i], a, &ymod, dyda, ma);
            sig2i = 1.0 / (sig[i] * sig[i]);
            dy = y[i] - ymod;
            for (j = 0, l = 1; l <= ma; l++) {
                if (ia[l]) {
                    wt = dyda[l] * sig2i;
                    for (j++, k = 0, m = 1; m <= l; m++)
                        if (ia[m])
                            alpha[j][++k] += wt * dyda[m];
                    beta[j] += dy * wt;
                }
            }
            *chisq += dy * dy * sig2i;
        }
        for (j = 2; j <= mfit; j++)
            for (k = 1; k < j; k++)
                alpha[k][j] = alpha[j][k];
        free_dvector(dyda, 1, ma);
    }

    void lfit(double x[], double y[], double sig[], int ndat, double a[], int ia[],
              int ma, double **covar, double *chisq, void (*funcs) (double, double[], int)) {
        void covsrt(double **covar, int ma, int ia[], int mfit);
        void gaussj(double **a, int n, double **b, int m);
        int i, j, k, l, m, mfit = 0;
        double ym, wt, sum, sig2i, **beta, *afunc;

        beta = dmatrix(1, ma, 1, 1);
        afunc = dvector(1, ma);
        for (j = 1; j <= ma; j++)
            if (ia[j])
                mfit++;
        if (mfit == 0)
            nrerror("lfit: no parameters to be fitted");
        for (j = 1; j <= mfit; j++) {
            for (k = 1; k <= mfit; k++)
                covar[j][k] = 0.0;
            beta[j][1] = 0.0;
        }
        for (i = 1; i <= ndat; i++) {
            (*funcs) (x[i], afunc, ma);
            ym = y[i];
            if (mfit < ma) {
                for (j = 1; j <= ma; j++)
                    if (!ia[j])
                        ym -= a[j] * afunc[j];
            }
            sig2i = 1.0 / SQR(sig[i]);
            for (j = 0, l = 1; l <= ma; l++) {
                if (ia[l]) {
                    wt = afunc[l] * sig2i;
                    for (j++, k = 0, m = 1; m <= l; m++)
                        if (ia[m])
                            covar[j][++k] += wt * afunc[m];
                    beta[j][1] += ym * wt;
                }
            }
        }
        for (j = 2; j <= mfit; j++)
            for (k = 1; k < j; k++)
                covar[k][j] = covar[j][k];
        gaussj(covar, mfit, beta, 1);
        for (j = 0, l = 1; l <= ma; l++)
            if (ia[l])
                a[l] = beta[++j][1];
        *chisq = 0.0;
        for (i = 1; i <= ndat; i++) {
            (*funcs) (x[i], afunc, ma);
            for (sum = 0.0, j = 1; j <= ma; j++)
                sum += a[j] * afunc[j];
            *chisq += SQR((y[i] - sum) / sig[i]);
        }
        covsrt(covar, ma, ia, mfit);
        free_dvector(afunc, 1, ma);
        free_dmatrix(beta, 1, ma, 1, 1);
    }
