#include "bob.h"
#ifdef ENABLE_OPENMP
#include <omp.h>
#endif
#include <stdio.h>
#include <string>

void omp_set_schedule_str(char *str, int chunksize) {
    #ifdef ENABLE_OPENMP 
    if (str == NULL) {
        printf("Warning: OMP_SCHEDULE not set, using default omp_sched_auto\n");
        omp_set_schedule(omp_sched_auto, chunksize);
    }
    else if (!strcasecmp(str, "guided")) {
        omp_set_schedule(omp_sched_guided, chunksize);
    }
    else if (!strcasecmp(str, "dynamic")) {
        omp_set_schedule(omp_sched_dynamic, chunksize);
    }
    else if (!strcasecmp(str, "static")) {
        omp_set_schedule(omp_sched_static, chunksize);
    }
    else if (!strcasecmp(str, "auto")) {
        omp_set_schedule(omp_sched_auto, chunksize);
    }
    else {
        fprintf(stderr,  "Invalid schedule type %s. Must be auto, static, dynamic, or guided.\n", str);
    }
    #endif
}

void print_omp_settings() {
#ifdef ENABLE_OPENMP
    int chunksize, nthreads;
    omp_sched_t kind;

    omp_get_schedule( &kind, &chunksize);
#pragma omp parallel 
    {
#pragma omp master
        nthreads = omp_get_num_threads();
    }
    
    printf("Running %d thread(s) with ", nthreads);

    
    switch(kind) {
    case omp_sched_auto:
        printf("automatic scheduling, ");
        break;
    case omp_sched_guided:
        printf("guided scheduling, ");
        break;
    case omp_sched_dynamic:
        printf("dynamic scheduling, ");
        break;
    case omp_sched_static:
        printf("static scheduling, ");
        break;
    }

    printf("chunksize of %d, ", chunksize);

#ifdef __INTEL_COMPILER
    int blocktime = kmp_get_blocktime();
    printf("and blocktime of %d\n", blocktime);
#else
    char *spincount =   getenv("GOMP_SPINCOUNT");
    char *wait_policy = getenv("OMP_WAIT_POLICY");
    printf("wait policy of %s, and spincount of %s\n", wait_policy, spincount);
#endif

#endif
}

void set_omp_settings(system_parameters *parameters) {
#ifdef ENABLE_OPENMP
#ifdef __INTEL_COMPILER
    kmp_set_blocktime(parameters->kmp_blocktime);
#else
    char spincount[256];
    char varname[] = "GOMP_SPINCOUNT";
    sprintf(spincount, "%d", parameters->gomp_spincount); 
    int flag = setenv(varname, spincount, 1);
    if (flag!=0) {
        printf("WARNING: Failed to set GOMP_SPINCOUNT2\n");
    }
    char wait_policy_name[] = "OMP_WAIT_POLICY";
    char wait_policy[] = "ACTIVE";
    flag = setenv(wait_policy_name, wait_policy, 1);
    if (flag!=0) {
        printf("WARNING: Failed to set OMP_WAIT_POLICY\n");
    }
#endif

    omp_set_num_threads(parameters->omp_num_threads);
    omp_set_schedule_str(parameters->omp_schedule, parameters->omp_chunksize);
#endif    
}
