/* This routine optimizes the neighbor list skin for a molecular dynamics simulation by maximizing
   the MD efficiency, defined as the simulation time per cpu second.

   input: optimization interval (n_opt)
          integration timestep (delta)
          pointer to neighbor list skin (skin)
          pointer to previous neighbor list skin (skin_old)
          elapsed cpu time since last calculation of displacement efficiency (elapsed_time)
          pointer to displacement efficiency (md_efficiency)

   output: skin, skin_old, and md_efficiency are modified on return. */

#include "bob.h"

#define SCALE_FACTOR 1.1

void optimize_skin(int n_opt, double delta, double *skin, double *skin_old,
                   double elapsed_time, double *md_efficiency)
{
    double md_efficiency_old, deriv;

    /* Save old MD efficiency. */
    md_efficiency_old = *md_efficiency;

    /* Calculate new MD efficiency. */
    *md_efficiency = dynamics_efficiency(n_opt, delta, elapsed_time);

    fprintf(stdout, "   skin_old = %g, md_efficiency_old = %g / cpu s\n", *skin_old,
            md_efficiency_old);
    fprintf(stdout, "   skin = %g, md_efficiency = %g / cpu s\n", *skin, *md_efficiency);

    /* Estimate derivative of efficiency with respect to neighbor list skin. */
    if (*skin == *skin_old)
        deriv = 0.0;
    else
        deriv = (*md_efficiency - md_efficiency_old) / (*skin - *skin_old);

    /* Adjust maximum displacement. */
    *skin_old = *skin;
    if (deriv < 0.0)
        *skin /= SCALE_FACTOR;
    else
        *skin *= SCALE_FACTOR;

    fprintf(stdout, "   new skin = %g\n\n", *skin);
    
    return;
}

#undef SCALE_FACTOR
