/* This routine computes the separation vector between two objects, in any number of dimensions,
 and for any type of boundary condition (periodic, free, or mixed).
 
 Input: number of spatial dimensions (n_dim)
 number of periodic spatial dimensions (n_periodic)
 unit cell matrix (h)
 real coordinate of first object (r_1)
 scaled coordinate of first object (s_1)
 real coordinate of second object (r_2)
 scaled coordinate of second object (s_2)
 
 Output: pair separation vector (dr) */

#include "bob.h"

void pair_separation(int n_dim, int n_periodic, double **h,
                     double *r_1, double *s_1, double *r_2, double *s_2, double *dr)
{
    int i, j;
    static int first_call = 1;
    static double *ds;

    /* Allocate memory for local arrays the first time the routine is called. */
    if (first_call) {
        ds = (double *) allocate_1d_array(n_dim, sizeof(double));
        first_call = 0;
    }

    /* Compute pair separation vector. */
    for (i = 0; i < n_periodic; ++i) {  /* First handle periodic subspace. */
        ds[i] = s_2[i] - s_1[i];
        ds[i] -= NINT(ds[i]);
    }
    for (i = 0; i < n_periodic; ++i) {
        dr[i] = 0.0;
        for (j = 0; j < n_periodic; ++j)
            dr[i] += h[i][j] * ds[j];
    }
    for (i = n_periodic; i < n_dim; ++i)        /* Then handle free subspace. */
        dr[i] = r_2[i] - r_1[i];

    return;
}
