/* This routine distributes a mixture of spheres and spherocylinders at random within an n_dim-dimensional
   parallelepiped, with periodic boundary conditions, avoiding close contacts.

   input: number of spatial dimensions (n_dim)
   number of spatial dimensions that are (n_periodic)
   unit cell matrix (h)
   inverse unit cell matrix (h_inv)
   number of spherocylinders (n_bonds)
   spherocylinder diameter (diameter)
   maximum number of trial insertions (max_trials)
   pointer to gsl rng state for random number generator (r_rng)
   array of real site positions (r)
   array of scaled site positions (s)
   array of real bond positions (r_bond)
   array of scaled bond positions (s_bond)
   array of bond directors (u_bond)
   array of bond lengths (length)
   array of labels of bond endpoints (site_1, site_2)

   output: r, s, r_bond, s_bond, and u_bond are modified on output */

#include "bob.h"

// Function to check if the an inserted spherocylinder is
// inside or outside hard wall boundaries.
bool check_non_periodic_insertion(int n_dim, int n_periodic, double **h_inv,
                                  double *r, double *u, double length){

    for ( int i = n_periodic; i < n_dim; ++i) {
        // Scaled positions of plus and minus tips of spherocylinders in unit cell
        double s_plus = 0;
        double s_minus = 0;

        // Get positions of plus and minus tips in scaled periodic coord_i
        for ( int j = n_periodic; j < n_dim; ++j){
            s_plus += h_inv[i][j] * ( r[j] + u[j]*.5*length);
            s_minus += h_inv[i][j] * ( r[j] - u[j]*.5*length);
        }

        // If outside scaled position try again
        if ( (ABS(s_plus) >= .5) || (ABS(s_minus) >= .5) ){
            return true;
        }
    }
    return false;
}


// Check to see if rod was inserted inside the droplet radius
// This will eventually be made more complex
bool check_droplet_insertion(system_parameters *parameters, int n_dim, double *r){

    double drop_r2 = SQR(parameters->droplet_rad);
    double pos_r2 = dot_product(n_dim, r, r);

    if (pos_r2 > drop_r2)
        return true;

    return false;

}

void parallelepiped_insertion_sphero_periodic(
        system_parameters *parameters, int n_dim, int n_periodic,
        double **h, double **h_inv, int n_bonds, double diameter,
        int max_trials, gsl_rng *r_rng, double **r, double **s,
        double **r_bond, double **s_bond, double **u_bond, double *length,
        int *site_1, int *site_2
        ) {

    int i_bond, j_bond, i1_site, i2_site, i, j, n_trials, successful, overlap;
    double length_i, length_j;
    double r_i[3], s_i[3], u_i[3], r_j[3], s_j[3], u_j[3];

    /* Loop over bonds to insert. */
    for (i_bond = 0; i_bond < n_bonds; ++i_bond) {

        /* Get bond length and labels of sites in bond. */
        length_i = length[i_bond];
        i1_site = site_1[i_bond];
        i2_site = site_2[i_bond];

        /* Loop over trial insertions. */
        n_trials = 0;
        do {

            /* Increment trial counter and set flag. */
            ++n_trials;
            successful = 1;

            /* Generate random spherocylinder position within unit hypercube (scaled coordinates). */
            for (i = 0; i < n_dim; ++i)
                s_i[i] = gsl_rng_uniform(r_rng) - 0.5;

            /* Calculate spherocylinder position in real coordinates. */
            for (i = 0; i < n_dim; ++i) {
                r_i[i] = 0.0;
                for (j = 0; j < n_dim; ++j)
                    r_i[i] += h[i][j] * s_i[j];
            }

            /* Generate a random orientation. */
            generate_random_unit_vector(n_dim, u_i, r_rng);

            bool outside = check_non_periodic_insertion( n_dim, n_periodic, h_inv, r_i, u_i, length_i);

            if (!outside && parameters->droplet_flag)
                outside = check_droplet_insertion(parameters, n_dim, r_i);

            // Check if spherocylinder outside non-periodic boundary dimensions or droplet radius.
            // Do not count this against the trials.
            if (outside) {
                successful = 0;
                --n_trials;
                continue;
            }

            /* Loop over previously inserted spherocylinders to test for close contacts. */
            for (j_bond = 0; j_bond < i_bond; ++j_bond) {

                /* Get bond position, orientation, and length, and labels of sites in bond. */
                for (i = 0; i < n_dim; ++i) {
                    r_j[i] = r_bond[j_bond][i];
                    s_j[i] = s_bond[j_bond][i];
                    u_j[i] = u_bond[j_bond][i];
                }
                length_j = length[j_bond];


                /* Check for overlap between pair of spherocylinders. */
                overlap = sphero_overlap(n_dim, n_dim, h, 0.0, diameter,
                                         r_i, s_i, u_i, length_i,
                                         r_j, s_j, u_j, length_j);

                /* If a close contact is found, set flag to zero and break from loop. */
                if (overlap) {
                    successful = 0;
                    break;
                }
            }
        } while(!successful && n_trials < max_trials);

        printf("i_bond = %d, n_trials = %d\n", i_bond, n_trials);
        fflush(NULL);

        /* Exit if a successful insertion was not made after max_trials trials. */
        if (!successful)
            error_exit("Maximum number of trial insertions exceeded in parallelepiped_insertion_sphero_periodic");

        /* Store bond position and orientation and positions of bond endpoints. */
        for (i = 0; i < n_dim; ++i) {
            r_bond[i_bond][i] = r_i[i];
            s_bond[i_bond][i] = s_i[i];
            u_bond[i_bond][i] = u_i[i];
            r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
            r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
        }

        /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);
    }

    return;
}

void parallelepiped_bundle_insertion_sphero_periodic(system_parameters *parameters, int n_dim, int n_periodic, double **h, double **h_inv, int n_bonds,
                                                     double diameter,
                                                     int max_trials, gsl_rng *r_rng,
                                                     double **r, double **s,
                                                     double **r_bond, double **s_bond, double **u_bond,
                                                     double *length,
                                                     int *site_1, int *site_2) {
    int i_bond, i1_site, i2_site, i;
    double length_i;
    double r_i[3], s_i[3], u_i[3], dr[3];

    if (n_dim == 3) {
        r_bond[0][0] = 0.0;
        r_bond[0][1] = 0.0;
        r_bond[0][2] = 10;
        u_bond[0][0] = 0.0;
        u_bond[0][1] = 0.0;
        u_bond[0][2] = 1.0;

        i1_site = site_1[0];
        i2_site = site_2[0];

        for (i = 0; i < n_dim; ++i) {
            r[i1_site][i] = r_bond[0][i] - 0.5 * length[0] * u_bond[0][i];
            r[i2_site][i] = r_bond[0][i] + 0.5 * length[0] * u_bond[0][i];
        }
        periodic_boundary_conditions_single(n_dim, h, h_inv, r_bond[0], s_bond[0]);

        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);
    }
    else {
        fprintf(stderr, "bundle builder only supported in 3D\n");
        exit(1);
    }

    double lattice_spacing = 1.2 * diameter;
    int counter = 1;
    /* Loop over bonds to insert. */
    u_i[0] = 0.0;
    u_i[1] = 0.0;
    u_i[2] = 1.0;

    for (i_bond = 1; i_bond < n_bonds; ++i_bond) {
        dr[0] = counter * lattice_spacing;
        dr[1] = 0.0;
        dr[2] = 0.0;
        int i_step;

        for (i_step = 0; i_step < counter; ++i_step) {
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];

            dr[0] -= 0.5 * lattice_spacing;
            dr[1] += 0.5 * sqrt(3.0) * lattice_spacing;
            for (i = 0; i < n_dim; ++i)
                r_i[i] = r_bond[0][i] + dr[i];

            periodic_boundary_conditions_single(n_dim, h, h_inv, r_i, s_i);

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);

            i_bond += 1;
            if (i_bond == n_bonds)
                return;
        }
        for (i_step = 0; i_step < counter; ++i_step) {
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];

            dr[0] -= lattice_spacing;
            for (i = 0; i < n_dim; ++i)
                r_i[i] = r_bond[0][i] + dr[i];

            periodic_boundary_conditions_single(n_dim, h, h_inv, r_i, s_i);

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);

            i_bond += 1;
            if (i_bond == n_bonds)
                return;
        }
        for (i_step = 0; i_step < counter; ++i_step) {
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];

            dr[0] -= 0.5 * lattice_spacing;
            dr[1] -= 0.5 * sqrt(3.0) * lattice_spacing;
            for (i = 0; i < n_dim; ++i)
                r_i[i] = r_bond[0][i] + dr[i];

            periodic_boundary_conditions_single(n_dim, h, h_inv, r_i, s_i);

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);

            i_bond += 1;
            if (i_bond == n_bonds)
                return;
        }
        for (i_step = 0; i_step < counter; ++i_step) {
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];

            dr[0] += 0.5 * lattice_spacing;
            dr[1] -= 0.5 * sqrt(3.0) * lattice_spacing;
            for (i = 0; i < n_dim; ++i)
                r_i[i] = r_bond[0][i] + dr[i];

            periodic_boundary_conditions_single(n_dim, h, h_inv, r_i, s_i);

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);

            i_bond += 1;
            if (i_bond == n_bonds)
                return;
        }
        for (i_step = 0; i_step < counter; ++i_step) {
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];

            dr[0] += lattice_spacing;
            for (i = 0; i < n_dim; ++i)
                r_i[i] = r_bond[0][i] + dr[i];

            periodic_boundary_conditions_single(n_dim, h, h_inv, r_i, s_i);

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);

            i_bond += 1;
            if (i_bond == n_bonds)
                return;
        }
        for (i_step = 0; i_step < counter; ++i_step) {
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];

            dr[0] += 0.5 * lattice_spacing;
            dr[1] += 0.5 * sqrt(3.0) * lattice_spacing;
            for (i = 0; i < n_dim; ++i)
                r_i[i] = r_bond[0][i] + dr[i];

            periodic_boundary_conditions_single(n_dim, h, h_inv, r_i, s_i);

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);

            i_bond++;
            if (i_bond == n_bonds)
                return;
        }


        counter += 1;
    }

    return;
}


/* This routine distributes a mixture of spheres and spherocylinders at random within an n_dim-dimensional
   parallelepiped, with periodic boundary conditions, avoiding close contacts.

   input: number of spatial dimensions (n_dim)
   number of spatial dimensions that are (n_periodic)
   unit cell matrix (h)
   inverse unit cell matrix (h_inv)
   number of spherocylinders (n_bonds)
   spherocylinder diameter (diameter)
   maximum number of trial insertions (max_trials)
   pointer to gsl rng state for random number generator (r_rng)
   array of real site positions (r)
   array of scaled site positions (s)
   array of real bond positions (r_bond)
   array of scaled bond positions (s_bond)
   array of bond directors (u_bond)
   array of bond lengths (length)
   array of labels of bond endpoints (site_1, site_2)

   output: r, s, r_bond, s_bond, and u_bond are modified on output */

void parallelepiped_nematic_insertion_sphero_periodic(system_parameters *parameters, int n_dim, int n_periodic, double **h, double **h_inv,
                                                      int n_bonds, double diameter,
                                                      int max_trials, gsl_rng *r_rng,
                                                      double **r, double **s,
                                                      double **r_bond, double **s_bond, double **u_bond,
                                                      double *length,
                                                      int *site_1, int *site_2, int sphere_flag) {
    int i_bond, j_bond, i1_site, i2_site, i, j, n_trials, successful, overlap;
    double length_i, length_j, r_mag_i2, r_mag_j2;
    double r_i[3], s_i[3], u_i[3], r_j[3], s_j[3], u_j[3], min_width, r_max2;

    min_width = h[0][0];
    for(i = 0; i < n_dim; ++i)
        min_width = MIN(min_width, h[i][i]);

    r_max2 = SQR(0.5 * min_width);



    /* Loop over bonds to insert. */
    for (i_bond = 0; i_bond < n_bonds; ++i_bond) {

        /* Get bond length and labels of sites in bond. */
        length_i = length[i_bond];
        i1_site = site_1[i_bond];
        i2_site = site_2[i_bond];

        /* Loop over trial insertions. */
        n_trials = 0;
        do {

            /* Increment trial counter and set flag. */
            ++n_trials;
            successful = 1;

            /* Align along the vertical axis */
            for (i = 0; i < n_dim; ++i)
                u_i[i] = 0.0;
            if ((gsl_rng_uniform(r_rng) - 0.5) < 0.0)
                u_i[n_dim - 1] = 1.0;
            else
                u_i[n_dim - 1] = -1.0;

            do {
                /* Generate random spherocylinder position within unit hypercube (scaled coordinates). */
                for (i = 0; i < n_dim ; ++i) {
                    s_i[i] = gsl_rng_uniform(r_rng) - 0.5;
                }

                r_mag_i2 = r_mag_j2 = 0.0;

                /* Calculate spherocylinder position in real coordinates. */
                for (i = 0; i < n_dim; ++i) {
                    r_i[i] = 0.0;
                    for (j = 0; j < n_dim; ++j)
                        r_i[i] += h[i][j] * s_i[j];
                    if (sphere_flag) {
                        r_mag_i2 += SQR(r_i[i] - length[i_bond] * u_i[i]);
                        r_mag_j2 += SQR(r_i[i] + length[i_bond] * u_i[i]);
                    }
                }

            } while (r_mag_i2 > r_max2 || r_mag_j2 > r_max2);

            bool outside = check_non_periodic_insertion( n_dim, n_periodic, h_inv, r_i, u_i, length_i);

            if (!outside && parameters->droplet_flag)
                outside = check_droplet_insertion(parameters, n_dim, r_i);

            // Check if spherocylinder outside non-periodic boundary dimensions
            if (outside) {
                --n_trials;
                successful = 0;
                continue;
            }
            /* Loop over previously inserted spherocylinders to test for close contacts. */
            for (j_bond = 0; j_bond < i_bond; ++j_bond) {

                /* Get bond position, orientation, and length, and labels of sites in bond. */
                for (i = 0; i < n_dim; ++i) {
                    r_j[i] = r_bond[j_bond][i];
                    s_j[i] = s_bond[j_bond][i];
                    u_j[i] = u_bond[j_bond][i];
                }
                length_j = length[j_bond];


                /* Check for overlap between pair of spherocylinders. */
                overlap = sphero_overlap(n_dim, n_dim, h, 0.0, diameter,
                                         r_i, s_i, u_i, length_i,
                                         r_j, s_j, u_j, length_j);

                /* If a close contact is found, set flag to zero and break from loop. */
                if (overlap) {
                    successful = 0;
                    break;
                }
            }
        } while(!successful && n_trials < max_trials);

        printf("i_bond = %d, n_trials = %d\n", i_bond, n_trials);
        fflush(NULL);

        /* Exit if a successful insertion was not made after max_trials trials. */
        if (!successful)
            error_exit("Maximum number of trial insertions exceeded in parallelepiped_insertion_sphero_periodic");

        /* Store bond position and orientation and positions of bond endpoints. */
        for (i = 0; i < n_dim; ++i) {
            r_bond[i_bond][i] = r_i[i];
            s_bond[i_bond][i] = s_i[i];
            u_bond[i_bond][i] = u_i[i];
            r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
            r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
        }

        /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);
    }

    return;
}

/* This routine distributes spherocylinders periodically within an n_dim-dimensional
   parallelepiped, with periodic boundary conditions, avoiding close contacts. Currently aligns
   rods along h[n_dim - 1] unit cell vector and assumes a square box.

   input: number of spatial dimensions (n_dim)
   number of spatial dimensions that are periodic (n_periodic)
   unit cell matrix (h)
   inverse unit cell matrix (h_inv)
   number of spherocylinders (n_bonds)
   spherocylinder diameter (diameter)
   maximum number of trial insertions (max_trials)
   pointer to gsl rng state for random number generator (r_rng)
   array of real site positions (r)
   array of scaled site positions (s)
   array of real bond positions (r_bond)
   array of scaled bond positions (s_bond)
   array of bond directors (u_bond)
   array of bond lengths (length)
   array of labels of bond endpoints (site_1, site_2)

   output: r, s, r_bond, s_bond, and u_bond are modified on output */

void parallelepiped_square_insertion_sphero_periodic(system_parameters *parameters, int n_dim, int n_periodic, double **h, double **h_inv,
                                                     int n_bonds, double diameter,
                                                     int max_trials, gsl_rng *r_rng,
                                                     double **r, double **s,
                                                     double **r_bond, double **s_bond, double **u_bond,
                                                     double *length,
                                                     int *site_1, int *site_2) {
    int i_bond, j_bond, i1_site, i2_site, i, j, n_trials, successful, overlap;
    double length_i, length_j;
    double r_i[3], s_i[3], u_i[3], r_j[3], s_j[3], u_j[3];

    double max_length = 0.0;
    for (i_bond = 0; i_bond < n_bonds; ++i_bond)
        max_length = MAX(max_length, length[i_bond]);

    double com_separation[3];
    double height = sqrt(dot_product(n_dim, h[n_dim - 1], h[n_dim - 1]));
    com_separation[n_dim - 1] = (max_length + 1.05);
    int total_bonds[3];
    total_bonds[n_dim - 1] = (int) (height / (com_separation[n_dim - 1]));

    for (i = 0; i < n_dim - 1; ++i) {
        if (n_dim == 3)
            total_bonds[i] = (int) (sqrt(n_bonds / total_bonds[n_dim - 1])) + 1;
        else
            total_bonds[i] = (int) (n_bonds / total_bonds[n_dim - 1]) + 1;
        com_separation[i] = height / total_bonds[i];
        if (com_separation[i] < 1.0) {
            printf("fluid too dense. to fit in box, try a larger side_length\n");
            exit(1);
        }
    }
    com_separation[n_dim - 1] = height / total_bonds[n_dim - 1];
    int n_holes = 1;
    for (i = 0; i < n_dim; ++i)
        n_holes *= total_bonds[i];
    n_holes = n_holes - n_bonds;

    double com_separation_s[3];
    for (i = 0; i < n_dim; ++i)
        com_separation_s[i] = com_separation[i] / height;

    int n_coord[3] = {0, 0, 0};
    int *skip_bond = (int*) allocate_1d_array(n_holes, sizeof(int));
    int i_hole, j_hole;
    /* Generate random holes when there are more lattice sites than spherocylinders */
    for (i_hole = 0; i_hole < n_holes; ++i_hole) {
        int success = 1;
        do {
            i_bond = (int) (gsl_rng_uniform(r_rng) * n_bonds);
            for (j_hole = 0; j_hole < i_hole; ++j_hole) {
                if (j_hole == i_hole)
                    success = 0;
            }
        } while (!success);
        skip_bond[i_hole] = i_bond;
    }

    /* Loop over bonds to insert. */
    for (i_bond = 0; i_bond < n_bonds; ++i_bond) {
        /* Skip lattice site if necessary */
        for (i_hole = 0; i_hole < n_holes; ++i_hole) {
            if (skip_bond[i_hole] == i_bond)
                n_coord[0] += 1;
        }

        /* Find current lattice site */
        for (i = 0; i < n_dim - 1; ++i) {
            if (n_coord[i] >= total_bonds[i]) {
                n_coord[i] -= total_bonds[i];
                n_coord[i+1] += 1;
            }
        }


        /* Get bond length and labels of sites in bond. */
        length_i = length[i_bond];
        i1_site = site_1[i_bond];
        i2_site = site_2[i_bond];

        /* Loop over trial insertions. */
        n_trials = 0;
        do {

            /* Increment trial counter and set flag. */
            ++n_trials;
            successful = 1;

            /* Generate spherocylinder position within unit hypercube (scaled coordinates). */
            for (i = 0; i < n_dim; ++i) {
                s_i[i] = n_coord[i] * com_separation_s[i] - 0.5;
            }

            /* Calculate spherocylinder position in real coordinates. */
            for (i = 0; i < n_dim; ++i) {
                r_i[i] = 0.0;
                for (j = 0; j < n_dim; ++j)
                    r_i[i] += h[i][j] * s_i[j];
            }


            /* Align along the h[n_dim - 1] axis */
            for (i = 0; i < n_dim - 1; ++i)
                u_i[i] = 0.0;

            if (gsl_rng_uniform(r_rng) > 0.5)
                u_i[n_dim - 1] = 1.0;
            else
                u_i[n_dim - 1] = -1.0;

            bool outside = check_non_periodic_insertion( n_dim, n_periodic, h_inv, r_i, u_i, length_i);

            if (!outside && parameters->droplet_flag)
                outside = check_droplet_insertion(parameters, n_dim, r_i);

            // Check if spherocylinder outside non-periodic boundary dimensions or droplet radius
            if (outside) {
                --n_trials;
                successful = 0;
                continue;
            }

            /* Loop over previously inserted spherocylinders to test for close contacts. */
            for (j_bond = 0; j_bond < i_bond; ++j_bond) {

                /* Get bond position, orientation, and length, and labels of sites in bond. */
                for (i = 0; i < n_dim; ++i) {
                    r_j[i] = r_bond[j_bond][i];
                    s_j[i] = s_bond[j_bond][i];
                    u_j[i] = u_bond[j_bond][i];
                }
                length_j = length[j_bond];


                /* Check for overlap between pair of spherocylinders. */
                overlap = sphero_overlap(n_dim, n_dim, h, 0.0, diameter,
                                         r_i, s_i, u_i, length_i,
                                         r_j, s_j, u_j, length_j);

                /* If a close contact is found, set flag to zero and break from loop. */
                if (overlap) {
                    successful = 0;
                    break;
                }
            }
        } while(!successful && n_trials < max_trials);
        /* move to next lattice site */
	n_coord[0] += 1;

        printf("i_bond = %d, n_trials = %d\n", i_bond, n_trials);
        fflush(NULL);

        /* Exit if a successful insertion was not made after max_trials trials. */
        if (!successful)
            error_exit("Maximum number of trial insertions exceeded in parallelepiped_insertion_sphero_periodic");

        /* Store bond position and orientation and positions of bond endpoints. */
        for (i = 0; i < n_dim; ++i) {
            r_bond[i_bond][i] = r_i[i];
            s_bond[i_bond][i] = s_i[i];
            u_bond[i_bond][i] = u_i[i];
            r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
            r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
        }

        /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
        periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);
    }

    return;
}

/* This routine distributes a mixture of spheres and spherocylinders at random within an n_dim-dimensional
   parallelepiped, with periodic boundary conditions, avoiding close contacts.

   input: number of spatial dimensions (n_dim)

   unit cell matrix (h)
   inverse unit cell matrix (h_inv)
   number of objects (n_objects)
   array of real site positions (r)
   array of scaled site positions (s)
   array of real bond positions (r_bond)
   array of scaled bond positions (s_bond)
   array of bond directors (u_bond)
   array of bond lengths (length)
   array of labels of bond endpoints (site_1, site_2)
   array of object type indices (type_index)
   array of object shape indices (shape_index)	   sphero or spherocylinder
   array of object labels (object_label)           label of object in site or bond lists
   ghost particle array (ghost_particle)           ghost particles don't interact with one another
   array of pair interaction cutoff distances (r_cutoff)

   output: r, s, r_bond, s_bond, and u_bond are modified on output */

void parallelepiped_insertion_sphere_sphero_periodic(int n_dim, double **h, double **h_inv, int n_objects,
                                                     int max_trials, gsl_rng *r_rng,
                                                     double **r, double **s,
                                                     double **r_bond, double **s_bond, double **u_bond, double *length,
                                                     int *site_1, int *site_2,
                                                     int *type_index, int *shape_index,
                                                     int *object_label, int *ghost_particle,
                                                     double **r_cutoff) {
    int i_obj, j_obj, i_type, j_type, i_shape, j_shape, i_site, j_site, i_bond, j_bond,
	i1_site, i2_site, i, j, n_trials, successful, overlap;
    double length_i, length_j, dr2;
    double r_i[3], s_i[3], u_i[3], r_j[3], s_j[3], u_j[3], dr[3];

    length_i = length_j = 0.0;
    /* Loop over objects to insert. */
    for (i_obj = 0; i_obj < n_objects; ++i_obj) {

        /* Get interaction type and shape indices for object i_obj. */
        i_type = type_index[i_obj];
        i_shape = shape_index[i_obj];

        if (i_shape == 0) {   /* Object i_obj is a sphere. */

            /* Get site label. */
            i_site = object_label[i_obj];
        }
        else {   /* Object i_obj is a spherocylinder. */

            /* Get bond label and bond length, and labels of sites in bond. */
            i_bond = object_label[i_obj];
            length_i = length[i_bond];
            i1_site = site_1[i_bond];
            i2_site = site_2[i_bond];
        }

        /* Loop over trial insertions. */
        n_trials = 0;
        do {

            /* Increment trial counter and set flag. */
            ++n_trials;
            successful = 1;

            /* Generate random object position within unit hypercube (scaled coordinates). */
            for (i = 0; i < n_dim; ++i)
                s_i[i] = gsl_rng_uniform(r_rng) - 0.5;

            /* Calculate object position in real coordinates. */
            for (i = 0; i < n_dim; ++i) {
                r_i[i] = 0.0;
                for (j = 0; j < n_dim; ++j)
                    r_i[i] += h[i][j] * s_i[j];
            }

            /* If object is a spherocylinder, generate a random orientation. */
            if (i_shape == 1)
                generate_random_unit_vector(n_dim, u_i, r_rng);

            /* Loop over previously inserted objects to test for close contacts. */
            for (j_obj = 0; j_obj < i_obj; ++j_obj) {

                /* Check for overlaps if at least one of the two objects isn't a ghost particle. */
                if (!ghost_particle[i_obj] || !ghost_particle[j_obj]) {

                    /* Get interaction type and shape indices for object j_obj. */
                    j_type = type_index[j_obj];
                    j_shape = shape_index[j_obj];

                    if (j_shape == 0) {   /* Object j_obj is a sphere. */

                        /* Get site label and position. */
                        j_site = object_label[j_obj];
                        for (i = 0; i < n_dim; ++i) {
                            r_j[i] = r[j_site][i];
                            s_j[i] = s[j_site][i];
                        }
                    }
                    else {   /* Object j_obj is a spherocylinder. */

                        /* Get bond label, position, orientation, and length, and labels of sites in bond. */
                        j_bond = object_label[j_obj];
                        for (i = 0; i < n_dim; ++i) {
                            r_j[i] = r_bond[j_bond][i];
                            s_j[i] = s_bond[j_bond][i];
                            u_j[i] = u_bond[j_bond][i];
                        }
                        length_j = length[j_bond];
                    }

                    /* Check for overlap between objects. */
                    if (i_shape == 0 && j_shape == 0) {   /* Objects i_obj and j_obj are spheres. */

                        /* Check for overlap between spheres. */
                        pair_separation(n_dim, n_dim, h, r_i, s_i, r_j, s_j, dr);
                        dr2 = 0.0;
                        for (i = 0; i < n_dim; ++i)
                            dr2 += SQR(dr[i]);
                        overlap = dr2 < SQR(r_cutoff[i_type][j_type]);
                    }
                    else if (i_shape == 0 && j_shape == 1) {   /* Object i_obj is a sphere and object j_obj is a spherocylinder. */

                        /* Check for overlap between sphere and spherocylinder. */
                        overlap = sphere_sphero_overlap(n_dim, n_dim, h, 0.0, r_cutoff[i_type][j_type],
                                                        r_i, s_i, r_j, s_j, u_j, length_j);
                    }
                    else if (i_shape == 1 && j_shape == 0) {   /* Object i_obj is a spherocylinder and object j_obj is a sphere. */

                        /* Check for overlap between sphere and spherocylinder. */
                        overlap = sphere_sphero_overlap(n_dim, n_dim, h, 0.0, r_cutoff[i_type][j_type],
                                                        r_j, s_j, r_i, s_i, u_i, length_i);
                    }
                    else {   /* Objects i_obj and j_obj are spherocylinders. */

                        /* Check for overlap between pair of spherocylinders. */
                        overlap = sphero_overlap(n_dim, n_dim, h, 0.0, r_cutoff[i_type][j_type],
                                                 r_i, s_i, u_i, length_i,
                                                 r_j, s_j, u_j, length_j);
                    }

                    /* If a close contact is found, set flag to zero and break from loop. */
                    if (overlap) {
                        successful = 0;
                        break;
                    }
                }
            }
        } while(!successful && n_trials < max_trials);

        printf("i_obj = %d, n_trials = %d\n", i_obj, n_trials);
        fflush(NULL);

        /* Exit if a successful insertion was not made after max_trials trials. */
        if (!successful)
            error_exit("Maximum number of trial insertions exceeded in parallelepiped_insertion_sphere_sphero_periodic");

        if (i_shape == 0) {   /* Object i_obj is a sphere. */

            /* Store site position. */
            for (i = 0; i < n_dim; ++i) {
                r[i_site][i] = r_i[i];
                s[i_site][i] = s_i[i];
            }
        }
        else {   /* Object i_obj is a spherocylinder. */

            /* Store bond position and orientation and positions of bond endpoints. */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] = r_i[i];
                s_bond[i_bond][i] = s_i[i];
                u_bond[i_bond][i] = u_i[i];
                r[i1_site][i] = r_i[i] - 0.5 * length_i * u_i[i];
                r[i2_site][i] = r_i[i] + 0.5 * length_i * u_i[i];
            }

            /* Apply periodic boundary conditions and compute scaled positions of bond endpoints. */
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i1_site], s[i1_site]);
            periodic_boundary_conditions_single(n_dim, h, h_inv, r[i2_site], s[i2_site]);
        }
    }

    return;
}

/* This routine distributes spherical sites at random within an n_dim-dimensional
   parallelepiped, with periodic boundary conditions, avoiding close contacts.

   Input: number of spatial dimensions (n_dim)
   //number of spatial dimensions that are (n_periodic)
   number of sites (n_sites)
   unit cell matrix (h)
   maximum number of trial insertions per site (max_trials)
   pointer to gsl rng state for random number generator (r_rng)
   minimum site separation (dr_min)
   array of scaled site positions (s)
   array of real site positions (r)

   Output: s and r are modified on output */

void parallelepiped_insertion_periodic(int n_dim, int n_periodic, int n_sites, double **h,
                                       int max_trials, gsl_rng *r_rng, double dr_min,
                                       double **s, double **r) {
    int i_site, j_site, i, j, n_trials, successful;
    double dr_min2 = dr_min * dr_min, dr2;
    double ds[3], dr[3];

    /* Loop over sites to insert. */
    for (i_site = 0; i_site < n_sites; ++i_site) {

        /* Loop over trial insertions. */
        n_trials = 0;
        do {

            /* Increment trial counter and set flag. */
            ++n_trials;
            successful = 1;

            /* Generate random position within unit hypercube (scaled coordinates). */
            for (i = 0; i < n_dim; ++i)
                s[i_site][i] = gsl_rng_uniform(r_rng) - 0.5;

            /* Loop over previously inserted sites to test for close contacts. */
            for (j_site = 0; j_site < i_site; ++j_site) {

                /* Compute scaled pair separation. */
                for (i = n_periodic; i < n_dim; ++i) {
                    ds[i] = s[i_site][i] - s[j_site][i];
                    ds[i] -= NINT(ds[i]);
                }

                /* Compute real pair separation. */
                dr2 = 0.0;
                for (i = 0; i < n_dim; ++i) {
                    dr[i] = 0.0;
                    for (j = 0; j < n_dim; ++j)
                        dr[i] += h[i][j] * ds[j];
                    dr2 += SQR(dr[i]);
                }

                /* If a close contact is found, set flag to zero and break from loop. */
                if (dr2 < dr_min2) {
                    successful = 0;
                    break;
                }
            }
        } while(!successful && n_trials < max_trials);

        printf("i_site = %d, n_trials = %d\n", i_site, n_trials);
        fflush(NULL);

        /* Exit if a successful insertion was not made after max_trials trials. */
        if (!successful)
            error_exit("Maximum number of trial insertions exceeded in parallelepiped_insertion_periodic");
    }

    /* Compute real coordinates. */
    for (i_site = 0; i_site < n_sites; ++i_site) {
        for (i = 0; i < n_dim; ++i) {
            r[i_site][i] = 0.0;
            for (j = 0; j < n_dim; ++j)
                r[i_site][i] += h[i][j] * s[i_site][j];
        }
    }

    return;
}
