#ifndef _NAB_PARSE_FLAGS_H_
#define _NAB_PARSE_FLAGS_H_

// Creation for parsing the command line options for newagebob (NAB) in a better way.
// Taken shamelessly from SimCORE

#include <getopt.h>
#include <iostream>
#include <vector>

// Define the flags here
static const int n_nab_flags = 6;
static struct option long_options[] = {
    {"help", no_argument, 0, 'h'},
    {"restart", required_argument, 0, 'r'},
    {"xlink_test", no_argument, 0, 't'},
    {"tomo", required_argument, 0, 'T'},
    {"posit", required_argument, 0, 'p'},
    {"config", no_argument, 0, 'c'},
    {0, 0, 0, 0}
};

// Long description of what the flags are
static const std::string desc[n_nab_flags][2] = {
    {"show this menu\n", "none"},
    {"restart the simulation\n", "none"},
    {"Run simulation with two stationary spherocylinders in order to test xlink dynamics\n", "none"},
    {"Enables planar view of spindle\n", "none"},
    {"where posit is the name of a posit file to load (IN ORDER!)\n", "posit"},
    {"configure the simuation from posit files\n", "none"}
};

// Run parameters that we might need
struct run_options {
    run_options() {default_file = "";
                   equil_file = "";
                   opt_file = "";
                   restart = false;
                   xlink_test = false;
                   tomogram = 0;
                   configure = false;}
    bool restart;
    bool xlink_test;
    bool configure;
    int startframe;
    int tomogram;
    std::string default_file;
    std::string equil_file;
    std::string opt_file;
    std::vector<std::string> posit_file_names;
};

// Show the help information
void show_help_info(std::string progname) {
    std::cout << "\n" << "New Age Bob (NAB) usage: " << std::endl;
    std::cout << "  " << progname << " default_file equil_file opt_file(OPTIONAL) --flag1 option1 ...\n";
    std::cout << "where flags are one of the following: " << std::endl;
    for (int i = 0; i < n_nab_flags; ++i) {
        std::cout << "   --" << long_options[i].name;
        if (desc[i][1].compare("none") != 0)
            std::cout << " " << desc[i][1];
        std::cout << ", -" << (char) long_options[i].val;
        if (desc[i][1].compare("none") != 0)
            std::cout << " " << desc[i][1];
        std::cout << ", " << desc[i][0];
    }
}

// Parse opts itself
run_options parse_opts(int argc, char *argv[], int minargs) {
    if (argc == 1 || argc < minargs) {
        show_help_info(argv[0]);
        exit(1);
    }

    run_options run_opts;

    //std::cout << "checking arguments: " << argc << std::endl;
    while(1) {
        int option_index = 0;

        int c = getopt_long(argc, argv, "hr:tT:p:c", long_options, &option_index);
        //std::cout << "getopt checking arg[" << option_index << "], c: " << char(c) << std::endl;

        if (c == -1)
            break;

        switch(c) {
            case 0:
                std::cout << "Made it into case 0\n";
                break;
            case 'h':
                show_help_info(argv[0]);
                exit(0);
            case 'r':
                std::cout << "RESTART Enabled!\n";
                run_opts.restart = true;
                run_opts.startframe = atoi(optarg);
                break;
            case 't':
                std::cout << "Text configuration enabled.\n";
                run_opts.xlink_test = true;
                break;
            case 'p':
                run_opts.posit_file_names.push_back(optarg);
                break;
            case 'T':
                run_opts.tomogram = atoi(optarg);
                std::cout << "tomogram arg = " << run_opts.tomogram << std::endl;
                break;
            case 'c':
                run_opts.configure = true;
                break;
            case '?':
                break;
            default:
                std::cout << "ERROR: Unrecognized option\n";
                exit(1);
        }
    }

    // Now check for normal ARGV elements
    while (optind < argc) {
        //std::cout << "normal arg[" << optind << "], " << argv[optind] << std::endl;
        if (run_opts.default_file == "") {
            run_opts.default_file = argv[optind];
        } else if (run_opts.equil_file == "") {
            run_opts.equil_file = argv[optind];
        } else if (run_opts.opt_file == "") {
            run_opts.opt_file = argv[optind];
        }
        optind++;
    }

    return run_opts;
}

#endif
