/* This routine computes scaled coordinates and applies periodic boundary conditions
   to a single object (site, bond, molecule, etc.).

   Input: number of periodic spatial dimensions (n_periodic)
          unit cell matrix (h)
          inverse unit cell matrix (h_inv)
          real object coordinate (r)
          scaled object coordinate (s)

   Output: r and s are modified on output. */

#include "bob.h"

void periodic_boundary_conditions_single(int n_periodic, double **h, double **h_inv,
                                         double *r, double *s)
{
    int i, j;

    /* Compute scaled coordinate and apply periodic boundary conditions. */
    for (i = 0; i < n_periodic; ++i) {
        s[i] = 0.0;
        for (j = 0; j < n_periodic; ++j)
            s[i] += h_inv[i][j] * r[j];
        s[i] -= NINT(s[i]);
    }

    /* Recompute real coordinates accounting for periodic boundary conditions. */
    for (i = 0; i < n_periodic; ++i) {
        r[i] = 0.0;
        for (j = 0; j < n_periodic; ++j)
            r[i] += h[i][j] * s[j];
    }

    return;
}
