/* This routine integrates the Langevin equation for a system of
   spherocylinders assuming the inertial term is 0 (brownian dynamics).
   Refer to THE JOURNAL OF CHEMICAL PHYSICS 122, 244903 2005 for discussion.
   Note this routine is independent of the force model used and proper torques
   need to be calculated outside of this routine. See brownian_sphero.c routines
   for some examples.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)

   Output: spherocylinder positions, forces, orientations and the
   potential energy and virial are modified on output */

#include "bob.h"


void position_step_bd_mp(system_parameters * parameters,
                         system_properties * properties,
                         system_potential * potential) {
    int i, j, i_bond, n_bonds, n_dim, n_periodic, motility_flag,
        reorientation_flag, i_site, j_site, *calc_matrix;
    double delta, temp, factor, uiuj,
        *gamma_par, *gamma_perp, *gamma_rot, **dr_tot,
        **r_bond, **s_bond, **h, **h_inv,
        **t_bond, **v_bond, **u_bond, *length, **f_bond;
    gsl_rng *r_rng;
    double dr[3];
    double u_bond_old[3], du[3];
    double body_frame[9];
    double xi[9], xi_inv[9];

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    n_periodic = parameters->n_periodic;
    delta = parameters->delta;
    temp = parameters->temp;
    h = properties->unit_cell.h;
    h_inv = properties->unit_cell.h_inv;
    r_bond = properties->bonds.r_bond;
    s_bond = properties->bonds.s_bond;
    n_bonds = properties->bonds.n_bonds;
    v_bond = properties->bonds.v_bond;
    u_bond = properties->bonds.u_bond;
    length = properties->bonds.length;
    gamma_par = properties->bonds.gamma_par;
    gamma_perp = properties->bonds.gamma_perp;
    gamma_rot = properties->bonds.gamma_rot;
    dr_tot = properties->sites.dr_tot;
    t_bond = properties->bonds.t_bond;
    calc_matrix = potential->calc_matrix;
    f_bond = properties->bonds.f_bond;
    motility_flag = parameters->motility_flag;
    reorientation_flag = parameters->reorientation_flag;

    /* Flag quantities that require updating or checking the next time they're needed. */
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.inter_site_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

    /* Evaluate forces and reorientations for BD integration routine */
    evaluate_forces_bd(parameters, properties, potential);

    #ifdef ENABLE_OPENMP
    #pragma omp parallel private(i_bond, i, j, xi, xi_inv, uiuj, dr, u_bond_old, du, factor, \
                             i_site, j_site, body_frame, r_rng)
    #endif
    {
        int i_thr = 0;
        #ifdef ENABLE_OPENMP
        i_thr = omp_get_thread_num();
        #endif
        r_rng = properties->rng_mt.rng[i_thr].r;
        #ifdef ENABLE_OPENMP
        #pragma omp for schedule(runtime) nowait
        #endif
        for(i_bond = 0; i_bond < n_bonds; ++i_bond) {
            /* Make copies of relevant data */
            for (i = 0; i < n_dim; ++i) {
                du[i] = 0.0;
                dr[i] = 0.0;
                u_bond_old[i] = u_bond[i_bond][i];
            }

            /* calculate body frame transformation. n_dim x n_dim matrix.
               each row contains a unit vector defining a direction relative
               to the rod
               body_frame[0]: parallel
               body_frame[i>0]: perpendicular */
            generate_body_frame(n_dim, u_bond[i_bond], body_frame);

            /* External forces on particle. Don't just add random force */
            if (calc_matrix[i_bond]) {
                /* Calculate drag matrix for i_bond */
                for(i = 0; i < n_dim; ++i) {
                    for(j = i; j < n_dim; ++j) {
                        uiuj = u_bond[i_bond][i] * u_bond[i_bond][j];
                        xi[i*n_dim + j] = xi[j*n_dim + i] = uiuj * (gamma_par[i_bond] - gamma_perp[i_bond]);
                    }
                }
                for (i = 0; i < n_dim; ++i)
                    xi[i*n_dim + i] += gamma_perp[i_bond];

                /* Calculate inverse drag matrix */
                if (n_dim == 2)
                    invert_sym_2d_matrix(xi, xi_inv);
                else if (n_dim == 3)
                    invert_sym_3d_matrix(xi, xi_inv);

                /* Add deterministic rod displacements to accumulator */
                for(i = 0; i < n_dim; ++i)
                    for(j = 0; j < n_dim; ++j) {
                        dr[i] += xi_inv[i*n_dim + j] * f_bond[i_bond][j] * delta;
                    }
            }
            /* External forces on particle from motility force don't require breakdown
               into components, only parallel */
            else if (motility_flag == 1) {
                for(i = 0; i < n_dim; ++i)
                    dr[i] += f_bond[i_bond][i] * delta / gamma_par[i_bond];
            }

            /* Calculate and add random parallel displacement from gaussian distribution */
            factor = sqrt(2.0 * temp * delta / gamma_par[i_bond]) *
                gsl_ran_gaussian_ziggurat(r_rng, 1.0);
            for(i = 0; i < n_dim; ++i)
                dr[i] += body_frame[i] * factor;

            /* Calculate and add random perpendicular displacement from gaussian distribution */
            for (i = 1; i < n_dim; ++i) {
                factor = sqrt(2.0 * temp * delta / gamma_perp[i_bond]) *
                    gsl_ran_gaussian_ziggurat(r_rng, 1.0);
                for(j = 0; j < n_dim; ++j)
                    dr[j] += body_frame[i*n_dim + j] * factor;
            }

            if (reorientation_flag) {
                /* Add torques to displacement */
                double u_i[3] = {u_bond[i_bond][0], u_bond[i_bond][1], 0.0};
                if (n_dim == 3)
                    u_i[2] = u_bond[i_bond][2];
                
                cross_product(t_bond[i_bond], u_i, du, n_dim);
                for (i = 0; i < n_dim; ++i)
                    du[i] *= delta / gamma_rot[i_bond];
            
                /* Calculate and add random rotation from gaussian distribution */
                for (i = 1; i < n_dim; ++i) {
                    factor = sqrt(2.0 * temp * delta / gamma_rot[i_bond]) * 
                        gsl_ran_gaussian_ziggurat(r_rng, 1.0);
                    for(j = 0; j < n_dim; ++j)
                        du[j] += body_frame[i*n_dim + j] * factor;
                }

                /* Add and rescale reorientations (along new director,
                   refer to discussion by Tao et. al.
                   THE JOURNAL OF CHEMICAL PHYSICS 122, 244903 2005 */
                factor = 0.0;
                for(i = 0; i < n_dim; ++i) {
                    u_bond[i_bond][i] += du[i];

                    factor += SQR(u_bond[i_bond][i]);
                }
                factor = 1.0/sqrt(factor);
                for (i = 0; i < n_dim; ++i)
                    u_bond[i_bond][i] *= factor;
            }

            /* Update v_bond to lie along new director */
            for(i = 0; i < n_dim; ++i)
                v_bond[i_bond][i] = u_bond[i_bond][i] * length[i_bond];


            /* Add accumulated displacement to rod position */
            for (i = 0; i < n_dim; ++i) {
                r_bond[i_bond][i] += dr[i];
            }

            /* Calculate end displacements and add them to accumulator for neighbor list check */
            i_site = properties->bonds.bond_site_1[i_bond];
            j_site = properties->bonds.bond_site_2[i_bond];
            for (i = 0; i < n_dim; ++i) {
                du[i] = u_bond[i_bond][i] - u_bond_old[i];

                /* Add displacements to accumulator */
                dr_tot[i_site][i] += dr[i] - 0.5 * length[i_bond] * du[i];
                dr_tot[j_site][i] += dr[i] + 0.5 * length[i_bond] * du[i];
            }
            periodic_boundary_conditions_single(n_periodic, h, h_inv, r_bond[i_bond], s_bond[i_bond]);
        }
    }

    return;
}
