/* This routine integrates the Langevin equation for a system of
   spherocylinders assuming the inertial term is 0 (brownian dynamics).
   Refer to THE JOURNAL OF CHEMICAL PHYSICS 122, 244903 2005 for discussion.
   Note this routine is independent of the force model used and proper torques
   need to be calculated outside of this routine. See brownian_sphero.c routines
   for some examples.

   Input: pointer to parameters structure (parameters)
   pointer to properties structure (properties)
   pointer to potential structure (potential)

   Output: spherocylinder positions, forces, orientations and the
   potential energy and virial are modified on output */

#include "bob.h"

#include <iostream>


void position_step_spindle_bd_mp(system_parameters * parameters,
                                 system_properties * properties,
                                 system_potential * potential) {
    int i, j, i_bond, n_bonds, n_dim, motility_flag, 
        i_anchor, n_anchors, i_site, n_sites, j_site, *calc_matrix;
    double delta, temp, factor, uiuj, conf_rad, diff_factor,
        *gamma_par, *gamma_perp, *gamma_rot, **dr_tot,
        **r_bond, **t_bond, **v_bond, **u_bond, *length, **f_bond;

    double **r_anchor;
    double dr[3];
    double u_bond_old[3], du[3];
    double body_frame[9];
    double xi[9], xi_inv[9];

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    delta = parameters->delta;
    temp = parameters->temp;
    r_bond = properties->bonds.r_bond;
    n_bonds = properties->bonds.n_bonds;
    v_bond = properties->bonds.v_bond;
    u_bond = properties->bonds.u_bond;
    length = properties->bonds.length;
    n_sites = properties->sites.n_sites;
    gamma_par = properties->bonds.gamma_par;
    gamma_perp = properties->bonds.gamma_perp;
    gamma_rot = properties->bonds.gamma_rot;
    dr_tot = properties->sites.dr_tot;
    t_bond = properties->bonds.t_bond;
    calc_matrix = potential->calc_matrix;
    f_bond = properties->bonds.f_bond;
    motility_flag = parameters->motility_flag;
    n_anchors = properties->anchors.n_anchors;
    r_anchor = properties->anchors.r_anchor;
    double thermal_diff_scale = parameters->thermal_diff_scale;

    /* Flag quantities that require updating or checking the next time they're needed. */
    properties->control.bond_vector_flag = 1;
    properties->control.bond_position_flag = 1;
    properties->control.inter_site_flag = 1;
    properties->control.cell_list_flag = 1;
    properties->control.neighbor_list_flag = 1;

    /* Evaluate forces and reorientations for BD integration routine */
    evaluate_forces_bd(parameters, properties, potential);

    if (parameters->mt_diffusion_flag) {

        //#ifdef ENABLE_OPENMP
        //#pragma omp parallel private(i_bond, i, j, xi, xi_inv, uiuj, dr, u_bond_old, du, factor, \
        //                         i_site, j_site, body_frame)
        //#endif
        { 
            //#ifdef ENABLE_OPENMP
            //#pragma omp for schedule(runtime) nowait
            //#endif
            for(i_bond = 0; i_bond < n_bonds; ++i_bond) {
                /* Make copies of relevant data */
                for (i = 0; i < n_dim; ++i) {
                    du[i] = 0.0;
                    dr[i] = 0.0;
                    u_bond_old[i] = u_bond[i_bond][i];
                }

                /* calculate body frame transformation. n_dim x n_dim matrix.
                   each row contains a unit vector defining a direction relative
                   to the rod
                   body_frame[0]: parallel
                   body_frame[i>0]: perpendicular */
                generate_body_frame(n_dim, u_bond[i_bond], body_frame);

                /* External forces on particle. Don't just add random force */
                if (calc_matrix[i_bond]) {
                    /* Calculate drag matrix for i_bond */
                    for(i = 0; i < n_dim; ++i) {
                        for(j = i; j < n_dim; ++j) {
                            uiuj = u_bond[i_bond][i] * u_bond[i_bond][j];
                            xi[i*n_dim + j] = xi[j*n_dim + i] = uiuj * (gamma_par[i_bond] - gamma_perp[i_bond]);
                        }
                    }
                    for (i = 0; i < n_dim; ++i)
                        xi[i*n_dim + i] += gamma_perp[i_bond];

                    /* Calculate inverse drag matrix */
                    if (n_dim == 2)
                        invert_sym_2d_matrix(xi, xi_inv);
                    else if (n_dim == 3)
                        invert_sym_3d_matrix(xi, xi_inv);

                    /* Add deterministic rod displacements to accumulator */
                    for(i = 0; i < n_dim; ++i) {
                        for(j = 0; j < n_dim; ++j) {
                            dr[i] += xi_inv[i*n_dim + j] * f_bond[i_bond][j] * delta;
                        }
                    }
                }
                /* External forces on particle from motility force don't require breakdown
                   into components, only parallel */
                else if (motility_flag == 1) {
                    for(i = 0; i < n_dim; ++i)
                        dr[i] += f_bond[i_bond][i] * delta / gamma_par[i_bond];
                }

                /* Calculate and add random parallel displacement from gaussian distribution */
                factor = thermal_diff_scale * sqrt(2.0 * temp * delta / gamma_par[i_bond]) * gsl_ran_gaussian_ziggurat(properties->bonds.rng_local[i_bond].r, 1.0);
                for(i = 0; i < n_dim; ++i)
                    dr[i] += body_frame[0*n_dim + i] * factor;

                /* Calculate and add random perpendicular displacement from gaussian distribution */
                for (i = 1; i < n_dim; ++i) {
                    factor = thermal_diff_scale * sqrt(2.0 * temp * delta / gamma_perp[i_bond]) * gsl_ran_gaussian_ziggurat(properties->bonds.rng_local[i_bond].r, 1.0);
                    for(j = 0; j < n_dim; ++j)
                        dr[j] += body_frame[i*n_dim + j] * factor;
                }

                //// FIXME XXX
                //bool trip_exit = false;
                //// Do a quick test to see if the microtubule is off in lala land
                //if (dot_product(n_dim, dr, dr) > 0.1) {
                //    std::cerr << "ERROR step: " << properties->i_current_step << ", bond: " << i_bond << std::endl;
                //    std::cerr << "ERROR in microtubule motion, dr: " << sqrt(dot_product(n_dim, dr, dr)) << std::endl;
                //    std::cerr << "Force (" << f_bond[i_bond][0] << ", "
                //                           << f_bond[i_bond][1] << ", "
                //                           << f_bond[i_bond][2] << ")\n";
                //    trip_exit = true;
                //}

                //conf_rad = 0.5 * properties->unit_cell.h[0][0];
                //double conf_rad2 = SQR(conf_rad);
                //if (dot_product(n_dim, r_bond[i_bond], r_bond[i_bond]) > conf_rad2) {
                //    std::cerr << "ERROR step: " << properties->i_current_step << ", bond: " << i_bond << std::endl;
                //    std::cerr << "ERROR in microtubule location\n";
                //    std::cerr << "loc (" << r_bond[i_bond][0] << ", "
                //                         << r_bond[i_bond][1] << ", "
                //                         << r_bond[i_bond][2] << ")\n";
                //    trip_exit = true;
                //}

                //if (trip_exit) {
                //    exit(1);
                //}
                //// FIXME XXX

                /* Calculate displacement due to torque */
                cross_product(t_bond[i_bond], u_bond[i_bond], du, n_dim);
                for (i = 0; i < n_dim; ++i)
                    du[i] *= delta / gamma_rot[i_bond];

                // DEBUG print out the du due to the torques on the system
                //std::cout << "du torque: " << sqrt(dot_product(n_dim, du, du)) << std::endl;

                /* Calculate and add random rotation from gaussian distribution */
                for (i = 1; i < n_dim; ++i) {
                    factor = thermal_diff_scale * sqrt(2.0 * temp * delta / gamma_rot[i_bond]) * gsl_ran_gaussian_ziggurat(properties->bonds.rng_local[i_bond].r, 1.0);
                    for(j = 0; j < n_dim; ++j)
                        du[j] += body_frame[i*n_dim + j] * factor;
                }

                // DEBUG prin tout the du due to all the torques in the system, including the stupid non deterministic torques
                //std::cout << "du all   :" << sqrt(dot_product(n_dim, du, du)) << std::endl;

                /* Add accumulated displacement to rod position */
                for (i = 0; i < n_dim; ++i) {
                    r_bond[i_bond][i] += dr[i];
                }

                /* Add and rescale reorientations (along new director, refer to discussion by Tao et. al.
                   THE JOURNAL OF CHEMICAL PHYSICS 122, 244903 2005 */
                factor = 0.0;
                for(i = 0; i < n_dim; ++i) {
                    u_bond[i_bond][i] += du[i];

                    factor += SQR(u_bond[i_bond][i]);
                }
                factor = 1.0/sqrt(factor);
                for (i = 0; i < n_dim; ++i)
                    u_bond[i_bond][i] *= factor;

                /* Update v_bond to lie along new director */
                for(i = 0; i < n_dim; ++i)
                    v_bond[i_bond][i] = u_bond[i_bond][i] * length[i_bond];

                /* Calculate end displacements and add them to accumulator for neighbor list check */
                i_site = properties->bonds.bond_site_1[i_bond];
                j_site = properties->bonds.bond_site_2[i_bond];
                for (i = 0; i < n_dim; ++i) {
                    du[i] = u_bond[i_bond][i] - u_bond_old[i];

                    /* Add displacements to accumulator */
                    dr_tot[i_site][i] += dr[i] - 0.5 * length[i_bond] * du[i];
                    dr_tot[j_site][i] += dr[i] + 0.5 * length[i_bond] * du[i];

                    // Attach to the chromosome information as well, and update the accumulators there for 
                    // the MTs, and this should even be thread-safe!
                    // FIXME: Probabily a more elegant way to do this
                    if (properties->chromosomes.nchromosomes_ > 0) {
                        properties->chromosomes.dr_tot_mt_[i_bond  ][i] += dr[i] - 0.5 * length[i_bond] * du[i];
                        properties->chromosomes.dr_tot_mt_[i_bond+1][i] += dr[i] + 0.5 * length[i_bond] * du[i];
                    }
                }
            }
        } // omp parallel region
    } // MTs are diffusing

    diff_factor = thermal_diff_scale * sqrt(2.0 * parameters->delta * parameters->temp /
                       parameters->sphere_diameter);
    //std::cout << "main diff factor: " << diff_factor << std::endl;
    //XXX FIXME: This is actually wrong, divide by the kc diameter instead, but leave for now!!!!
    for (i_site = n_bonds * 2; i_site < n_sites; ++i_site) {
        for (i = 0; i < n_dim; ++i) {
            properties->sites.r[i_site][i] += properties->sites.f[i_site][i] * delta / diff_factor;
        }
    }

    for (i_site = n_bonds * 2; i_site < n_sites; ++i_site) {
        for (i = 0; i < n_dim; ++i) {
            properties->sites.r[i_site][i] += thermal_diff_scale * gsl_ran_gaussian_ziggurat(properties->rng.r, 1.0) * diff_factor;
        }
    }

    /* Update the information for the anchor.  There are two torques, the global one maps to
     * translation of the anchor, and the local one to the rotation of the anchor.  So
     * we have to be careful, as we have both a translation and rotation, and have to update
     * only the necessary components */
    conf_rad = 0.5 * properties->unit_cell.h[0][0];
    if (parameters->use_spb_flag == 1) {
        for (i_anchor = 0; i_anchor < n_anchors; ++i_anchor) {
            double gamma_t = properties->anchors.gamma_tra[i_anchor] * conf_rad;
            double gamma_r = properties->anchors.gamma_rot[i_anchor];

            // The translation of the body is the only thing the anchor cares about
            double u_anchor_old[3];
            double *u_anchor = properties->anchors.u_anchor[i_anchor];
            double *v_anchor = properties->anchors.v_anchor[i_anchor];
            double *w_anchor = properties->anchors.w_anchor[i_anchor];
            double *f_anchor = properties->anchors.f_anchor[i_anchor];
            double f_dot_u = dot_product(n_dim, f_anchor, u_anchor);
            double f_rand[3] = {
                thermal_diff_scale * gsl_ran_gaussian_ziggurat(properties->rng.r, sqrt(2.0 * gamma_t / delta)),
                thermal_diff_scale * gsl_ran_gaussian_ziggurat(properties->rng.r, sqrt(2.0 * gamma_t / delta)),
                thermal_diff_scale * gsl_ran_gaussian_ziggurat(properties->rng.r, sqrt(2.0 * gamma_t / delta))
            };

            f_dot_u += dot_product(n_dim, f_rand, u_anchor);
            
            for (int i = 0; i < n_dim; ++i) {
                u_anchor_old[i] = u_anchor[i];
                u_anchor[i] -= (f_anchor[i]+f_rand[i] -
                                f_dot_u * u_anchor[i]) * delta / gamma_t;
            }

            double norm_factor = sqrt(dot_product(n_dim, u_anchor, u_anchor));
            for (int i = 0; i < n_dim; ++i) {
                u_anchor[i] /= norm_factor;
                r_anchor[i_anchor][i] = -u_anchor[i] * conf_rad;
            }

            double tau_random = thermal_diff_scale * gsl_ran_gaussian(properties->rng.r,
                                                 sqrt(2.0*gamma_r / delta));
            double *tau_anchor_body = properties->anchors.tau_local[i_anchor];
            double tau_body_full[3] = {
                tau_anchor_body[0] + u_anchor[0] * tau_random,
                tau_anchor_body[1] + u_anchor[1] * tau_random,
                tau_anchor_body[2] + u_anchor[2] * tau_random
            };
            double dv[3]; cross_product(tau_body_full, v_anchor, dv, 3);
            for (int i = 0; i < 3; ++i) dv[i] *= delta / gamma_r;
            for (int i = 0; i < 3; ++i) v_anchor[i] += dv[i];
            double orthocorrection = -dot_product(n_dim, v_anchor, u_anchor);
            for (int i = 0; i < 3; ++i) v_anchor[i] += orthocorrection * u_anchor_old[i];
            norm_factor = sqrt(dot_product(n_dim, v_anchor, v_anchor));
            for (int i = 0; i < 3; ++i) v_anchor[i] /= norm_factor;

            // double du[3] = {u_anchor[0] - u_anchor_old[0],
            //                 u_anchor[1] - u_anchor_old[1],
            //                 u_anchor[2] - u_anchor_old[2],
            // };
            // printf("%g %g\n", sqrt(dot_product(3, dr, dr)),
            //        sqrt(dot_product(3, dv, dv)));

            // FIXME: This looks like the rodrigues rotation formula, figure out why....
            double n[3]; cross_product(u_anchor_old, u_anchor, n, n_dim);
            double sin_omega = dot_product(n_dim, n, n);
            norm_factor = sqrt(sin_omega);
            if (norm_factor > 1e-8) {
                for (int i = 0; i < 3; ++i) n[i] /= norm_factor;
                double cos_omega = dot_product(n_dim, u_anchor, u_anchor_old);
                double vec1[3]; cross_product(n, v_anchor, vec1, n_dim);
                double factor2 = dot_product(n_dim, n, v_anchor) * (1 - cos_omega);
                for (int i = 0; i < 3; ++i) {
                    v_anchor[i] = v_anchor[i] * cos_omega +
                        vec1[i] * sin_omega + n[i] * factor2;
                }
            }

            cross_product(u_anchor, v_anchor, w_anchor, 3);

            for (al_list::iterator p = properties->anchors.anchor_list[i_anchor].begin();
                 p < properties->anchors.anchor_list[i_anchor].end();
                 p++) {
                double factor_u = dot_product(n_dim, p->pos_rel, u_anchor);
                double factor_v = dot_product(n_dim, p->pos_rel, v_anchor);
                double factor_w = dot_product(n_dim, p->pos_rel, w_anchor);

                // Calculate new lab frame coordinate
                for (int i = 0; i < 3; ++i) {
                    p->pos[i] = r_anchor[i_anchor][i] + factor_u * u_anchor[i] +
                        factor_v * v_anchor[i] + factor_w * w_anchor[i];
                }
                //Yank it back to the surface of sphere because numerical errors are annoying
                norm_factor = conf_rad / sqrt(dot_product(n_dim, p->pos, p->pos));
                for (int i = 0; i < 3; ++i) p->pos[i] *= norm_factor; 

                // Make sure that we properly attack the orientation of the anchor list
                // Orientations!
                double original_u[3] = {0.0};
                for (int i = 0; i < 3; ++i) {
                    original_u[i] = p->u[i];
                    p->u[i] = p->u_rel[0] * u_anchor[i] +
                              p->u_rel[1] * v_anchor[i] +
                              p->u_rel[2] * w_anchor[i];
                }
                //std::cout << "original anchor orientation (lab frame): u: ["
                //            << original_u[0] << ", " 
                //            << original_u[1] << ", "
                //            << original_u[2] << "]\n";
                //std::cout << "new      anchor orientation (lab frame): u: ["
                //            << p->u[0] << ", " 
                //            << p->u[1] << ", "
                //            << p->u[2] << "]\n";
                //double unormnew = dot_product(n_dim, p->u, p->u);
                //std::cout << "new norm: " << unormnew << std::endl;
            }
        }
    }

    // Let the chromosome class handle updating their positions
    if (properties->chromosomes.enabled_) {
        properties->chromosomes.UpdatePositions();
    }

    return;
}
