/* This routine reads the unit cell matrix and site positions from an input file.

   Input: number of spatial dimensions (n_dim)
          number of sites (n_sites)
          unit cell matrix (h)
          array of site positions (r)
          pointer to input file (f_posit)

   Output: the unit cell matrix and site positions are read from the input file */

#include "bob.h"

extern int posit_double_flag;

int read_positions(int n_dim, int n_sites, double **h, double **r, FILE * f_posit)
{
    int i, j, i_site;

    /* Read unit cell matrix from input file. */
    if (posit_double_flag) {
        for (i = 0; i < n_dim; ++i)
            fread(h[i], sizeof(double), n_dim, f_posit);
        
        /* Read site positions from input file. */
        for (i_site = 0; i_site < n_sites; ++i_site) {
            fread(r[i_site], sizeof(double), n_dim, f_posit);
        }
    }
    else {
        for (i = 0; i < n_dim; ++i)
            for (j = 0; j < n_dim; ++j) {
                float tmp;
                fread(&tmp, sizeof(float), 1, f_posit);
                h[i][j] = tmp;
            }
        for (i_site = 0; i_site < n_sites; ++i_site) {
            for (i = 0; i < n_dim; ++i) {
                float tmp;
                fread(&tmp, sizeof(float), 1, f_posit);
                r[i_site][i] = tmp;
            }
        }
    }

    return feof(f_posit);
}

int read_positions_spindle(int n_dim, int n_sites, int n_anchors, 
                           double *time_s, double **h, double **r, 
                           double **r_a, double **uref_a, 
                           FILE * f_posit)
{
    int i, j, i_site;
    float tmp;

    /* Read unit cell matrix from input file. */
    if (posit_double_flag) {
        for (i = 0; i < n_dim; ++i)
            fread(h[i], sizeof(double), n_dim, f_posit);

        /* Read time from input file */
        fread(time_s, sizeof(double), 1, f_posit);
        
        /* Read site positions from input file. */
        for (i_site = 0; i_site < n_sites; ++i_site)
            fread(r[i_site], sizeof(double), n_dim, f_posit);

        /* Read anchor positions from input file. */
        for (i_site = 0; i_site < n_anchors; ++i_site)
            fread(r_a[i_site], sizeof(double), n_dim, f_posit);

        /* Read anchor refs from input file. */
        for (i_site = 0; i_site < n_anchors; ++i_site)
            fread(uref_a[i_site], sizeof(double), n_dim, f_posit);
    }
    else {
        for (i = 0; i < n_dim; ++i) {
            for (j = 0; j < n_dim; ++j) {
                fread(&tmp, sizeof(float), 1, f_posit);
                h[i][j] = tmp;
            }
        }
        fread(&tmp, sizeof(float), 1, f_posit);
        *time_s = tmp;

        for (i_site = 0; i_site < n_sites; ++i_site) {
            for (i = 0; i < n_dim; ++i) {
                fread(&tmp, sizeof(float), 1, f_posit);
                r[i_site][i] = tmp;
            }
        }

        for (i_site = 0; i_site < n_anchors; ++i_site) {
            for (i = 0; i < n_dim; ++i) {
                fread(&tmp, sizeof(float), 1, f_posit);
                r_a[i_site][i] = tmp;
            }
        }

        for (i_site = 0; i_site < n_anchors; ++i_site) {
            for (i = 0; i < n_dim; ++i) {
                fread(&tmp, sizeof(float), 1, f_posit);
                uref_a[i_site][i] = tmp;
            }
        }
    }

    return feof(f_posit);
}
