#include "bob.h"
#include <iostream>

void check_wall_stability(system_parameters * parameters, 
                            poly_state_enum * poly_state){
    if (!parameters->force_induced_catastrophe_flag &&
        !parameters->force_poly_flag &&
        parameters->dynamic_instability_flag) {
        (*poly_state) = SHRINK;
    }
}

//Function to calculate wall force by a non-monotonic membrane
//See "Formation and Interaction of Membrane Tubes"
//Imre Derényi - Frank Jülicher - Jacques Prost - Phys. Rev. Lett. Physical Review Letters - 2002
//Inputs: ne_ratio = ratio between membrane bending rigidity and its surface tension ( also can be calculated from membrane tube radius i.e. ne_ratio = 2*(radius of tube)^2
//        f0 = asymptotic wall force i.e. f(inf) = f0
//        dr = protrusion of rod tip through wall i.e. delta_r = tip distance from center - radius of spherical enclosure
//Outputs: Magnitude of force exerted by wall radially inward
double calc_non_monotonic_wall_force(double ne_ratio, double f0, double delta_r){
    //Constants from paper
    double a1=0.746; double a2=0.726; 
    double alpha1=0.347; double alpha2=3.691;
    double a = a1*a2; double b = sqrt(ne_ratio); double c = alpha1 + alpha2;
    // Calculate location of maximum force
    double xm = b*(((7.0/4.0)*M_PI) - c); //value of dr that gives max force
    // Calculate wall force
    // Exponential prefactor added to make forces conform to boundary conditions. 
    // Characteristic length chosen to be 1/30th the value of delta_r that gives max force
    double factor = (1-exp(-delta_r*30/xm))*(2.0*f0*a*exp(-1.0*delta_r/b) * cos( (delta_r/b)+c ) + f0);
    return factor;
}

/*WCA wall potential*/
double wca_sphero_wall_potential_bd(system_parameters *parameters, 
                                            system_properties *properties,
                                            double **f_bond, double **virial, 
                                            double **t_bond, int *calc_matrix) {
    double r_cutoff = pow(2,1.0/6.0);
    double conf_rad = 0.5 * properties->unit_cell.h[0][0] - r_cutoff + 0.5;
    double conf_rad2 = SQR(conf_rad);

    // Set the bond tip to zero force and use on JUST THE PLUS TIP
    double **fbondtip_ne = properties->bonds.fbondtip_ne;

    /* Zero vdw forces, potential energy and virial. */
    double u = 0.0;
    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));
    memset(fbondtip_ne[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    
    if (properties->control.virial_flag)
        memset(virial[0], 0, SQR(parameters->n_dim) * sizeof(double));

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        double f_cutoff = 0.1 / parameters->delta * properties->bonds.gamma_par[i_bond];
        
        double r_mag2_0 = 0.0; double r_mag2_1 = 0.0;
        double r_contact_0[3], r_contact_1[3];
        for (int i = 0; i < parameters->n_dim; ++i) {
            r_contact_0[i] = -0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            r_contact_1[i] =  0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            r_mag2_0 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_0[i]);
            r_mag2_1 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_1[i]);
        }
            
        //Minus end of the MT touching the wall
        if (r_mag2_0 > conf_rad2 && !parameters->force_poly_flag) {
            double r_mag = sqrt(r_mag2_0);
            
            double delta_r = -r_mag + conf_rad + r_cutoff;
            double rho2 = 1.0 / SQR(delta_r);
            double rho6 = CUBE(rho2);
            double rho12 = SQR(rho6);
            double factor = 24.0 * (2.0 * rho12 - rho6) * rho2;
            
            if (factor * delta_r > f_cutoff) {
                //std::cout << "NOTE tripping fcutoff in single site, resetting force factor, original " << factor << " to ";
                factor = f_cutoff / delta_r;
                //std::cout << factor << std::endl;
            }
            double f[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                f[i] = -factor * delta_r *
                    (properties->bonds.r_bond[i_bond][i] + r_contact_0[i]) / r_mag;
            }
            
            double tau[3];
            cross_product(r_contact_0, f, tau, parameters->n_dim);
            for (int i = 0; i < parameters->n_dim; ++i) {
                f_bond[i_bond][i] += f[i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += tau[i];
            }

            calc_matrix[i_bond] = 1;

            // Do the center of mass virial force
            // This is somewhat ill defined because of the 'cm' of the nuclear envelope, should be correct?
            if (properties->control.virial_flag) {
                //double r_cm[3] = {0.0};
                //for (int i = 0; i < parameters->n_dim; ++i) {
                //    r_cm[i] = delta_r * (properties->bonds.r_bond[i_bond][i] + r_contact_0[i]);
                //}
                for (int i = 0; i < parameters->n_dim; ++i) {
                    for (int j = 0; j < parameters->n_dim; ++j) {
                        virial[i][j] += -factor * delta_r * delta_r * (properties->bonds.r_bond[i_bond][i] + r_contact_0[i]) / r_mag;
                    }
                }
            }
            
            u += 4.0 * (rho12 - rho6) + 1.0;
        }
        if (r_mag2_1 > conf_rad2) {
            double r_mag = sqrt(r_mag2_1);
            
            double delta_r = -r_mag + conf_rad + r_cutoff;
            double rho2 = 1.0 / SQR(delta_r);
            double rho6 = CUBE(rho2);
            double rho12 = SQR(rho6);
            double factor = 24.0 * (2.0 * rho12 - rho6) * rho2;
            
            if (factor * delta_r > f_cutoff)
                factor = f_cutoff / delta_r;
            double f[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                f[i] = -factor * delta_r *
                    (properties->bonds.r_bond[i_bond][i] + r_contact_1[i]) / r_mag;
            }
            
            double tau[3];
            cross_product(r_contact_1, f, tau, parameters->n_dim);
            for (int i = 0; i < parameters->n_dim; ++i) {
                f_bond[i_bond][i] += f[i];
                // Only the tip cares about the NE stuff
                fbondtip_ne[i_bond][i] += f[i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += tau[i];
            }

            calc_matrix[i_bond] = 1;

            check_wall_stability(parameters, &(properties->bonds.poly_state[i_bond]));
            
            if (properties->control.virial_flag) {
                for (int i = 0; i < parameters->n_dim; ++i) {
                    for (int j = 0; j < parameters->n_dim; ++j) {
                        virial[i][j] += -factor * delta_r * delta_r * (properties->bonds.r_bond[i_bond][i] + r_contact_1[i]) / r_mag;
                    }
                }
            }
            
            u += 4.0 * (rho12 - rho6) + 1.0;
        }
    }

    return u;
}

// Wall potential for a hypercubic system with 1 or more non-periodic boundaries
// Not set up for parallelbiped systems yet
double wca_cubic_wall_sphero_potential_bd(system_parameters *parameters, 
                                          system_properties *properties, 
                                          double **f_bond, double **virial,
                                          double **t_bond, int *calc_matrix){ 

    int n_dim = parameters->n_dim;
    int n_periodic = parameters->n_periodic;
    double** h = properties->unit_cell.h;
    // Set the bond tip to zero force and use on JUST THE PLUS TIP
    double **fbondtip_ne = properties->bonds.fbondtip_ne;

    // Potential energy of system
    double u = 0.0;

    // Distance of interaction for WCA potential
    double r_cutoff = pow(2,1.0/6.0);

    // Location of interaction region for walls (+.5 for rod tip)
    double conf_dist[3];
    for ( int i=n_periodic; i<n_dim; ++i){
        conf_dist[i] = 0.5 * properties->unit_cell.h[i][i] - r_cutoff + 0.5;
    }

    // Initialize memory for force and torque vectors
    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));
    memset(fbondtip_ne[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));


    // Initiallize memomry for virial tensor
    if (properties->control.virial_flag)
        memset(virial[0], 0, SQR(parameters->n_dim) * sizeof(double));

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        // Cutoff force for potentials
        double f_cutoff = 0.1 / parameters->delta * (properties->bonds.gamma_par[i_bond]);

        // Force and torque
        double f_0[3]={0}, tau_0[3]={0}, f_1[3]={0}, tau_1[3]={0};
        
        // The minus and plus tips distance from the center of the system
        // and their magnitudes squared.
        double r_plus[3], r_minus[3], r_contact_0[3], r_contact_1[3]; 
        for( int i=0; i < n_dim; ++i){
            r_minus[i] = - (0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i]);
            r_plus[i] = (0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i]);
            r_contact_0[i] = properties->bonds.r_bond[i_bond][i] + r_minus[i]; 
            r_contact_1[i] = properties->bonds.r_bond[i_bond][i] + r_plus[i]; 
        }

        // Non-periodic dimension loop
        for (int i = n_periodic; i < n_dim; ++i) {
            
            if (ABS(r_contact_0[i]) > conf_dist[i]) {

                // Distance tip is in the interaction zone of the wall
                double r_comp_mag = ABS(r_contact_0[i]);
                double delta_r = -1.0 * r_comp_mag + conf_dist[i] + r_cutoff;

                //if (delta_r < 0)
                    //fprintf(stderr, " --- Outside of boundary ");
                
                // Factors for WCA potential
                double rho2 = 1.0 / SQR(delta_r);
                double rho6 = CUBE(rho2);
                double rho12 = SQR(rho6);
                double factor = 24.0 * (2.0 * rho12 - rho6) * rho2;

                // Apply cutoff is potential is too strong
                if (factor * delta_r > f_cutoff) {
                    factor = f_cutoff / delta_r;
                }

                // Component of force exerted by wall on rod
                //f_0[i] = -factor * delta_r * r_contact_0[i]/r_comp_mag;
                f_0[i] = -factor * delta_r * SIGN(1.0, r_contact_0[i]);
                u += 4.0 * (rho12 - rho6) + 1.0;
            }

            if (ABS(r_contact_1[i]) > conf_dist[i]) {

                // Distance tip is in the interaction zone of the wall
                double r_comp_mag = ABS(r_contact_1[i]);
                double delta_r = -1.0 * r_comp_mag + conf_dist[i] + r_cutoff;

                
                // Factors for WCA potential
                double rho2 = 1.0 / SQR(delta_r);
                double rho6 = CUBE(rho2);
                double rho12 = SQR(rho6);
                double factor = 24.0 * (2.0 * rho12 - rho6) * rho2;

                // Apply cutoff is potential is too strong
                if (factor * delta_r > f_cutoff)
                    factor = f_cutoff / delta_r;


                // Component of force exerted by wall on rod
                //f_1[i] = -factor * delta_r * r_contact_1[i]/r_comp_mag;
                f_1[i] = -factor * delta_r * SIGN(1.0, r_contact_1[i]) ;
                u += 4.0 * (rho12 - rho6) + 1.0;
            }
        }

        // Find torque on the system by wall
        cross_product(r_minus, f_0, tau_0, n_dim);
        cross_product(r_plus, f_1, tau_1, n_dim);


        // Add all calculated forces to the bonds properties
        for (int i = 0; i < parameters->n_dim; ++i) {
            f_bond[i_bond][i] += f_0[i];
            f_bond[i_bond][i] += f_1[i];
            t_bond[i_bond][i] += tau_0[i];
            t_bond[i_bond][i] += tau_1[i];
            // Only the tip cares about the NE stuff
            fbondtip_ne[i_bond][i] += f_1[i];
        }
        calc_matrix[i_bond] = 1;

        if (properties->control.virial_flag) {
            //FIXME: We should measure the virial here? 
        }
    }
    return u;
}

/*Exponentially asymptotic wall potential*/
double soft_sphero_wall_potential_bd(system_parameters *parameters, /*{{{*/
                                     system_properties *properties, 
                                     double **f_bond, double **virial,
                                     double **t_bond, int *calc_matrix){ 

    double conf_rad = 0.5 * properties->unit_cell.h[0][0];
    double conf_rad2 = SQR(conf_rad);
    // Set the bond tip to zero force and use on JUST THE PLUS TIP
    double **fbondtip_ne = properties->bonds.fbondtip_ne;

    /* Zero vdw forces, potential energy and virial. */
    double u = 0.0;
    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));
    memset(fbondtip_ne[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    
    if (properties->control.virial_flag)
        memset(virial[0], 0, SQR(parameters->n_dim) * sizeof(double));

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        //double f_cutoff = 0.1 / parameters->delta * properties->bonds.gamma_par[i_bond];
        
        double r_mag2_0 = 0.0; double r_mag2_1 = 0.0;
        double r_contact_0[3], r_contact_1[3];
        for (int i = 0; i < parameters->n_dim; ++i) {
            //Position of the minus end relative to mt center
            r_contact_0[i] = -0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            //Position of the plus end relative to mt center
            r_contact_1[i] =  0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];

            r_mag2_0 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_0[i]);
            r_mag2_1 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_1[i]);
        }
        
        //If positive end is touching the nuclear envelope
        if (r_mag2_1 > conf_rad2) {
            double r_mag = sqrt(r_mag2_1);
            double delta_r = ABS(-r_mag + conf_rad);
            double factor = parameters->wall_force_factor;
            
            double f[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                f[i] = -factor *(1.0 - exp(-1.0*delta_r))*((properties->bonds.r_bond[i_bond][i] + r_contact_1[i]) / r_mag);
            }
            
            double tau[3];
            cross_product(r_contact_1, f, tau, parameters->n_dim);
            for (int i = 0; i < parameters->n_dim; ++i) {
                f_bond[i_bond][i] += f[i];
                // Only the tip cares about the NE stuff
                fbondtip_ne[i_bond][i] += f[i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += tau[i];
            }

            calc_matrix[i_bond] = 1;

            //If force_induced_catastrophe is not enabled have the rod shrink as soon as it encounters a wall
            check_wall_stability(parameters, &(properties->bonds.poly_state[i_bond]));
            
            if (properties->control.virial_flag) {
                /* FIXME: We should measure the virial here? */
            }
            
            u += factor*(delta_r + 8.0*(exp(delta_r/8.0) - 1.0));
        }
    }

    return u;

}/*}}}*/

/*Potential for a wall of constant force*/
double linear_sphero_wall_potential_bd(system_parameters *parameters, /*{{{*/
                                     system_properties *properties, 
                                     double **f_bond, double **virial,
                                     double **t_bond, int *calc_matrix){ 

    double conf_rad = 0.5 * properties->unit_cell.h[0][0];
    double conf_rad2 = SQR(conf_rad);
    // Set the bond tip to zero force and use on JUST THE PLUS TIP
    double **fbondtip_ne = properties->bonds.fbondtip_ne;

    /* Zero vdw forces, potential energy and virial. */
    double u = 0.0;
    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));
    memset(fbondtip_ne[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    
    if (properties->control.virial_flag)
        memset(virial[0], 0, SQR(parameters->n_dim) * sizeof(double));

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        //double f_cutoff = 0.1 / parameters->delta * properties->bonds.gamma_par[i_bond];
        
        double r_mag2_0 = 0.0; double r_mag2_1 = 0.0;
        double r_contact_0[3], r_contact_1[3];
        for (int i = 0; i < parameters->n_dim; ++i) {
            //Position of the minus end relative to mt center
            r_contact_0[i] = -0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            //Position of the plus end relative to mt center
            r_contact_1[i] =  0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];

            r_mag2_0 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_0[i]);
            r_mag2_1 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_1[i]);
        }

        //If positive end is touching the nuclear envelope
        if (r_mag2_1 > conf_rad2) {
            double r_mag = sqrt(r_mag2_1);
            double delta_r = ABS(-r_mag + conf_rad);
            double factor = parameters->wall_force_factor;
            
            double f[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                f[i] = -factor * (properties->bonds.r_bond[i_bond][i] + r_contact_1[i]) / r_mag;
            }
            
            double tau[3];
            cross_product(r_contact_1, f, tau, parameters->n_dim);
            for (int i = 0; i < parameters->n_dim; ++i) {
                f_bond[i_bond][i] += f[i];
                // Only the tip cares about the NE stuff
                fbondtip_ne[i_bond][i] += f[i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += tau[i];
            }

            calc_matrix[i_bond] = 1;

            //If force_induced_catastrophe is not enabled have the rod shrink as soon as it encounters a wall
            check_wall_stability(parameters, &(properties->bonds.poly_state[i_bond]));
            
            if (properties->control.virial_flag) {
                /* FIXME: We should measure the virial here? */
            }
            
            u += factor*delta_r;
        }
    }
    return u;

}/*}}}*/

/*Potential for a wall that acts like a Hookian spring*/
double quadratic_sphero_wall_potential_bd(system_parameters *parameters, /*{{{*/
                                     system_properties *properties, 
                                     double **f_bond, double **virial,
                                     double **t_bond, int *calc_matrix){
    double conf_rad = 0.5 * properties->unit_cell.h[0][0];
    double conf_rad2 = SQR(conf_rad);
    // Set the bond tip to zero force and use on JUST THE PLUS TIP
    double **fbondtip_ne = properties->bonds.fbondtip_ne;

    /* Zero vdw forces, potential energy and virial. */
    double u = 0.0;
    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));
    memset(fbondtip_ne[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    
    if (properties->control.virial_flag)
        memset(virial[0], 0, SQR(parameters->n_dim) * sizeof(double));

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        //double f_cutoff = 0.1 / parameters->delta * properties->bonds.gamma_par[i_bond];
        
        double r_mag2_0 = 0.0; double r_mag2_1 = 0.0;
        double r_contact_0[3], r_contact_1[3];
        for (int i = 0; i < parameters->n_dim; ++i) {
            //Position of the minus end relative to mt center
            r_contact_0[i] = -0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            //Position of the plus end relative to mt center
            r_contact_1[i] =  0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];

            r_mag2_0 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_0[i]);
            r_mag2_1 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_1[i]);
        }

        //If positive end is touching the nuclear envelope
        if (r_mag2_1 > conf_rad2) {
            double r_mag = sqrt(r_mag2_1);
            double delta_r = ABS(-r_mag + conf_rad);
            double factor = parameters->wall_force_factor;
            
            double f[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                f[i] = -factor *delta_r*(properties->bonds.r_bond[i_bond][i] + r_contact_1[i]) / r_mag;
            }
            
            double tau[3];
            cross_product(r_contact_1, f, tau, parameters->n_dim);
            for (int i = 0; i < parameters->n_dim; ++i) {
                f_bond[i_bond][i] += f[i];
                // Only the tip cares about the NE stuff
                fbondtip_ne[i_bond][i] += f[i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += tau[i];
            }

            calc_matrix[i_bond] = 1;

            //If force_induced_catastrophe is not enabled 
            //have the rod shrink as soon as it encounters a wall
            check_wall_stability(parameters, &(properties->bonds.poly_state[i_bond]));
            //if (parameters->force_induced_catastrophe_flag == 0 &&
                //parameters->dynamic_instability_flag) {
                //properties->bonds.poly_state[i_bond] = SHRINK;
            //}
            
            if (properties->control.virial_flag) {
                /* FIXME: We should measure the virial here? */
            }
            
            u += factor*SQR(delta_r);
        }
    }
    return u;
}/*}}}*/

//Non-Monotonic(NM) wall potential
//See "Formation and Interaction of Membrane Tubes"/*{{{*/
//Imre Derényi - Frank Jülicher - Jacques Prost - Phys. Rev. Lett. Physical Review Letters - 2002
double NM_sphero_wall_potential_bd(system_parameters *parameters, 
                                     system_properties *properties, 
                                     double **f_bond, double **virial,
                                     double **t_bond, int *calc_matrix){

    //double r0=3.5084; //Characteristic radius of protrusion
    //Constants from paper
    double a1=0.746; double a2=0.726; 
    double alpha1=0.347; double alpha2=3.691;

    double a = a1*a2; double b = sqrt(parameters->ne_ratio); double c = alpha1 + alpha2;

    double f0 = parameters->wall_force_factor;
    double xm = b*(((7.0/4.0)*M_PI) - c); //value of dr that gives max force

    double conf_rad = 0.5 * properties->unit_cell.h[0][0];
    double conf_rad2 = SQR(conf_rad);
    // Set the bond tip to zero force and use on JUST THE PLUS TIP
    double **fbondtip_ne = properties->bonds.fbondtip_ne;

    /* Zero vdw forces, potential energy and virial. */
    double u = 0.0;
    memset(f_bond[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    memset(t_bond[0], 0, properties->bonds.n_bonds * 3 * sizeof(double));
    memset(fbondtip_ne[0], 0, properties->bonds.n_bonds * parameters->n_dim * sizeof(double));
    
    if (properties->control.virial_flag)
        memset(virial[0], 0, SQR(parameters->n_dim) * sizeof(double));

    for (int i_bond = 0; i_bond < properties->bonds.n_bonds; ++i_bond) {
        
        double r_mag2_1 = 0.0;
        double r_contact_1[3];
        for (int i = 0; i < parameters->n_dim; ++i) {
            //Position of the minus end relative to mt center for free MTs
            //r_contact_0[i] = -0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            //r_mag2_0 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_0[i]);
            
            //Position of the plus end relative to mt center
            r_contact_1[i] =  0.5 * properties->bonds.length[i_bond] * properties->bonds.u_bond[i_bond][i];
            r_mag2_1 += SQR(properties->bonds.r_bond[i_bond][i] + r_contact_1[i]);
        }
        
        //If positive end is touching the nuclear envelope
        if (r_mag2_1 > conf_rad2) {
            double r_mag = sqrt(r_mag2_1);
            double delta_r = ABS(-r_mag + conf_rad);

            //Analytical approximation for non-monoticity. Does not go to zero at dr=0
            //double factor = 2.0*f0*a*exp(-1.0*delta_r/b) * cos( (delta_r/b)+c ) + f0;

            //Characteristic length chosen to be 1/30th the value of dr that gives max for function
            //double factor = (1-exp(-delta_r*30/xm))*(2.0*f0*a*exp(-1.0*delta_r/b) * cos( (delta_r/b)+c ) + f0);
            double factor = calc_non_monotonic_wall_force( parameters->ne_ratio, 
                                                           parameters->wall_force_factor, 
                                                           delta_r);
            
            double f[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                f[i] =  -1.0*factor* (properties->bonds.r_bond[i_bond][i] + r_contact_1[i]) / r_mag;
            }
            
            double tau[3];
            cross_product(r_contact_1, f, tau, parameters->n_dim);
            for (int i = 0; i < parameters->n_dim; ++i) {
                f_bond[i_bond][i] += f[i];
                // Only the tip cares about the NE stuff
                fbondtip_ne[i_bond][i] += f[i];
            }
            for (int i = 0; i < 3; ++i) {
                t_bond[i_bond][i] += tau[i];
            }

            calc_matrix[i_bond] = 1;

            //If force_induced_catastrophe is not enabled have the rod shrink as soon as it encounters a wall
            check_wall_stability(parameters, &(properties->bonds.poly_state[i_bond]));
            
            if (properties->control.virial_flag) {
                /* FIXME: We should measure the virial here? */
            }
            //FIXME figure out what the potential is 
            //u += f0*delta_r - a*b*f0*exp(-1.0*delta_r/b)*( cos(delta_r/b)*(cos(c) - sin(c)) + sin(delta_r/b)*(cos(c) + sin(c)) );
        }
    }
    return u;
}
