// Implementation of spatial helpers functions
#include <cmath>
#include <iostream>

#include "spatial_helpers.h"

// Prototypes for later use
void cross_product(double *a, double *b, double *c, int n_dim);
double dot_product(int n_dim, double *a, double *b);

// Create an axis angle representation for rotation between two vectors
void create_axisangle(double *theta, double *eaxis, double *a, double *b) {
    *theta = acos(dot_product(3, a, b));
    eaxis[0] = 0.0;
    eaxis[1] = 0.0;
    eaxis[2] = 0.0;
    if (fabs(*theta) > 1e-5) {
        cross_product(a, b, eaxis, 3);
        double norm_factor = sqrt(dot_product(3, eaxis, eaxis));
        for (int i = 0; i < 3; ++i) {
            eaxis[i] /= norm_factor;
        }
    }
}

// Create a quaternion from axis angle representation
void quaternion_from_axisangle(double *q, double theta, double *eaxis) {
    q[0] = cos(theta/2.);
    q[1] = eaxis[0] * sin(theta/2.);
    q[2] = eaxis[1] * sin(theta/2.);
    q[3] = eaxis[2] * sin(theta/2.);
}

// Recover the axis angle representation from the quaternion
void axisangle_from_quaternion(double *theta, double *eaxis, double *q) {
    *theta = 2. * atan2(sqrt(dot_product(3, &(q[1]), &(q[1]))), q[0]);
    eaxis[0] = 0.0;
    eaxis[1] = 0.0;
    eaxis[2] = 0.0;
    if (*theta != 0.0) {
        eaxis[0] = q[1]/sin(*theta/2.);
        eaxis[1] = q[2]/sin(*theta/2.);
        eaxis[2] = q[3]/sin(*theta/2.);
    }
}

// Rodrigues rotation formula to rotate v to vrot by e and theta
void rodrigues_axisangle(double *vrot, double *v, double theta, double *e) {
    double sintheta = sin(theta);
    double costheta = cos(theta);
    double e_cross_v[3] = {0.0};
    double e_dot_v = dot_product(3, e, v);
    cross_product(e, v, e_cross_v, 3);
    for (int i = 0; i < 3; ++i) {
        vrot[i] = costheta*v[i] + sintheta*e_cross_v[i] + (1 - costheta)*e_dot_v*e[i];
    }
}

// Get the quaternion between 2 vectors
void quaternion_between_vectors(double *q, double *a, double *b) {
    double na[3] = {0.0};
    double nb[3] = {0.0};
    // normalize start and end
    double norma = sqrt(dot_product(3, a, a));
    double normb = sqrt(dot_product(3, b, b));
    for (int i = 0; i < 3; ++i) {
        na[i] /= norma;
        nb[i] /= normb;
    }

    // Create the axis angle representation
    double theta;
    double e[3] = {0.0};
    create_axisangle(&theta, e, a, b);
    quaternion_from_axisangle(q, theta, e);

    return;
}

// Get the norm of a quaternion
double quaternion_norm(double *q) {
    return q[0]*q[0] + q[1]*q[1] + q[2]*q[2] + q[3]*q[3];
}

// Normalize the quaternion
void normalize_quaternion(double *q) {
    double qres = quaternion_norm(q);
    for (int i = 0; i < 4; ++i) {
        q[i] = q[i] / qres;
    }
}

// Multiply quaternions
void quaternion_multiply(double *t, double *q, double *r) {
    t[0] = r[0]*q[0] - r[1]*q[1] - r[2]*q[2] - r[3]*q[3];
    t[1] = r[0]*q[1] + r[1]*q[0] - r[2]*q[3] + r[3]*q[2];
    t[2] = r[0]*q[2] + r[1]*q[3] + r[2]*q[0] - r[3]*q[1];
    t[3] = r[0]*q[3] - r[1]*q[2] + r[2]*q[1] + r[3]*q[0];
}

// Generate a max distance (angular) between successive frames!
void rotate_towards(double *q1, double *q2, float maxAngle, double *q) {
    if (maxAngle < 0.001f) {
        q[0] = q1[0];
        q[1] = q1[1];
        q[2] = q1[2];
        q[3] = q1[3];
        return;
    }

    double cosTheta = dot_product(4, q1, q2);

    if (cosTheta > 0.9999f) {
        q[0] = q2[0];
        q[1] = q2[1];
        q[2] = q2[2];
        q[3] = q2[3];
        return;
    }

    // Avoid taking the long path
    if (cosTheta < 0) {
        q1[0] = -1.0 * q1[0];
        q1[1] = -1.0 * q1[1];
        q1[2] = -1.0 * q1[2];
        q1[3] = -1.0 * q1[3];
        cosTheta *= -1.0;
    }

    // Get the actual angle
    double angle = acos(cosTheta);

    // If we are only close, we have arrived
    if (angle < maxAngle) {
        q[0] = q2[0];
        q[1] = q2[1];
        q[2] = q2[2];
        q[3] = q2[3];
        return;
    }

    double fT = maxAngle / angle;
    angle = maxAngle;

    double result[4] = {0.0};
    result[0] = (sin((1.0 - fT) * angle) * q1[0] + sin(fT * angle) * q2[0]) / sin(angle);
    result[1] = (sin((1.0 - fT) * angle) * q1[1] + sin(fT * angle) * q2[1]) / sin(angle);
    result[2] = (sin((1.0 - fT) * angle) * q1[2] + sin(fT * angle) * q2[2]) / sin(angle);
    result[3] = (sin((1.0 - fT) * angle) * q1[3] + sin(fT * angle) * q2[3]) / sin(angle);

    double qres = quaternion_norm(result);
    for (int i = 0; i < 4; ++i) {
        q[i] = result[i] / qres;
    }
}

void print_quaternion(double *q) {
    std::cout << "q(" << q[0] << ", " << q[1] << ", " << q[2] << ", " << q[3] << ")\n";
}






