/* This routine checks for overlap between a sphere and a spherocylinder, which could be nonadditive
 (a different overlap criterion can be applied to every distinct pair of types).
 If skin > 0.0, "near" overlaps are also detected (this feature is used
 in constructing the neighbor lists).
 
 input: number of spatial dimensions (n_dim)
 number of periodic dimensions (n_periodic)
 unit cell matrix (h)
 neighbor list skin, if applicable (skin)
 overlap distance (diameter_12)
 real position of sphere (r_1)
 scaled position of sphere (s_1)
 real position of spherocylinder (r_2)
 scaled position of spherocylinder (s_2)
 director of spherocylinder (u_2)
 length of spherocylinder (length_2)
 
 output: flag indicating whether spherocylinders are overlapping;
 1 if so, 0 if not (return value). */

#include "bob.h"

int sphere_sphero_overlap(int n_dim, int n_periodic, double **h, double skin, double diameter_12,
                          double *r_1, double *s_1,
                          double *r_2, double *s_2, double *u_2, double length_2)
{
    int i, j;
    double half_length_2, dr2, dr_dot_u_2, mu, mu_mag, r_min2;
    static int first_call = 1;
    static double *ds, *dr, *r_min;

    /* Allocate memory for local arrays the first time the routine is called. */
    if (first_call) {
        ds = (double *) allocate_1d_array(n_dim, sizeof(double));
        dr = (double *) allocate_1d_array(n_dim, sizeof(double));
        r_min = (double *) allocate_1d_array(n_dim, sizeof(double));
        first_call = 0;
    }

    /* Compute various constants. */
    half_length_2 = 0.5 * length_2;

    /* Compute pair separation vector. */
    for (i = 0; i < n_periodic; ++i) {  /* First handle periodic subspace. */
        ds[i] = s_2[i] - s_1[i];
        ds[i] -= NINT(ds[i]);
    }
    for (i = 0; i < n_periodic; ++i) {
        dr[i] = 0.0;
        for (j = 0; j < n_periodic; ++j)
            dr[i] += h[i][j] * ds[j];
    }
    for (i = n_periodic; i < n_dim; ++i)        /* Then handle free subspace. */
        dr[i] = r_2[i] - r_1[i];

    /* Compute squared pair separation. */
    dr2 = 0.0;
    for (i = 0; i < n_dim; ++i)
        dr2 += SQR(dr[i]);

    /* If the pair separation exceeds the spherocylinder length plus 
       diameter plus the skin thickness, the sphere and spherocylinder don't overlap. */
    if (dr2 >= SQR(half_length_2 + diameter_12 + skin))
        return 0;

    /* Compute minimum distance (see Allen et al., Adv. Chem. Phys. 86, 1 (1993)).
       First consider a point and an infinitely long line. */
    dr_dot_u_2 = 0.0;
    for (i = 0; i < n_dim; ++i)
        dr_dot_u_2 += dr[i] * u_2[i];
    mu = -dr_dot_u_2;
    mu_mag = ABS(mu);

    /* Now take into account the fact that the line segment is of finite length. */
    if (mu_mag > half_length_2)
        mu = SIGN(half_length_2, mu);

    /* Calculate minimum distance between sphere and spherocylinder. */
    r_min2 = 0.0;
    for (i = 0; i < n_dim; ++i) {
        r_min[i] = dr[i] + mu * u_2[i];
        r_min2 += SQR(r_min[i]);
    }

    /* Return 1 if sphere and spherocylinder overlap, 0 if not. */
    return (r_min2 <= SQR(diameter_12 + skin));
}
