/* Molecular dynamics program to simulate mixtures of soft spheres and soft spherocylinders
 * in the presence of anchors (Spindle pole bodies) and kinetochores

   Command-line input: name of parameter file (param_file)

   Output: various thermodynamic quantities are written to standard output */

/*
 * Authors: Christopher Edelmaier, Robert Blackwell, Adam Lamson, Matt Glaser
 * Group: Betterton-Glaser-Hough-McIntosh group, CU Boulder
 * email: christopher.edelmaier@colorado.edu
 * Copyright (c) 2019, CU Biophysics Group
 * http://www.colorado.edu/physics-biophysics
 */

/* License:
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BY NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGE OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT, OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OF THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * */

/**
 * @file spindle_bd_mp.cpp
 * @brief Spindle Assembly Main Program
 */

#include "bob.h"
#include "graphics.h"
#include "data_collection.h"
#include "time.h"
#include "triangle_mesh.h"
#include <iostream>

using namespace std;
#include "parse_flags.h"

int main(int argc, char *argv[]) {
    clock_t t1,t2;
    t1 = clock();

    system_parameters parameters;
    system_properties properties;
    system_potential potential;
    FILE *f_posit, *f_thermo, *f_spb, *f_pole, *f_forces;
    char default_file[F_MAX], param_file[F_MAX];
    int n_dim, n_sites, n_anchors, n_posit, n_graph, n_thermo, graph_flag, grab_flag,
        dynamic_instability_flag, i_step,
        n_bonds, n_steps, n_print_percent;
    int i_final_frame;
    double **h, **r, **s, **r_bond, **u_bond, *length;
    double **ra, **ua;

    // New awesome way of initializing the programs
    run_options run_opts = parse_opts(argc, argv, 3);

    strcpy(default_file, run_opts.default_file.c_str());
    strcpy(param_file, run_opts.equil_file.c_str());

    /* Read in default parameters. */
    parse_parameters(default_file, &parameters);

    /* Read in run-specific input parameters. */
    parse_parameters(param_file, &parameters);

    /* Initialize variables and allocate memory. */
    init_spindle_bd_mp(&parameters, &properties, &potential);

    /* Set up shortcuts to data structures. */
    n_dim = parameters.n_dim;
    n_posit = parameters.n_posit;
    n_graph = parameters.n_graph;
    n_thermo = parameters.n_thermo;
    graph_flag = parameters.graph_flag;
    grab_flag = parameters.grab_flag;
    h = properties.unit_cell.h;
    n_sites = properties.sites.n_sites;
    r = properties.sites.r;
    s = properties.sites.s;
    r_bond = properties.bonds.r_bond;
    u_bond = properties.bonds.u_bond;
    length = properties.bonds.length;
    dynamic_instability_flag = parameters.dynamic_instability_flag;
    n_bonds = properties.bonds.n_bonds;
    n_steps = parameters.n_steps;
    n_anchors = properties.anchors.n_anchors;
    ra = properties.anchors.r_anchor;
    ua = properties.anchors.u_anchor;

    //DataCollection data(&parameters, &properties);
    //properties.data = &data;

    // Restart enabled
    if (run_opts.restart) {
        if (run_opts.posit_file_names.size() < 3) {
            std::cout << "Please provide the correct number to restart spindle_bd_mp\n";
            exit(1);
        }
        std::cout << "RESTART Enabled, starting from previously defined files!\n";
        std::string checkpointfile = run_opts.posit_file_names[0];
        std::cout << "   POSITIONS: " << checkpointfile << std::endl;
        //FILE *f_restart_posit = gfopen(checkpointfile.c_str(), "rb+");
        f_posit = gfopen(checkpointfile.c_str(), "rb+");
        properties.read_header_func(&parameters, &properties, f_posit);

        // Do some arithmetic to figure out how many frames there are, and which one we
        // are going to load, loop until we hit the required frame number
        int iframe = 0;
        for (iframe = 0; iframe < run_opts.startframe+1; ++iframe) {
            std::cout << "Reading posit frame: " << iframe << std::endl;
            read_positions_spindle(parameters.n_dim,
                                   properties.sites.n_sites,
                                   properties.anchors.n_anchors,
                                   &properties.time,
                                   properties.unit_cell.h,
                                   properties.sites.r,
                                   properties.anchors.r_anchor,
                                   properties.anchors.u_anchor,
                                   f_posit);
        }
        i_final_frame = iframe;

        f_thermo = gfopen("spindle_bd_mp.thermo", "w");
        f_pole = gfopen("spindle_bd_mp.pole.forces", "w");
        f_forces = gfopen("spindle_bd_mp.forces", "w");

        /* Update bond vectors. */
        update_bond_vectors(parameters.n_dim,
                            parameters.n_periodic,
                            properties.bonds.n_bonds,
                            properties.unit_cell.h,
                            properties.sites.s,
                            properties.sites.r,
                            properties.bonds.bond_site_1,
                            properties.bonds.bond_site_2,
                            properties.bonds.v_bond,
                            properties.bonds.u_bond,
                            properties.bonds.length,
                            properties.bonds.length2);
        update_bond_positions(parameters.n_dim,
                              parameters.n_periodic,
                              properties.bonds.n_bonds,
                              properties.unit_cell.h,
                              properties.unit_cell.h_inv,
                              properties.sites.r,
                              properties.bonds.bond_site_1,
                              properties.bonds.v_bond,
                              properties.bonds.r_bond,
                              properties.bonds.s_bond);
        update_bond_site_positions(parameters.n_dim,
                                   parameters.n_periodic,
                                   properties.bonds.n_bonds,
                                   properties.sites.n_sites,
                                   properties.unit_cell.h,
                                   properties.unit_cell.h_inv,
                                   properties.bonds.bond_site_1,
                                   properties.bonds.bond_site_2,
                                   properties.bonds.r_bond,
                                   properties.bonds.u_bond,
                                   properties.bonds.length,
                                   properties.sites.r,
                                   properties.sites.s);

        std::string xlink_restart = run_opts.posit_file_names[1];
        std::cout << "   XLINKS: " << xlink_restart << ".posit.stage().type()\n";
        properties.crosslinks.Restart(&parameters, &properties, iframe);

        std::string chromosome_restart = run_opts.posit_file_names[2];
        std::cout << "   CHROMOSOMES: " << chromosome_restart << ".posit\n";
        properties.chromosomes.Restart(&parameters, &properties, iframe);
    } else if (run_opts.configure) {
        if (run_opts.posit_file_names.size() < 3) {
            std::cout << "Please provide the correct number to restart spindle_bd_mp\n";
            exit(1);
        }
        std::cout << "CONFIGURE Enabled, starting from previously defined files!\n";
        std::string checkpointfile = run_opts.posit_file_names[0];
        std::string spbfile = "spb.final.posit";
        std::cout << "   POSITIONS: " << checkpointfile << std::endl;
        f_posit = gfopen(checkpointfile.c_str(), "rb");
        f_spb = gfopen(spbfile.c_str(), "rb");
        properties.read_header_func(&parameters, &properties, f_posit);
        // We are expecting a single frame ONLY that matches everything
        read_positions_spindle(parameters.n_dim,
                               properties.sites.n_sites,
                               properties.anchors.n_anchors,
                               &properties.time,
                               properties.unit_cell.h,
                               properties.sites.r,
                               properties.anchors.r_anchor,
                               properties.anchors.u_anchor,
                               f_posit);
        read_positions_spindle_pole_body(&parameters, &properties, f_spb);
        fclose(f_posit);
        fclose(f_spb);

        update_spindle_state(&parameters, &properties);

        std::string xlink_restart = run_opts.posit_file_names[1];
        std::cout << "   XLINKS: " << xlink_restart << ".posit.stage().type()\n";
        properties.crosslinks.Restart(&parameters, &properties, xlink_restart);

        std::string chromosome_restart = run_opts.posit_file_names[2];
        std::cout << "   CHROMOSOMES: " << chromosome_restart << ".posit\n";
        properties.chromosomes.Restart(&parameters, &properties, chromosome_restart);

        // Open the output files (first time)
        f_posit = gfopen("spindle_bd_mp.posit", "w");
        f_thermo = gfopen("spindle_bd_mp.thermo", "w");
        f_pole = gfopen("spindle_bd_mp.pole.forces", "w");
        f_spb = gfopen("spb.posit", "w");
        f_forces = gfopen("spindle_bd_mp.forces", "w");

        // Write the header information and time information
        properties.write_header_func(&parameters, &properties, f_posit);
        properties.time = 0.0;
        i_final_frame = 0;

        print_simulation_step(&parameters, &properties);

    } else {
        // Open the output files (first time)
        f_posit = gfopen("spindle_bd_mp.posit", "w");
        f_thermo = gfopen("spindle_bd_mp.thermo", "w");
        f_pole = gfopen("spindle_bd_mp.pole.forces", "w");
        f_spb = gfopen("spb.posit", "w");
        f_forces = gfopen("spindle_bd_mp.forces", "w");

        // Write the header information and time information
        properties.write_header_func(&parameters, &properties, f_posit);
        properties.time = 0.0;
        i_final_frame = 0;
    }

    #ifndef NOGRAPH
    /* Activate graphics if graph_flag == 1. */
    Graphics graphics;
    if (graph_flag == 1) {
        graphics.Init(&parameters, parameters.n_dim,
                      properties.unit_cell.h, n_anchors,
                      &(properties.chromosomes.tris_[0]));
        //Fix for retina display macs
        graphics.ResizeWindow(800,800);
        graphics.SetBoundaryType("sphere");
        graphics.tomogram_view_ = run_opts.tomogram;
        graphics.DrawLoop(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.crosslinks.n_types_,
                          properties.crosslinks.stage_0_xlinks_,
                          properties.crosslinks.stage_1_xlinks_,
                          properties.crosslinks.stage_2_xlinks_,
                          properties.anchors.n_anchors,
                          1,
                          properties.anchors.color_,
                          properties.anchors.r_anchor,
                          properties.anchors.u_anchor,
                          properties.anchors.v_anchor,
                          properties.anchors.w_anchor,
                          properties.anchors.diameter,
                          properties.chromosomes.nchromosomes_,
                          &properties.chromosomes);
    }
    #endif

    if (n_steps > 100) {
        n_print_percent = n_steps / 100;
    } else {
        n_print_percent = 1;
    }

    bool first_call_spb_flag = true;
    bool write_spb_header = true;
    for (i_step = 0; i_step < n_steps; ++i_step) {
        // Print out the current percentage complete
        if (i_step % n_print_percent == 0) {
            fprintf(stdout, "%d%% Complete\n", (int)(100 * (float)i_step / (float)n_steps));
            fflush(stdout);
        }

        if (i_step % n_thermo == 0)
            properties.control.virial_flag = 1;
        else
            properties.control.virial_flag = 0;

        properties.time += parameters.delta;
        properties.i_current_step = i_step;

        if (dynamic_instability_flag == 1) {
            dynamic_instability(&parameters, &properties, &potential);
        }

        /* Run crosslink Kinetic Monte Carlo cycle */
        properties.crosslinks.StepKMC(&parameters, &properties);

        // Run Chromosome Monte Carlo Cycle
        properties.chromosomes.StepKMC(&parameters, &properties);

        // Run Vesicles Monte Carlo Cycle
        properties.vesicles.StepKMC(&parameters, &properties);

        if (properties.time > parameters.wait_time) {
            if (first_call_spb_flag) {
                std::cout << "Time: " << properties.time << " turning on spb flag, "
                    << "step: " << i_step << std::endl;
                first_call_spb_flag = false;
            }
        }
        parameters.use_spb_flag = (properties.time > parameters.wait_time) ? 1 : 0;

        position_step_spindle_bd_mp(&parameters, &properties, &potential);


        /* Output to the posit file */
        if (i_step % n_posit == 0) {
            update_bond_site_positions_mp(n_dim, parameters.n_periodic, n_bonds,
                                       n_sites, h, properties.unit_cell.h_inv,
                                       properties.bonds.bond_site_1,
                                       properties.bonds.bond_site_2,
                                       r_bond, u_bond, length, r, s);

            write_positions_spb_dynamics(n_dim, n_sites, n_bonds, n_anchors,
                                         h, r, ra, ua,
                                         properties.time, f_posit);
            write_positions_spindle_pole_body(&parameters,
                                              &properties,
                                              f_spb,
                                              write_spb_header);
            
            fflush(f_posit);
            fflush(f_spb);
            properties.crosslinks.WriteState(&parameters, &properties);
            properties.chromosomes.WriteState(&parameters, &properties);

            fflush(stdout);

            //std::cout << "SubFrame: " << (int) i_step / n_posit << std::endl;
            //std::cout << "   Frame: " << i_final_frame << std::endl;
            i_final_frame++;

            //print_simulation_step(&parameters, &properties);
            //properties.crosslinks.PrintSimulationStep();
            //properties.chromosomes.PrintFrame();
        }

        //properties.crosslinks.ConsistencyCheck();
        //properties.chromosomes.ConsistencyCheck();
        #ifndef NOGRAPH
        /* Display configuration. */
        if (graph_flag && i_step % n_graph == 0) {
            graphics.Draw(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.crosslinks.n_types_,
                          properties.crosslinks.stage_0_xlinks_,
                          properties.crosslinks.stage_1_xlinks_,
                          properties.crosslinks.stage_2_xlinks_,
                          properties.anchors.n_anchors,
                          1,
                          properties.anchors.color_,
                          properties.anchors.r_anchor,
                          properties.anchors.u_anchor,
                          properties.anchors.v_anchor,
                          properties.anchors.w_anchor,
                          properties.anchors.diameter,
                          properties.chromosomes.nchromosomes_,
                          &properties.chromosomes);

            // Write out the information on the chromosomes kinetochore attachments
            //properties.chromosomes.PrintFrame();

            /* Record bmp image of frame */
            if (grab_flag) {
                grabber(graphics.width_fb_, graphics.height_fb_,
                        std::string(parameters.grab_file), (int) i_step/n_graph);
            }
        }
        #endif

        //data.CalculateValues();

        if (i_step % n_thermo == 0) {
            // Calculate stress and pressure tensors
            thermodynamics_sphero(&parameters, &properties);

            write_thermo_spindle(&parameters, &properties, &potential, f_thermo, f_pole, f_forces);
        }
    }

    std::cout << "Closing files and writing final configuration!\n";
    std::cout << "Final frame number (including the restart): " << i_final_frame << std::endl;

    /* Close output files. */
    fclose(f_posit);
    fclose(f_spb);
    fclose(f_thermo);
    fclose(f_forces);
    properties.crosslinks.CloseAll();
    properties.chromosomes.CloseAll();
    properties.vesicles.CloseAll();

    // Write out the final configuration of the spindle simulation
    // Update final particle positions
    update_bond_site_positions_mp(n_dim, parameters.n_periodic, n_bonds,
                                  n_sites, h, properties.unit_cell.h_inv,
                                  properties.bonds.bond_site_1,
                                  properties.bonds.bond_site_2,
                                  r_bond, u_bond, length, r, s);

    // Write location information
    f_posit = gfopen("spindle_bd_mp.final.posit", "w");
    properties.write_header_func(&parameters, &properties, f_posit);
    write_positions_spb_dynamics(n_dim, n_sites, n_bonds, n_anchors,
                                 h, r, ra, ua,
                                 properties.time, f_posit);
    bool dummy_var = true;
    f_spb = gfopen("spb.final.posit", "w");
    write_positions_spindle_pole_body(&parameters,
                                      &properties,
                                      f_spb,
                                      dummy_var);
    fflush(f_posit);
    fflush(f_spb);

    // Write out xlink informaiton to new filenames
    properties.crosslinks.xlink_filename_ = "crosslinks.final";
    properties.crosslinks.WriteState(&parameters, &properties);

    // Chromosomes too
    properties.chromosomes.ChangeFilename("chromosomes.final.posit");
    properties.chromosomes.WriteState(&parameters, &properties);

    /* Output runtime of code. */
    t2 = clock();
    float diff = (float)t2 - (float)t1;
    float seconds = diff / CLOCKS_PER_SEC;
    cout << "spindle_bd_mp.cpp runtime (seconds): " << seconds << endl;

    /* Normal termination. */
    //exit(0);
    std::cout << "Successful run!\n";
    return 0;
}
