#ifndef _STRUCTURES_H
#define _STRUCTURES_H
/* Structure definitions for bob (bunch o' balls/beads/bodies/blobs). */

/* Include parameters structure, which is generated automatically from from a
 parameter configuration file using the utility routine configure_parameters. */


#include "cell_list.h"
#include <vector>
#include "anchor_list.h"
#include "neighbors.h"
#include "parameters.h"

class DataCollection;

typedef enum {
    SHRINK = 0,
    GROW = 1,
    PAUSE
} poly_state_enum;

typedef struct {
    int n_dim;
    int *n_bin;
    int n_linear;
    
    double *bin_size;

    double **h;
    double *dist;
} correlation_data;

typedef struct {
    int n_blocks;
    double *mean;
    double *var;
    double *var_of_mean;
    double *var_of_mean_err;
} blocking_data;

/* The <line_list> structure contains a list of vertices and the first element
   of that list. */

typedef struct {
    int n_elements;
    int max_elements;
    int curr_head;

    double **s;
} line_list;

typedef struct {
    int n_bin[3], 
        n_linear,
        n_bin_axes[3],
        n_linear_axes,
        n_bin_radial,
        n_bin_struct[3],
        n_linear_struct,
        n_bin_cl_dens[2],
        n_linear_cl_dens,
        n_bin_local_order,
        n_bin_contact,
        n_bin_p_i,
        n_bin_theta;
        
    FILE *f_polarity,
        *f_order, 
        *f_length,
        *f_op,
        *f_cl_dist,
        *f_autocorr_u,
        *f_autocorr_p,
        *f_diffusion;

    double curr_director[3];
    
    double *pair_dist, 
        *pair_dist_radial, 
        *pair_dist_axes,
        *pair_dist_nem_par,
        *p1_nem_par,
        *p1_nem_antipar,
        *p2_nem_par,
        *p2_nem_antipar,
        *pair_dist_nem_antipar,
        *pair_dist_axes_par,
        *pair_dist_axes_antipar,
        *pair_dist_polar_order,
        *pair_dist_xl_axes_par,
        *pair_dist_xl_axes_antipar,
        ***pair_dist_vir_nem_par,
        ***pair_dist_vir_nem_antipar,
        ***pair_dist_vir_axes_par,
        ***pair_dist_vir_axes_antipar,
        *p1_axes,
        *p2_axes, 
        *p1_radial,
        *p2_radial,
        *cl_axes_par,
        *cl_axes_naught_par,
        *cl_axes_antipar, 
        *cl_axes_naught_antipar,
        *cl_dens_par,
        *cl_dens_antipar,
        *struct_factor,
        *local_order_dist,
        *dP_of_p,
        *dP_cl_of_p,
        *cl_xx_of_p,
        *cl_xx_of_p_par,
        *cl_xx_of_p_antipar,
        *cl_yy_of_p,
        *cl_yy_of_p_par,
        *cl_yy_of_p_antipar,
        *dP_sphero_of_p,
        *sphero_xx_of_p,
        *sphero_xx_of_p_par,
        *sphero_xx_of_p_antipar,
        *sphero_yy_of_p,
        *sphero_yy_of_p_par,
        *sphero_yy_of_p_antipar,
        *n_links_of_p,
        *n_neighbs_of_p,
        *v_par_of_p,
        *theta_dist,
        *contact_dist,
        *p_i_dist,
        *contact_p_i_dist,
        *contact,
        *p_i;

    double **virial_par_plus_wca,
        **virial_antipar_plus_wca,
        **virial_par_minus_wca,
        **virial_antipar_minus_wca,
        **virial_par_plus_cl,
        **virial_antipar_plus_cl,
        **virial_par_minus_cl,
        **virial_antipar_minus_cl;

    double ***virial_site_wca_antipar,
        ***virial_site_wca_par,
        ***virial_site_cl_par,
        ***virial_site_cl_antipar;

    double delta_par, 
        delta_perp,
        r_max_axes,
        n_interacting_par,
        n_interacting_antipar;

    correlation_data cl_dist_dy_par;
    correlation_data cl_extension_dy_par;
    correlation_data cl_dist_dy_antipar;
    correlation_data cl_extension_dy_antipar;
    correlation_data cl_extension_dist_axes_par;
    correlation_data cl_extension_dist_axes_antipar;
    correlation_data cl_extension_dist_par;
    correlation_data cl_extension_dist_antipar;
    

    int *N_ij;
    int *n_neighbs;

    double *Q_ij;

    fftw_complex *in, *out;
    fftw_plan p;
} analysis_properties;

typedef struct {
    int n_anchors;
    int interaction_type_int; /* SPB interaction type, 0=WCA, 1=LJ126 */

    double spb_interaction_strength; /* SPB LJ126 interaction strength in units of epsilon */
    double spb_interaction_range; /* SPB LJ126 interaction stregnth in units of sigma */

    double **f_anchor,		/* Force on the anchor */
        **r_anchor,		/* Global position of anchor */
	    **u_anchor,		/* Reference direction for anchor, points toward
                                   center of spb (-r_anchor/R) */
        **v_anchor,             /* Reference direction for anchor, arbitrary but
                                   perpendicular to u_anchor */
        **w_anchor,             /* Reference direction for anchor, arbitrary but
                                   perpendicular to u_anchor and v_anchor (rhr)  */
        **tau_local,		/* Internal torque on the anchor, about its axis */
        *gamma_tra,		/* Translational drag */
        *gamma_rot,		/* Rotational drag */
        *diameter,		/* Diameter for interaction between two SPBs */
        *k,                     /* Bond attachment spring constant */
        *attach_diameter,       /* Diameter which MTs can attach */
        *r0,			/* Bond attachment rest length */
        *kr;            // Angular spring constant
    float **color_; // Color of the SPBs
    
    al_list *anchor_list;
} anchor_properties;

/* The <unit_cell_properties> structure contains attributes of the computational unit cell. */
typedef struct {

    double **h,                 /* Unit cell matrix. */
    **h_inv,                    /* Inverse unit cell matrix. */
    **a,                        /* Array of unit cell vectors. */
    **b,                        /* Array of reciprocal unit cell vectors. */
    *a_perp,                    /* Orthogonal dimensions of unit cell. */
    **fh, **dvdh, volume;       /* Volume of unit cell. */
    int **scaling_matrix,       /* Unit cell scaling matrix (for NPT/NPH MD and MC). */
     n_scaling_sets;            /* Number of coupled sets of scaling matrix elements. */

} unit_cell_properties;

/* The <site_properties> structure contains attributes of sites. */

typedef struct {

    int n_sites;                /* Total number of sites. */

    double **r,                 /* Array of site coordinates. */
    **s,                        /* Array of scaled site coordinates. */
    **dr_tot,                   /* Array of site displacements since last neighbor list update. */
    **r_rel,                    /* Array of relative site coordinates. */
    **r_body,                   /* Array of site coordinates (body-fixed). */
    *frac_posit,                /* Array of fractional positions of sites along bonds. */
    **v,                        /* Array of site velocities. */
    **f,                        /* Array of site forces. */
    **f_con,                    /* Array of constraint forces. */
    **fs, *m,                   /* Array of site masses. */
    *q,                         /* Array of site charges. */
    *sigma,                     /* Array of length parameters. */
    *epsilon,                   /* Array of energy parameters. */
    *diameter,                  /* Array of site diameters. */
    *u_site;                    /* Array of site potential energies. */

    int *mol_label,             /* Array of labels of molecules to which sites belong. */
    *bond_label,                /* Array of labels of bonds to which sites belong. */
    *intra_label,               /* Array of labels of sites within molecules. */
    *type_index,                /* Array of site type indices. */
    *n_site_bonds,              /* Array of number of bonds connected to sites. */
    **site_bond,                /* Array of labels of bonds connected to sites. */
    *floop;                     /* Array of 'floop' states of sites. */

} site_properties;

/* The <rng_properties> structure contains the state information for the gsl random number generator */
typedef struct {
    gsl_rng *r;
    const gsl_rng_type *T;
} rng_properties;

/* The <bond_properties> structure contains attributes of bonds. */
typedef struct {

    int n_bonds;                /* Total number of bonds. */

    double **r_bond,            /* Array of bond positions. */
    **s_bond,                   /* Array of scaled bond positions. */
    **r_rel,                    /* Array of relative bond positions. */
    **v_bond,                   /* Array of bond vectors. */
    **u_bond,                   /* Array of unit bond vectors. */
    **f_bond,                   /* Array of force vectors */
    **t_bond,                   /* Array of torques */
    *local_order,               /* Array measuring local polar order */
    *length,                    /* Array of bond lengths. */
    *length2,                   /* Array of squared bond lengths. */
    *diameter,                  /* Array of bond diameters. */
    *r_equil,                   /* Array of equilibrium bond lengths. */
    *k_stretch,                 /* Array of spring constants. */
    *sigma,                     /* Array of LJ length parameters. */
    *epsilon,                   /* Array of LJ energy parameters. */
    *gamma_perp, *gamma_par, *gamma_rot, 
    *diff_perp, *diff_par, *diff_rot,
    *length_pos,                /* Array of starting positions of sum of total lengths */
     total_length,              /* total length of all bonds currently in system */
     length_max;                /* Maximum bond length. */

    double **fbondtip_ne;       // Tip force on the bond from the nuclear envelope


    line_list *trajectory;      /* History of bond positions */

    int *mol_label,             /* Array of labels of molecules to which bonds belong. */
    *intra_label,               /* Array of intramolecular bond labels. */
    *bond_site_1,               /* Arrays of bond endpoint site labels. */
    *bond_site_2, *type_index;  /* Array of bond type indices. */

    poly_state_enum *poly_state; /* Array of bond polymerization states. see poly_state_enum */

    int *stab_state;   /* Array of bond stability state. Either stabilized or not */

    rng_properties *rng_local; // Array of local random number generators for gsl calcs

} bond_properties;

typedef struct {
    int n_bonds;                /* Total number of bonds. */

    double **r_bond,            /* Array of bond positions. */
    **s_bond,                   /* Array of scaled bond positions. */
     length_max;                /* Maximum bond length. */

    int *parent_bond;           /* Array of labels of parent bond to which child bonds belong. */

    nl_entry **search_head;
    nl_entry **search_tail;

} child_bond_properties;

/* The <thermo_properties> structure contains the thermodynamic properties of the system. */
typedef struct {

    double u,                   /* Potential energy. */
     t,                         /* Translational kinetic energy. */
     e,                         /* Total energy. */
     t_mol,                     /* Molecular center of mass kinetic energy. */
     temp,                      /* Site temperature. */
     temp_mol,                  /* Molecular kinetic energy. */
     t_rot,                     /* Rotational kinetic energy. */
    *angular_moment,            /* Angular momentum. */
     volume,                    /* Volume (of periodic subspace?). */
     dens,                      /* Number density. */
    **virial,                   /* Virial. */
    **virial_con,               /* Constraint contribution to virial. */
    ***virial_xlink,            /* Xlink contribution to virial per species. */
    **stress,                   /* Stress tensor. */
    **press_tensor,             /* Pressure tensor. */
     press_iso;                 /* Isotropic pressure. */

} thermo_properties;

/* The <control_structure> structure contains flags and switches that control various aspects of
 program execution. */
typedef struct {

    int bond_vector_flag,       /* 1 if array of bond vectors needs updating, 0 otherwise. */
        bond_position_flag,     /* 1 if array of bond positions needs updating, 0 otherwise. */
        inter_site_flag,        /* 1 if array of interaction site positions needs updating, 0 otherwise. */
        cell_list_flag,         /* 1 if cell lists need updating, 0 otherwise. */
        neighbor_list_flag,     /* 1 if neighbor lists need checking, 0 otherwise. */
        output_local_order_flag,/* 1 if local order distribution vs time needs to be output, 0 otherwise. */
        calc_interaction_flag,  /* 1 if analysis needs to calculation pair interactions, 0 otherwise. */
        pair_dist_flag,         /* 1 if any pair distribution function needs measuring, 0 otherwise. */
        output_unrolled_flag,   /* 1 if we want to make an unrolled.posit file, 0 otherwise */
        crosslink_input_flag,   /* 1 if crosslink trajectory needs to be read, 0 otherwise */
        virial_flag;            /* 1 if virial needs updating, 0 otherwise */
    
} control_structure;

typedef struct {
    int n_threads;
    rng_properties *rng;
} rng_mt_properties;

typedef struct {
    int n_threads;
    double ***f_local;
    double ***t_local;
    double ***virial_local;
    double ****virial_local_xlink;
    int **calc_local;
} mp_local_properties;

/* The <system_properties> structure contains a variety of properties 
 of the simulated system and its constituents. */
#ifndef _SYSTEM_PROPERTIES
#define _SYSTEM_PROPERTIES
typedef struct system_properties system_properties;
#include "chromosome_management.h"
#include "vesicle_management.h"
#include "xlink_management.h"

struct system_properties {
    XlinkManagement crosslinks;

    ChromosomeManagement chromosomes;

    VesicleManagement vesicles;

    unit_cell_properties unit_cell;

    site_properties sites;      /* Note: later we should create an array of
                                   structures to support multiple sets of
                                   sites. */

    bond_properties bonds;      /* Note: later we should create an array of
                                   structures to support multiple sets of
                                   bonds. */

    child_bond_properties child_bonds;  /* child bond structure for dividing
                                           long rods into smaller ones, used in
                                           child cell search routine */

    thermo_properties thermo;

    double time;
    int i_current_step;

    int n_cell_lists;
    cell_list *cells;

    neighbor_list neighbors;

    DataCollection *data;

    control_structure control;

    void (*read_header_func) (system_parameters *,
                              system_properties *,
                              FILE *);
    void (*write_header_func) (system_parameters *,
                               system_properties *,
                               FILE *);
    
    rng_properties rng;
    
    rng_mt_properties rng_mt;

    mp_local_properties mp_local;

    anchor_properties anchors;
};
#endif

/* The <system_potential> structure contains data associated with
 evaluation of the potential energy expression. */
typedef struct {
    int n_comp,                 /* Number of potential energy/force components. */
        *calc_matrix;           /* For BD: n_bond array detailing if a force is currently
                                   acting on the bond. 
                                   0: no force
                                   1: force */

    double (**pot_func) (system_parameters *,
                         system_properties *,
                         double **,
                         double **,
                         double **,
                         int *);     /* Array of pointers to potential energy/force evaluation functions. */

    double *u_comp,             /* Array of potential energy components. */
    ***f_comp,                  /* Array of force components. */
    ***virial_comp,             /* Array of virial components. */
    ***t_comp;                 /* For BD: Array of integrated torques. */

} system_potential;
#endif
