// Implementation of testing the chromatin potential

#include "bob.h"
#include "helpers.h"
#include "kinetochore.h"
#include "test_chromatin_potential.h"

#include <iostream>

void TestChromatinPotential::RunTests() {
    std::cout << "****************\n";
    std::cout << "Test Chromatin Potential run tests\n";
    std::cout << "****************\n";

    for (auto &kv : tests_) {
        std::cout << "----------------\n";
        std::cout << "Test : " << kv.first << std::endl;
        std::cout << "----------------\n";
        auto result = kv.second();
        if (!result) {
            std::cout << "Test : " << kv.first << " failed, check output!\n";
            exit(1);
        }
    }
}

void TestChromatinPotential::SetTests() {
    YAML::Node node = YAML::LoadFile(filename_);

    if (node["BasicLinearTest"]) {
        tests_["BasicLinearTest"] = std::bind(&TestChromatinPotential::BasicLinearTest, this);
    }
    if (node["BasicAngularTest"]) {
        tests_["BasicAngularTest"] = std::bind(&TestChromatinPotential::BasicAngularTest, this);
    }
    if (node["TestRandomSetup"]) {
        tests_["TestRandomSetup"] = std::bind(&TestChromatinPotential::TestRandomSetup, this);
    }
    if (node["BasicAngularTestV"]) {
        tests_["BasicAngularTestV"] = std::bind(&TestChromatinPotential::BasicAngularTestV, this);
    }
}

void TestChromatinPotential::InitSystem() {
    seed_++;
    init_default_params(&parameters_);

    parameters_.n_dim = 3;
    parameters_.n_periodic = 0;
    parameters_.delta = 0.001;
    parameters_.temp = 1.0;
    // Init h by hand
    properties_.unit_cell.h = (double**) allocate_2d_array(parameters_.n_dim, parameters_.n_dim, sizeof(double));
    for (int i = 0; i < parameters_.n_dim; ++i) {
        properties_.unit_cell.h[i][i] = 100;
    }

    gsl_rng_env_setup();
    properties_.rng.T = gsl_rng_default;
    properties_.rng.r = gsl_rng_alloc(properties_.rng.T);
    gsl_rng_set(properties_.rng.r, seed_);
    
    init_unit_cell_structure(&parameters_, &properties_);

    // Initialize the chromosomes
    properties_.chromosomes.Init(&parameters_,
                                 &properties_,
                                 testfile_.c_str(),
                                 NULL);
    fake_array_ = (double***) allocate_3d_array(1, 0, 3, sizeof(double));

    // Save off position information
    Kinetochore *kA = &(properties_.chromosomes.kinetochores_[0]);
    Kinetochore *kB = &(properties_.chromosomes.kinetochores_[1]);
    for (int i = 0; i < parameters_.n_dim; ++i) {
        rA[i] = kA->r_[i];
        uA[i] = kA->u_[i];
        vA[i] = kA->v_[i];
        wA[i] = kA->w_[i];
        rB[i] = kB->r_[i];
        uB[i] = kB->u_[i];
        vB[i] = kB->v_[i];
        wB[i] = kB->w_[i];
    }
}

bool TestChromatinPotential::BasicLinearTest() {
    bool success = false;
    testfile_ = "test_chromosomes_linear.yaml";
    properties_.chromosomes.print_info_ = false;

    InitSystem();

    double uret = chromosome_chromatin_potential(&parameters_,
                                                 &properties_,
                                                 fake_array_[0],
                                                 fake_array_[0],
                                                 fake_array_[0],
                                                 NULL);
    
    success = TestForceTorqueBalance(uret);

    PrintUpdateData();

    success = success && TestDeltas();

    return success;
}

bool TestChromatinPotential::BasicAngularTest() {
    bool success = false;
    testfile_ = "test_chromosomes_angular.yaml";
    properties_.chromosomes.print_info_ = false;

    InitSystem();

    double uret = chromosome_chromatin_potential(&parameters_,
                                                 &properties_,
                                                 fake_array_[0],
                                                 fake_array_[0],
                                                 fake_array_[0],
                                                 NULL);


    success = TestForceTorqueBalance(uret);

    PrintUpdateData();

    success = success && TestDeltas();

    return success;
}

bool TestChromatinPotential::BasicAngularTestV() {
    bool success = false;
    testfile_ = "test_chromosomes_angularV.yaml";
    properties_.chromosomes.print_info_ = false;

    InitSystem();

    double uret = chromosome_chromatin_potential(&parameters_,
                                                 &properties_,
                                                 fake_array_[0],
                                                 fake_array_[0],
                                                 fake_array_[0],
                                                 NULL);


    success = TestForceTorqueBalance(uret);

    PrintUpdateData();

    success = success && TestDeltas();

    return success;
}

bool TestChromatinPotential::TestRandomSetup() {
    bool success = true;
    int n = 1000;
    testfile_ = "test_chromosomes_random.yaml";
    properties_.chromosomes.print_info_ = false;

    for (int i = 0; i < n; ++i) {
        InitSystem();
        double uret = chromosome_chromatin_potential(&parameters_,
                                                     &properties_,
                                                     fake_array_[0],
                                                     fake_array_[0],
                                                     fake_array_[0],
                                                     NULL);
        success = success && TestForceTorqueBalance(uret, false);
        if (!success)
            break;
    }

    return success;
}

bool TestChromatinPotential::TestForceTorqueBalance(double uret, bool print_info) {
    // tA + tB + r x f = 0

    bool success = true;
    int n_dim = parameters_.n_dim;
    double r[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        r[i] = properties_.chromosomes.r_[0][i] - 
               properties_.chromosomes.r_[1][i];
    }
    double f[3] = {0.0};
    double tA[3] = {0.0};
    double tB[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        f[i] = properties_.chromosomes.f_[0][i]; 
        tA[i] = properties_.chromosomes.t_[0][i];
        tB[i] = properties_.chromosomes.t_[1][i];
    }

    if (print_info) {
        std::cout << "energy  [" << uret << "]\n";
        std::cout << "force   (" << f[0] << ", " << f[1] << ", " << f[2] << ")\n";
        std::cout << "torque A(" << tA[0] << ", " << tA[1] << ", " << tA[2] << ")\n";
        std::cout << "torque B(" << tB[0] << ", " << tB[1] << ", " << tB[2] << ")\n";
    }

    double rcrossf[3] = {0.0};
    cross_product(r, f, rcrossf, n_dim);

    double finalans[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        finalans[i] = tA[i] + tB[i] + rcrossf[i];
    }
    if (print_info) {
        std::cout << "final   (" << finalans[0] << ", " << finalans[1] << ", " << finalans[2] << ")\n";
    }

    if (fabs(finalans[0]) > 1e-10 ||
        fabs(finalans[1]) > 1e-10 ||
        fabs(finalans[2]) > 1e-10) {
        std::cout << "Force balance didin't work correctly, failing test!\n";
        success = false;
    }
    return success;
}

bool TestChromatinPotential::TestDeltas() {
    bool success = true;
    // Just write out the deltas of the change in positions
    // final - initial
    double dr[3] = {0.0};
    double du[3] = {0.0};
    double dv[3] = {0.0};
    double dw[3] = {0.0};

    YAML::Node node = YAML::LoadFile(testfile_);

    std::cout << "kA deltas: \n";
    Kinetochore *kA = &(properties_.chromosomes.kinetochores_[0]);
    Kinetochore *kB = &(properties_.chromosomes.kinetochores_[1]);
    for (int i = 0; i < parameters_.n_dim; ++i) {
        dr[i] = kA->r_[i] - rA[i];
        du[i] = kA->u_[i] - uA[i];
        dv[i] = kA->v_[i] - vA[i];
        dw[i] = kA->w_[i] - wA[i];
    }
    std::cout << "  drA(" << dr[0] << ", " << dr[1] << ", " << dr[2] << ")\n";
    std::cout << "  duA(" << du[0] << ", " << du[1] << ", " << du[2] << ")\n";
    std::cout << "  dvA(" << dv[0] << ", " << dv[1] << ", " << dv[2] << ")\n";
    std::cout << "  dwA(" << dw[0] << ", " << dw[1] << ", " << dw[2] << ")\n";

    // Test against results
    for (int i = 0; i < parameters_.n_dim; ++i) {
        if (fabs(dr[i] - node["results"]["deltas"]["kA"]["dr"][i].as<double>()) > 1e-10) {
            std::cout << "delta: " << fabs(dr[i] - node["results"]["deltas"]["kA"]["dr"][i].as<double>()) << std::endl;
            success = success && false;
        }
        if (fabs(du[i] - node["results"]["deltas"]["kA"]["du"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dv[i] - node["results"]["deltas"]["kA"]["dv"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dw[i] - node["results"]["deltas"]["kA"]["dw"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
    }
    if (!success) {
        std::cout << "Kinetochore A failed delta test!\n";
        return success;
    }

    std::cout << "kB deltas: \n";
    for (int i = 0; i < parameters_.n_dim; ++i) {
        dr[i] = kB->r_[i] - rB[i];
        du[i] = kB->u_[i] - uB[i];
        dv[i] = kB->v_[i] - vB[i];
        dw[i] = kB->w_[i] - wB[i];
    }
    std::cout << "  drB(" << dr[0] << ", " << dr[1] << ", " << dr[2] << ")\n";
    std::cout << "  duB(" << du[0] << ", " << du[1] << ", " << du[2] << ")\n";
    std::cout << "  dvB(" << dv[0] << ", " << dv[1] << ", " << dv[2] << ")\n";
    std::cout << "  dwB(" << dw[0] << ", " << dw[1] << ", " << dw[2] << ")\n";

    // Test against results
    for (int i = 0; i < parameters_.n_dim; ++i) {
        if (fabs(dr[i] - node["results"]["deltas"]["kB"]["dr"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(du[i] - node["results"]["deltas"]["kB"]["du"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dv[i] - node["results"]["deltas"]["kB"]["dv"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dw[i] - node["results"]["deltas"]["kB"]["dw"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
    }
    if (!success) {
        std::cout << "Kinetochore B failed delta test!\n";
        return success;
    }

    return success;
}

void TestChromatinPotential::PrintUpdateData() {
    std::cout << "Starting configuration: " << std::endl;
    properties_.chromosomes.PrintPositions(0);
    properties_.chromosomes.UpdatePositions();
    std::cout << "Final configuration: " << std::endl;
    properties_.chromosomes.PrintPositions(0);
}
