// Implementation of testing the chromatin potential

#include "bob.h"
#include "helpers.h"
#include "kinetochore.h"
#include "test_chromosomes.h"

#include <iostream>

void TestChromosomes::RunTests() {
    std::cout << "****************\n";
    std::cout << "Test Chromosomes run tests\n";
    std::cout << "****************\n";

    for (auto &kv : tests_) {
        std::cout << "----------------\n";
        std::cout << "Test : " << kv.first << std::endl;
        std::cout << "----------------\n";
        auto result = kv.second();
        if (!result) {
            std::cout << "Test : " << kv.first << " failed, check output!\n";
            exit(1);
        }
    }
}

void TestChromosomes::SetTests() {
    YAML::Node node = YAML::LoadFile(filename_);

    if (node["TestChromosomesKMCInit"]) {
        tests_["TestChromosomesKMCInit"] = std::bind(&TestChromosomes::TestChromosomesKMCInit, this);
    }
}

void TestChromosomes::InitSystem() {
    seed_++;
    init_default_params(&parameters_);

    ndim_ = 3;
    parameters_.n_dim = ndim_;
    parameters_.n_periodic = 0;
    parameters_.delta = 0.001;
    parameters_.temp = 1.0;

    // Init h by hand
    properties_.unit_cell.h = (double**) allocate_2d_array(parameters_.n_dim, parameters_.n_dim, sizeof(double));
    for (int i = 0; i < parameters_.n_dim; ++i) {
        properties_.unit_cell.h[i][i] = 100;
    }

    gsl_rng_env_setup();
    properties_.rng.T = gsl_rng_default;
    properties_.rng.r = gsl_rng_alloc(properties_.rng.T);
    gsl_rng_set(properties_.rng.r, seed_);

    // Create the microtubules
    YAML::Node node = YAML::LoadFile(testfile_);
    nmts_ = node["mt"].size();
    parameters_.n_spheros = properties_.bonds.n_bonds = nmts_;

    // Init the structures for the bonds
    {
        init_unit_cell_structure(&parameters_, &properties_);
        int nsites = 2 * nmts_;
        properties_.sites.v = (double**) allocate_2d_array(nsites, ndim_, sizeof(double));
        properties_.sites.r = (double**) allocate_2d_array(nsites, ndim_, sizeof(double));
        init_site_structure_sphero(&parameters_, &properties_);
        init_bond_structure_sphero(&parameters_, &properties_);
    }

    for (int imt = 0; imt < nmts_; ++imt) {
        for (int i = 0; i < ndim_; ++i) {
            properties_.bonds.r_bond[imt][i] = node["mt"][imt]["r"][i].as<double>();
            properties_.bonds.u_bond[imt][i] = node["mt"][imt]["u"][i].as<double>();
        }
        properties_.bonds.length[imt] = node["mt"][imt]["l"].as<double>();
    }

    // Print the bond information
    for (int imt = 0; imt < nmts_; ++imt) {
        std::cout << "New Microtubule:\n";
        std::cout << "  r(" << properties_.bonds.r_bond[imt][0] << ", "
                            << properties_.bonds.r_bond[imt][1] << ", "
                            << properties_.bonds.r_bond[imt][2] << ")\n";
        std::cout << "  u(" << properties_.bonds.u_bond[imt][0] << ", "
                            << properties_.bonds.u_bond[imt][1] << ", "
                            << properties_.bonds.u_bond[imt][2] << ")\n";
        std::cout << "  l " << properties_.bonds.length[imt] << std::endl;
    }

    // Initialize the chromosomes
    properties_.chromosomes.Init(&parameters_,
                                 &properties_,
                                 testfile_.c_str(),
                                 NULL);

    // Potential needs some additional information
    f_comp_ = (double **) allocate_2d_array(nmts_, ndim_, sizeof(double));
    t_comp_ = (double **) allocate_2d_array(nmts_, ndim_, sizeof(double));
    calc_matrix_ = (int *) allocate_1d_array(nmts_, sizeof(int));

    // Save off position information
    Kinetochore *kA = &(properties_.chromosomes.kinetochores_[0]);
    Kinetochore *kB = &(properties_.chromosomes.kinetochores_[1]);
    for (int i = 0; i < parameters_.n_dim; ++i) {
        rA[i] = kA->r_[i];
        uA[i] = kA->u_[i];
        vA[i] = kA->v_[i];
        wA[i] = kA->w_[i];
        rB[i] = kB->r_[i];
        uB[i] = kB->u_[i];
        vB[i] = kB->v_[i];
        wB[i] = kB->w_[i];
    }

}

bool TestChromosomes::TestForceTorqueBalance(double uret, bool print_info) {
    // tA + tB + r x f = 0

    bool success = true;
    int n_dim = parameters_.n_dim;
    double r[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        r[i] = properties_.chromosomes.r_[0][i] - 
               properties_.chromosomes.r_[1][i];
    }
    double f[3] = {0.0};
    double tA[3] = {0.0};
    double tB[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        f[i] = properties_.chromosomes.f_[0][i]; 
        tA[i] = properties_.chromosomes.t_[0][i];
        tB[i] = properties_.chromosomes.t_[1][i];
    }

    if (print_info) {
        std::cout << "energy  [" << uret << "]\n";
        std::cout << "force   (" << f[0] << ", " << f[1] << ", " << f[2] << ")\n";
        std::cout << "torque A(" << tA[0] << ", " << tA[1] << ", " << tA[2] << ")\n";
        std::cout << "torque B(" << tB[0] << ", " << tB[1] << ", " << tB[2] << ")\n";
    }

    double rcrossf[3] = {0.0};
    cross_product(r, f, rcrossf, n_dim);

    double finalans[3] = {0.0};
    for (int i = 0; i < n_dim; ++i) {
        finalans[i] = tA[i] + tB[i] + rcrossf[i];
    }
    if (print_info) {
        std::cout << "final   (" << finalans[0] << ", " << finalans[1] << ", " << finalans[2] << ")\n";
    }

    if (fabs(finalans[0]) > 1e-10 ||
        fabs(finalans[1]) > 1e-10 ||
        fabs(finalans[2]) > 1e-10) {
        std::cout << "Force balance didin't work correctly, failing test!\n";
        success = false;
    }
    return success;
}

bool TestChromosomes::TestDeltas() {
    bool success = true;
    // Just write out the deltas of the change in positions
    // final - initial
    double dr[3] = {0.0};
    double du[3] = {0.0};
    double dv[3] = {0.0};
    double dw[3] = {0.0};

    YAML::Node node = YAML::LoadFile(testfile_);

    std::cout << "kA deltas: \n";
    Kinetochore *kA = &(properties_.chromosomes.kinetochores_[0]);
    Kinetochore *kB = &(properties_.chromosomes.kinetochores_[1]);
    for (int i = 0; i < parameters_.n_dim; ++i) {
        dr[i] = kA->r_[i] - rA[i];
        du[i] = kA->u_[i] - uA[i];
        dv[i] = kA->v_[i] - vA[i];
        dw[i] = kA->w_[i] - wA[i];
    }
    std::cout << "  drA(" << dr[0] << ", " << dr[1] << ", " << dr[2] << ")\n";
    std::cout << "  duA(" << du[0] << ", " << du[1] << ", " << du[2] << ")\n";
    std::cout << "  dvA(" << dv[0] << ", " << dv[1] << ", " << dv[2] << ")\n";
    std::cout << "  dwA(" << dw[0] << ", " << dw[1] << ", " << dw[2] << ")\n";

    // Test against results
    for (int i = 0; i < parameters_.n_dim; ++i) {
        if (fabs(dr[i] - node["results"]["deltas"]["kA"]["dr"][i].as<double>()) > 1e-10) {
            std::cout << "delta: " << fabs(dr[i] - node["results"]["deltas"]["kA"]["dr"][i].as<double>()) << std::endl;
            success = success && false;
        }
        if (fabs(du[i] - node["results"]["deltas"]["kA"]["du"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dv[i] - node["results"]["deltas"]["kA"]["dv"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dw[i] - node["results"]["deltas"]["kA"]["dw"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
    }
    if (!success) {
        std::cout << "Kinetochore A failed delta test!\n";
        return success;
    }

    std::cout << "kB deltas: \n";
    for (int i = 0; i < parameters_.n_dim; ++i) {
        dr[i] = kB->r_[i] - rB[i];
        du[i] = kB->u_[i] - uB[i];
        dv[i] = kB->v_[i] - vB[i];
        dw[i] = kB->w_[i] - wB[i];
    }
    std::cout << "  drB(" << dr[0] << ", " << dr[1] << ", " << dr[2] << ")\n";
    std::cout << "  duB(" << du[0] << ", " << du[1] << ", " << du[2] << ")\n";
    std::cout << "  dvB(" << dv[0] << ", " << dv[1] << ", " << dv[2] << ")\n";
    std::cout << "  dwB(" << dw[0] << ", " << dw[1] << ", " << dw[2] << ")\n";

    // Test against results
    for (int i = 0; i < parameters_.n_dim; ++i) {
        if (fabs(dr[i] - node["results"]["deltas"]["kB"]["dr"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(du[i] - node["results"]["deltas"]["kB"]["du"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dv[i] - node["results"]["deltas"]["kB"]["dv"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
        if (fabs(dw[i] - node["results"]["deltas"]["kB"]["dw"][i].as<double>()) > 1e-10) {
            success = success && false;
        }
    }
    if (!success) {
        std::cout << "Kinetochore B failed delta test!\n";
        return success;
    }

    return success;
}

bool TestChromosomes::TestChromosomesKMCInit() {
    bool success = true;
    testfile_ = "test_chromosomes_kmc_init.yaml";

    InitSystem();

    // Check the neighbor lists
    for (int ikc = 0; ikc < properties_.chromosomes.nkcs_; ++ikc) {
        Kinetochore *kc = &(properties_.chromosomes.kinetochores_[ikc]);
        for (int isite = 0; isite < kc->nsites_; ++isite) {
            for (auto nit =  kc->neighbors_[isite].begin();
                      nit != kc->neighbors_[isite].end();
                      ++nit) {
                auto dist = std::distance(kc->neighbors_[isite].begin(), nit);
                std::cout << "kc[" << ikc << "], site[" << isite << "], neighbor[" << dist << "], label: "
                    << nit->label << ", value: " << nit->value << ", dup: " << nit->duplicate_ << std::endl;
            }
        }
    }

    return success;
}
