// Implementation of probabilitiy tests

#include "test_triangles.h"
#include "minimum_distance.h"

#include "bob.h"
#include "helpers.h"
#include "triangle_mesh.h"

#include <iostream>

void TestTriangles::RunTests() {
    std::cout << "****************\n";
    std::cout << "Test Triangles run tests\n";
    std::cout << "****************\n";

    for (auto &kv : tests_) {
        std::cout << "----------------\n";
        std::cout << "Test : " << kv.first << std::endl;
        std::cout << "----------------\n";
        auto result = kv.second();
        if (!result) {
            std::cout << "Test : " << kv.first << " failed, check output!\n";
            exit(1);
        }
    }
}

void TestTriangles::SetTests() {
    YAML::Node node = YAML::LoadFile(filename_);

    if (node["Triangles"]["TestMinimumDistance"]) {
        tests_["TestMinimumDistance"] = std::bind(&TestTriangles::TestMinimumDistance, this);
    }
    if (node["Triangles"]["TestMinimumDistancePoly"]) {
        tests_["TestMinimumDistancePoly"] = std::bind(&TestTriangles::TestMinimumDistancePoly, this);
    }
}

void TestTriangles::UnitTriangle(double **verts) {
    verts[0][0] = 1.0;
    verts[0][1] = 0.0;
    verts[0][2] = 0.0;
    
    verts[1][0] = -0.5;
    verts[1][1] = sqrt(3)/2.;
    verts[1][2] = 0.0;
    
    verts[2][0] = -0.5;
    verts[2][1] = -sqrt(3)/2.;
    verts[2][2] = 0.0;
}

void TestTriangles::UnitTriangle(std::vector<std::array<double, 3>> &verts,
                                 std::vector<int> &indicies) {
    std::array<double, 3> vertex;
    for (int iv = 0; iv < 3; ++iv) {
        double x = cos(2*M_PI/3*iv);
        double y = sin(2*M_PI/3*iv);
        double z = 0.0;

        vertex[0] = x;
        vertex[1] = y;
        vertex[2] = z;

        verts.push_back(vertex);
    }

    indicies.push_back(0);
    indicies.push_back(1);
    indicies.push_back(2);
}

void TestTriangles::UnitSquare(std::vector<std::array<double, 3>> &verts,
                               std::vector<int> &indicies) {
    std::array<double, 3> vertex;

    vertex[0] = 0.5;
    vertex[1] = 0.5;
    vertex[2] = 0.0;
    verts.push_back(vertex);
    
    vertex[0] = -0.5;
    vertex[1] = 0.5;
    vertex[2] = 0.0;
    verts.push_back(vertex);

    vertex[0] = -0.5;
    vertex[1] = -0.5;
    vertex[2] = 0.0;
    verts.push_back(vertex);

    vertex[0] = 0.5;
    vertex[1] = -0.5;
    vertex[2] = 0.0;
    verts.push_back(vertex);

    indicies.push_back(0);
    indicies.push_back(1);
    indicies.push_back(2);

    indicies.push_back(2);
    indicies.push_back(3);
    indicies.push_back(0);
}

void TestTriangles::UnitHexagon(std::vector<std::array<double, 3>> &verts,
                                std::vector<int> &indicies) {
    std::array<double, 3> vertex;

    for (int iv = 0; iv < 6; ++iv) {
        double x = cos(2*M_PI/6*iv);
        double y = sin(2*M_PI/6*iv);
        double z = 0.0;

        vertex[0] = x;
        vertex[1] = y;
        vertex[2] = z;

        verts.push_back(vertex);
    }
    indicies.push_back(0);
    indicies.push_back(1);
    indicies.push_back(2);

    indicies.push_back(2);
    indicies.push_back(3);
    indicies.push_back(4);

    indicies.push_back(4);
    indicies.push_back(5);
    indicies.push_back(0);

    indicies.push_back(0);
    indicies.push_back(2);
    indicies.push_back(4);
}

bool TestTriangles::TestMinimumDistance() {
    bool success = true;

    testfile_ = "test_triangles.yaml";
    YAML::Node node = YAML::LoadFile(testfile_);

    std::cout << node << std::endl;

    int ntests = node["Triangles"]["TestMinimumDistance"].size();
    double **verts = (double **)allocate_2d_array(3, 3, sizeof(double));
    double r1[3] = {0.0};
    double r2[3] = {0.0};
    double xRot, yRot;
    double t, dist;

    std::cout << "ntests " << ntests << std::endl;
    for (int itest = 0; itest < ntests; ++itest) {
        for (int i = 0; i < 3; ++i) {
            r1[i] = node["Triangles"]["TestMinimumDistance"][itest]["r1"][i].as<double>();
            r2[i] = node["Triangles"]["TestMinimumDistance"][itest]["r2"][i].as<double>();
        }
        auto tritype = node["Triangles"]["TestMinimumDistance"][itest]["triangle"].as<std::string>();
        if (tritype == "unittriangle") {
            UnitTriangle(verts);
        } else {
            std::cout << "Type " << tritype << " not implemented yet\n";
            exit(1);
        }

        RunMinimumDistance(verts, r1, r2);
    }

    return success;
}

void TestTriangles::RunMinimumDistance(double **verts, double *r1, double *r2) {
    double xRot, yRot;
    double t, dist;
    triangle_mesh tri;
    init_triangle_mesh(&tri);
    create_basic_triangle(&tri, verts);
    print_triangle(&tri, 0);

    segment_to_triangle(r1[0], r1[1], r1[2],
                        r2[0], r2[1], r2[2],
                        &tri, 0, &t,
                        &xRot, &yRot, &dist);

    std::cout << "Final results\n";
    std::cout << "t: " << t << std::endl;
    std::cout << "xRot,yRot: (" << xRot << ", " << yRot << ")\n";
    std::cout << "dist: " << dist << std::endl;
    std::cout << "********\n";
}

bool TestTriangles::TestMinimumDistancePoly() {
    bool success = true;

    testfile_ = "test_triangles.yaml";
    YAML::Node node = YAML::LoadFile(testfile_);

    int ntests = node["Triangles"]["TestMinimumDistancePoly"].size();
    std::vector<std::array<double, 3>> verts;
    std::vector<int> indicies;
    double r1[3] = {0.0};
    double r2[3] = {0.0};
    double xRot, yRot;
    double t, dist;
    double rcontact[3];

    for (int itest = 0; itest < ntests; ++itest) {
        verts.clear();
        indicies.clear();
        for (int i = 0; i < 3; ++i) {
            r1[i] = node["Triangles"]["TestMinimumDistancePoly"][itest]["r1"][i].as<double>();
            r2[i] = node["Triangles"]["TestMinimumDistancePoly"][itest]["r2"][i].as<double>();
        }
        auto ptype = node["Triangles"]["TestMinimumDistancePoly"][itest]["polygon"].as<std::string>();
        if (ptype == "unithexagon") {
            UnitHexagon(verts, indicies);
        } else if (ptype == "unitsquare") {
            UnitSquare(verts, indicies); 
        } else if (ptype == "unittriangle") {
            UnitTriangle(verts, indicies);
        } else {
            std::cout << "Type " << ptype << " not implemented yet\n";
            exit(1);
        }

        // Do the min distance calc
        triangle_mesh tri;
        create_polygon(&tri, verts, indicies);
        print_polygon(&tri);
        segment_to_polygon(r1[0], r1[1], r1[2],
                           r2[0], r2[1], r2[2],
                           &tri, &t,
                           &xRot, &yRot, &dist,
                           rcontact);
        std::cout << "Final results\n";
        std::cout << "t: " << t << std::endl;
        std::cout << "xRot,yRot: (" << xRot << ", " << yRot << ")\n";
        std::cout << "dist: " << dist << std::endl;
        std::cout << "********\n";
    }

    return success;
}
