/* This routine computes a variety of thermodynamic quantities
   using the brownian assumption 

   Input: pointer to parameters structure (parameters)
          pointer to properties structure (properties)

   Output: a variety of thermodynamic quantities are computed */

#include "bob.h"

void thermodynamics_sphero(system_parameters * parameters, system_properties * properties)
{
    int n_dim, i, j, n_bonds;
    double temp, **virial, **stress,
        **press_tensor, press_iso, volume;

    /* Set up shortcuts to data structures. */
    n_dim = parameters->n_dim;
    n_bonds = properties->bonds.n_bonds;
    temp = parameters->temp;

    /* Compute stress tensor. */
    virial = properties->thermo.virial;
    stress = properties->thermo.stress;
    for (i = 0; i < n_dim; ++i) {
        for (j = 0; j < n_dim; ++j) {
            stress[i][j] = virial[i][j];
        }
    }
    for (i = 0; i < n_dim; ++i) 
        stress[i][i] += n_bonds * temp;
    
    /* Compute volume and density. */
    volume = properties->thermo.volume = properties->unit_cell.volume;
    properties->thermo.dens = n_bonds / volume;

    /* Compute pressure tensor. Warning! This is only well-defined for a periodic system, or
       for the periodic subspace. */
    press_tensor = properties->thermo.press_tensor;
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j)
            press_tensor[i][j] = stress[i][j] / volume;

    /* Compute isotropic pressure. */
    press_iso = 0.0;
    for (i = 0; i < n_dim; ++i)
        press_iso += press_tensor[i][i];
    press_iso /= n_dim;
    properties->thermo.press_iso = press_iso;

    return;
}
