// Implementaiton of triangle mesh structure, mirroring fortran version

#include "bob.h"
#include "minimum_distance.h"
#include "triangle_mesh.h"

#include <iostream>

// Function prototypes
void cross_product(double *a, double *b, double *c, int n_dim);

// Initialize the mesh
void init_triangle_mesh(triangle_mesh *tri) {
    tri->numVerts = -1;
    tri->numTriang = -1;
}

// Allocate a known size mesh
// Using its own information about triangles and verticies
void allocate_triangle_mesh(triangle_mesh *tri) {
    tri->verts = (double **) allocate_2d_array(3, tri->numVerts, sizeof(double));
    tri->indVert = (int **) allocate_2d_array(3, tri->numTriang, sizeof(int));
    tri->cosGamma = (double *) allocate_1d_array(tri->numTriang, sizeof(double));
    tri->sinGamma = (double *) allocate_1d_array(tri->numTriang, sizeof(double));
    tri->cosBeta = (double *) allocate_1d_array(tri->numTriang, sizeof(double));
    tri->sinBeta = (double *) allocate_1d_array(tri->numTriang, sizeof(double));
    tri->triXYrot = (double ***) allocate_3d_array(tri->numTriang, 3, 3, sizeof(double));
    tri->triZrot = (double *) allocate_1d_array(tri->numTriang, sizeof(double));
}

// Deallocate the triangle mesh
void deallocate_triangle_mesh(triangle_mesh *tri) {
    if (tri->verts) {
        delete tri->verts;
    }
    if (tri->indVert) {
        delete tri->indVert;
    }
    if (tri->cosGamma) {
        delete tri->cosGamma;
    }
    if (tri->sinGamma) {
        delete tri->sinGamma;
    }
    if (tri->cosBeta) {
        delete tri->cosBeta;
    }
    if (tri->sinBeta) {
        delete tri->sinBeta;
    }
    if (tri->triXYrot) {
        delete tri->triXYrot;
    }
    if (tri->triZrot) {
        delete tri->triZrot;
    }
}

// Print triangle itri
void print_triangle(triangle_mesh *tri, int itri) {
    std::cout << "Triangle[" << itri << "]\n";
    for (int vid = 0; vid < 3; ++vid) {
        // Vertex index information
        int ind = tri->indVert[vid][itri];
        std::cout << "  Vertex[" << ind << "] ("
            << tri->verts[0][ind] << ", "
            << tri->verts[1][ind] << ", "
            << tri->verts[2][ind] << ")\n";
    }
    std::cout << "  cosGamma, sinGamma (" << tri->cosGamma[itri] << ", " << tri->sinGamma[itri] << ")\n";
    std::cout << "  cosBeta,  sinBeta  (" << tri->cosBeta[itri] << ", " << tri->sinBeta[itri] << ")\n";
    for (int vid = 0; vid < 3; ++vid) {
        int ind = tri->indVert[vid][itri];
        std::cout << "  Vertex[" << ind << "] triXYrot("
                  << tri->triXYrot[itri][0][vid] << ", "
                  << tri->triXYrot[itri][1][vid] << ")\n";
    }
    std::cout << "  triZrot " << tri->triZrot[itri] << std::endl;
}

// Print all of my triangles
void print_polygon(triangle_mesh *tri) {
    int numtri = tri->numTriang;

    for (int itri = 0; itri < numtri; ++itri) {
        print_triangle(tri, itri);
    }
}

// Create a basic triangle in the mesh
void create_basic_triangle(triangle_mesh *tri, double **verts) {
    std::cout << "NOTICE: Creating a known triangle structure, should not use outside of testing!\n";
    tri->numVerts = 3;
    tri->numTriang = 1;
    int itri = 0; // triangle index
    int ind0 = 0;
    int ind1 = 1;
    int ind2 = 2;
    double aVector[3] = {0.0};
    double bVector[3] = {0.0};
    double cVector[3] = {0.0};

    // Allocate the mesh
    allocate_triangle_mesh(tri);

    // Copy the vertex information
    for (int ivert = 0; ivert < tri->numVerts; ++ivert) {
        for (int i = 0; i < 3; ++i) {
            tri->verts[i][ivert] = verts[ivert][i]; // NOTICE THIS IS BACKWARDS@!!!!!
        }
    }

    // Create the vertex index map
    tri->indVert[0][itri] = ind0;
    tri->indVert[1][itri] = ind1;
    tri->indVert[2][itri] = ind2;

    // Compute rotation to a plane and rotated coordinates
    for (int i = 0; i < 3; ++i) {
        aVector[i] = tri->verts[i][ind1] - tri->verts[i][ind0];
        bVector[i] = tri->verts[i][ind2] - tri->verts[i][ind1];
    }
    cross_product(aVector, bVector, cVector, 3);
    double cLenInXY = sqrt(cVector[0]*cVector[0] + cVector[1]*cVector[1]);
    double cLength = sqrt(cVector[0]*cVector[0] + cVector[1]*cVector[1] + cVector[2]*cVector[2]);

    double sinGam, cosGam, sinBet, cosBet;
    if (cLenInXY < 1.0e-10) {
        sinGam = 0.0;
        cosGam = 1.0;
    } else {
        sinGam = -cVector[1] / cLenInXY;
        cosGam = cVector[0] / cLenInXY;
    }
    sinBet = cLenInXY / cLength;
    cosBet = cVector[2] / cLength;
    
    tri->cosGamma[itri] = cosGam;
    tri->sinGamma[itri] = sinGam;
    tri->cosBeta[itri] = cosBet;
    tri->sinBeta[itri] = sinBet;

    // Calculate and save rotated coordiantes
    double zRotSum = 0;
    for (int ivert = 0; ivert < 3; ++ivert) {
        int iv = tri->indVert[ivert][itri];
        tri->triXYrot[itri][0][ivert] = (tri->verts[0][iv] * cosGam -
                tri->verts[1][iv] * sinGam) * cosBet - tri->verts[2][iv] * sinBet;
        tri->triXYrot[itri][1][ivert] = tri->verts[0][iv] * sinGam +
                tri->verts[1][iv] * cosGam;
        zRotSum = zRotSum + (tri->verts[0][iv] * cosGam -
                tri->verts[1][iv] * sinGam) * sinBet + tri->verts[2][iv] * cosBet;
    }
    tri->triZrot[itri] = zRotSum / 3.;
}

// Create a polygon in the mesh
// VERTS is the vertex array to load
// INDICIES is the indexes (triplets) of the triangles
void create_polygon(triangle_mesh *tri,
                    std::vector<std::array<double, 3>> verts,
                    std::vector<int> indicies) {
    std::cout << "NOTICE: Creating a polygonal triangle mesh, multiple triangles!\n";
    tri->numVerts = (int)verts.size();
    tri->numTriang = (int)indicies.size()/3;
    std::cout << "Found " << tri->numTriang << " triangles and " << tri->numVerts << " verticies\n";
    int itri = 0; // triangle index
    int ind0 = 0;
    int ind1 = 0;
    int ind2 = 0;

    allocate_triangle_mesh(tri);

    // Copy in vertex information
    for (int iv = 0; iv < tri->numVerts; ++iv) {
        for (int i = 0; i < 3; ++i) {
            tri->verts[i][iv] = verts[iv][i]; // NOTICE THIS IS BACKWARDS@!!!!!
        }
    }

    for (itri = 0; itri < tri->numTriang; ++itri) {
        // Create the vertex index map
        ind0 = indicies[3*itri + 0];
        ind1 = indicies[3*itri + 1];
        ind2 = indicies[3*itri + 2];

        // Create the vertex index map
        tri->indVert[0][itri] = ind0;
        tri->indVert[1][itri] = ind1;
        tri->indVert[2][itri] = ind2;
    }

    update_polygon(tri);
}

// Create the index entries for a type of polygon
void create_polygon_mesh(int nverts, std::vector<int> &indicies) {
    if (nverts == 3) {
        // Triangle
        indicies.push_back(0);
        indicies.push_back(1);
        indicies.push_back(2);
    } else if (nverts == 4) {
        // Square
        indicies.push_back(0);
        indicies.push_back(1);
        indicies.push_back(2);

        indicies.push_back(2);
        indicies.push_back(3);
        indicies.push_back(0);
    } else if (nverts == 6) {
        // Hexagon
        indicies.push_back(0);
        indicies.push_back(1);
        indicies.push_back(2);

        indicies.push_back(2);
        indicies.push_back(3);
        indicies.push_back(4);

        indicies.push_back(4);
        indicies.push_back(5);
        indicies.push_back(0);

        indicies.push_back(0);
        indicies.push_back(2);
        indicies.push_back(4);
    } else {
        std::cout << "NVERTS " << nverts << " not polygonalize yet!\n";
        exit(1);
    }
}

// Update the polygon to the new verticies (unchanged incidies)
void update_polygon(triangle_mesh *tri) {
    int ind0 = 0;
    int ind1 = 0;
    int ind2 = 0;
    double aVector[3] = {0.0};
    double bVector[3] = {0.0};
    double cVector[3] = {0.0};

    for (int itri = 0; itri < tri->numTriang; ++itri) {
        // Load the index map
        ind0 = tri->indVert[0][itri];
        ind1 = tri->indVert[1][itri];
        ind2 = tri->indVert[2][itri];

        // Compute rotation to a plane and rotated coordinates
        for (int i = 0; i < 3; ++i) {
            aVector[i] = tri->verts[i][ind1] - tri->verts[i][ind0];
            bVector[i] = tri->verts[i][ind2] - tri->verts[i][ind1];
        }
        cross_product(aVector, bVector, cVector, 3);
        double cLenInXY = sqrt(cVector[0]*cVector[0] + cVector[1]*cVector[1]);
        double cLength = sqrt(cVector[0]*cVector[0] + cVector[1]*cVector[1] + cVector[2]*cVector[2]);

        double sinGam, cosGam, sinBet, cosBet;
        if (cLenInXY < 1.0e-10) {
            sinGam = 0.0;
            cosGam = 1.0;
        } else {
            sinGam = -cVector[1] / cLenInXY;
            cosGam = cVector[0] / cLenInXY;
        }
        sinBet = cLenInXY / cLength;
        cosBet = cVector[2] / cLength;
        
        tri->cosGamma[itri] = cosGam;
        tri->sinGamma[itri] = sinGam;
        tri->cosBeta[itri] = cosBet;
        tri->sinBeta[itri] = sinBet;

        // Calculate and save rotated coordiantes
        double zRotSum = 0;
        for (int ivert = 0; ivert < 3; ++ivert) {
            int iv = tri->indVert[ivert][itri];
            tri->triXYrot[itri][0][ivert] = (tri->verts[0][iv] * cosGam -
                    tri->verts[1][iv] * sinGam) * cosBet - tri->verts[2][iv] * sinBet;
            tri->triXYrot[itri][1][ivert] = tri->verts[0][iv] * sinGam +
                    tri->verts[1][iv] * cosGam;
            zRotSum = zRotSum + (tri->verts[0][iv] * cosGam -
                    tri->verts[1][iv] * sinGam) * sinBet + tri->verts[2][iv] * cosBet;
        }
        tri->triZrot[itri] = zRotSum / 3.;
    }
}

// calcluate the distance between a pair of segments
// INPUT:
// (x1, y1, z1) describes the start/end of first segment
// (x2, y2, z2) describes the start/end of second segment
// OUTPUT:
// (t1, t2) linear distance along segments of closest approach
// dist the squared distance of closest approach
void segment_dist(double xStart1, double yStart1, double zStart1,
                  double xEnd1, double yEnd1, double zEnd1,
                  double xStart2, double yStart2, double zStart2,
                  double xEnd2, double yEnd2, double zEnd2,
                  double *t1, double *t2, double *dist) {
    //std::cout << "Segment distance\n";
    //std::cout << "  line1 ("
    //    << xStart1 << ", " << yStart1 << ", " << zStart1 << ")("
    //    << xEnd1 << ", " << yEnd1 << ", " << zEnd1 << ")\n";
    //std::cout << "  line2 ("
    //    << xStart2 << ", " << yStart2 << ", " << zStart2 << ")("
    //    << xEnd2 << ", " << yEnd2 << ", " << zEnd2 << ")\n";
    // First, find z1, z2 position of global minimum
    double a1 = xEnd1 - xStart1;
    double b1 = yEnd1 - yStart1;
    double c1 = zEnd1 - zStart1;
    double a2 = xEnd2 - xStart2;
    double b2 = yEnd2 - yStart2;
    double c2 = zEnd2 - zStart2;
    double sqr1 = a1*a1 + b1*b1 + c1*c1;
    double sqr2 = a2*a2 + b2*b2 + c2*c2;
    double crossTerm = -(a1 * a2 + b1 * b2 + c1 * c2);
    double const1 = a1 * (xStart2 - xStart1) + b1 * (yStart2 - yStart1) + c1 * (zStart2 - zStart1);
    double const2 = -(a2 * (xStart2 - xStart1) + b2 * (yStart2 - yStart1) +
            c2 * (zStart2 - zStart1));
    double den = sqr1 * sqr2 - crossTerm*crossTerm;
    double t1Numer = const1 * sqr2 - const2 * crossTerm;
    double t2Numer = sqr1 * const2 - const1 * crossTerm;

    if (fabs(den) < 1e-20 ||
        fabs(den) < 1e-6 * std::max(fabs(t1Numer), fabs(t2Numer))) {
        //std::cout << "  Found parallel lines\n";
        // Parallel lines: "just" check the 4 endpoints
        // start of line1 versus line2
        double t2Trunc, t1Trunc, tdist;
        t2Trunc = std::max(0.0, std::min(1.0, tri_point_to_line(xStart2, yStart2, zStart2, a2, b2, c2,
                        xStart1, yStart1, zStart1)));
        *dist =
            (a2 * t2Trunc + xStart2 - xStart1) *
            (a2 * t2Trunc + xStart2 - xStart1) +
            (b2 * t2Trunc + yStart2 - yStart1) *
            (b2 * t2Trunc + yStart2 - yStart1) +
            (c2 * t2Trunc + zStart2 - zStart1) *
            (c2 * t2Trunc + zStart2 - zStart1);
        *t1 = 0.0;
        *t2 = t2Trunc;
        // End of line1 versus line2
        t2Trunc = std::max(0.0, std::min(1.0, tri_point_to_line(xStart2, yStart2, zStart2, a2, b2, c2,
                        xEnd1, yEnd1, zEnd1)));
        tdist = 
            (a2 * t2Trunc + xStart2 - xEnd1) *
            (a2 * t2Trunc + xStart2 - xEnd1) +
            (b2 * t2Trunc + yStart2 - yEnd1) *
            (b2 * t2Trunc + yStart2 - yEnd1) +
            (c2 * t2Trunc + zStart2 - zEnd1) *
            (c2 * t2Trunc + zStart2 - zEnd1);
        if (tdist < *dist) {
            *dist = tdist;
            *t1 = 1.0;
            *t2 = t2Trunc;
        }
        // Start of line2 versus line1
        t1Trunc = std::max(0.0, std::min(1.0, tri_point_to_line(xStart1, yStart1, zStart1, a1, b1, c1,
                        xStart2, yStart2, zStart2)));
        tdist =
            (a1 * t1Trunc + xStart1 - xStart2) *
            (a1 * t1Trunc + xStart1 - xStart2) +
            (b1 * t1Trunc + yStart1 - yStart2) *
            (b1 * t1Trunc + yStart1 - yStart2) +
            (c1 * t1Trunc + zStart1 - zStart2) *
            (c1 * t1Trunc + zStart1 - zStart2);
        if (tdist < *dist) {
            *dist = tdist;
            *t1 = t1Trunc;
            *t2 = 0.0;
        }
        // End of line2 versus line1
        t1Trunc = std::max(0.0, std::min(1.0, tri_point_to_line(xStart1, yStart1, zStart1, a1, b1, c1,
                        xEnd2, yEnd2, zEnd2)));
        tdist =
            (a1 * t1Trunc + xStart1 - xEnd2) *
            (a1 * t1Trunc + xStart1 - xEnd2) +
            (b1 * t1Trunc + yStart1 - yEnd2) *
            (b1 * t1Trunc + yStart1 - yEnd2) +
            (c1 * t1Trunc + zStart1 - zEnd2) *
            (c1 * t1Trunc + zStart1 - zEnd2);
        if (tdist < *dist) {
            *dist = tdist;
            *t1 = t1Trunc;
            *t2 = 1.0;
        }
    } else {
        // Non parallel lines
        //std::cout << "  Non parallel lines\n";
        *t1 = t1Numer / den;
        *t2 = t2Numer / den;
        bool out1 = (*t1 < 0.0 || *t1 > 1.0);
        bool out2 = (*t2 < 0.0 || *t2 > 1.0);
        if (out1 && out2) {
            // If both closest points are out of bounds, truncate each one to
            // its segment, then find closest point on other segment to that
            // truncated point. If this gives different answers, pick the pair
            // with the closest approach
            //std::cout << "  Both closest points out of bounds\n";
            *t1 = std::max(0.0, std::min(1.0, *t1));
            double t2Trunc = std::max(0.0, std::min(1.0, tri_point_to_line(xStart2, yStart2, zStart2, a2, b2, c2,
                            a1 * *t1 + xStart1, b1 * *t1 + yStart1, c1 * *t1 + zStart1)));
            *t2 = std::max(0.0, std::min(1.0, *t2));
            double t1Trunc = std::max(0.0, std::min(1.0, tri_point_to_line(xStart1, yStart1, zStart1, a1, b1, c1,
                            a2 * *t2 + xStart2, b2 * *t2 + yStart2, c2 * *t2 + zStart2)));
            if (*t1 != t1Trunc || *t2 != t2Trunc) {
                //std::cout << "  or not equals\n";
                *dist =
                    (a1 * *t1 + xStart1 - a2 * t2Trunc - xStart2) *
                    (a1 * *t1 + xStart1 - a2 * t2Trunc - xStart2) +
                    (b1 * *t1 + yStart1 - b2 * t2Trunc - yStart2) *
                    (b1 * *t1 + yStart1 - b2 * t2Trunc - yStart2) +
                    (c1 * *t1 + zStart1 - c2 * t2Trunc - zStart2) *
                    (c1 * *t1 + zStart1 - c2 * t2Trunc - zStart2);
                double tdist = 
                    (a1 * t1Trunc + xStart1 - a2 * *t2 - xStart2) *
                    (a1 * t1Trunc + xStart1 - a2 * *t2 - xStart2) +
                    (b1 * t1Trunc + yStart1 - b2 * *t2 - yStart2) *
                    (b1 * t1Trunc + yStart1 - b2 * *t2 - yStart2) +
                    (c1 * t1Trunc + zStart1 - c2 * *t2 - zStart2) *
                    (c1 * t1Trunc + zStart1 - c2 * *t2 - zStart2);
                if (*dist < tdist) {
                    *t2 = t2Trunc;
                } else {
                    *dist = tdist;
                    *t1 = t1Trunc;
                }
            } else {
                //std::cout << "   simple dist\n";
                *dist = 
                    (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) *
                    (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) +
                    (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) *
                    (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) +
                    (c1 * *t1 + zStart1 - c2 * *t2 - zStart2) *
                    (c1 * *t1 + zStart1 - c2 * *t2 - zStart2);
            }
        } else if (out1) {
            //std::cout << "  Segment point 1 outside bounds\n";
            *t1 = std::max(0.0, std::min(1.0, *t1));
            *t2 = std::max(0.0, std::min(1.0, tri_point_to_line(xStart2, yStart2, zStart2, a2, b2, c2,
                            a1 * *t1 + xStart1, b1 * *t1 + yStart1, c1 * *t1 + zStart1)));
            *dist = 
                (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) *
                (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) +
                (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) *
                (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) +
                (c1 * *t1 + zStart1 - c2 * *t2 - zStart2) *
                (c1 * *t1 + zStart1 - c2 * *t2 - zStart2);
        } else if (out2) {
            //std::cout << "  Segment point 2 outside bounds\n";
            *t2 = std::max(0.0, std::min(1.0, *t2));
            *t1 = std::max(0.0, std::min(1.0, tri_point_to_line(xStart1, yStart1, zStart1, a1, b1, c1,
                            a2 * *t2 + xStart2, b2 * *t2 + yStart2, c2 * *t2 + zStart2)));
            *dist = 
                (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) *
                (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) +
                (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) *
                (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) +
                (c1 * *t1 + zStart1 - c2 * *t2 - zStart2) *
                (c1 * *t1 + zStart1 - c2 * *t2 - zStart2);
        } else {
            //std::cout << "  Segment points both inside bounds\n";
            *dist = 
                (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) *
                (a1 * *t1 + xStart1 - a2 * *t2 - xStart2) +
                (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) *
                (b1 * *t1 + yStart1 - b2 * *t2 - yStart2) +
                (c1 * *t1 + zStart1 - c2 * *t2 - zStart2) *
                (c1 * *t1 + zStart1 - c2 * *t2 - zStart2);
        }
    }
}

// inside triangle determines if xtest, ytest lies within the boundaries defined
// by bx and by
bool inside_triangle(double *bx, double *by, double xTest, double yTest) {
    bool inside_triangle = false;

    //std::cout << "Testing (" << xTest << ", " << yTest << ") inside triangle\n";
    //for (int iv = 0; iv < 3; ++iv) {
    //    std::cout << "Vertex[" << iv << "] (" << bx[iv] << ", " << by[iv] << ")\n";
    //}
    // Compute signed area of each triangle between the point and an edge
    double area0 = (bx[0] - xTest) * (by[1] - yTest) - (bx[1] - xTest) * (by[0] - yTest);
    double area1 = (bx[1] - xTest) * (by[2] - yTest) - (bx[2] - xTest) * (by[1] - yTest);
    double area2 = (bx[2] - xTest) * (by[0] - yTest) - (bx[0] - xTest) * (by[2] - yTest);

    if (area0 != 0.0 && area1 != 0.0 && area2 != 0.0) {
        //std::cout << "Testing first case of all nonzero areas\n";
        inside_triangle = ((area0 > 0.0 && area1 > 0.0 && area2 > 0.0) ||
                          (area0 < 0.0 && area1 < 0.0 && area2 < 0.0));
        return inside_triangle;
    }
    if (area0 == 0.0 && area1 == 0.0 && area2 == 0.0) {
        std::cout << "ALL AREAS 0 NOT IMPLEMENTED YET!\n";
        exit(1);
    }
    //std::cout << "Testing third case of wacky triangles\n";
    inside_triangle = area0 == 0. and area1 > 0. and area2 > 0.
        or area0 == 0. and area1 < 0. and area2 < 0.
        or area1 == 0. and area0 > 0. and area2 > 0.
        or area1 == 0. and area0 < 0. and area2 < 0.
        or area2 == 0. and area0 > 0. and area1 > 0.
        or area2 == 0. and area0 < 0. and area1 < 0.
        or area0 == 0. and area1 == 0.
        or area0 == 0. and area2 == 0.
        or area0 == 1. and area2 == 0.;

    return inside_triangle;
}

// line segment to triangle calculation
// INPUT
// (x, y, z) segment line begin/end
// tri, itriang triangle to calculate distance to
// OUTPUT
// (xRot, yRot) rotated point of closest approach to triangle
// dist distance of closest approach
// t fractional distance along line
void segment_to_triangle(double xStart, double yStart, double zStart,
                         double xEnd, double yEnd, double zEnd,
                         triangle_mesh *tri, int itriang, double *t,
                         double *xRot, double *yRot, double *dist) {
    //std::cout << "Calculating segment to triangle distance\n";
    //std::cout << "xStart (" << xStart << ", " << yStart << ", " << zStart << ")\n";
    //std::cout << "xEnd   (" << xEnd << ", " << yEnd << ", " << zEnd << ")\n";

    // Set up shortcuts, because we can
    double *cosGamma = tri->cosGamma;
    double *sinGamma = tri->sinGamma;
    double *cosBeta = tri->cosBeta;
    double *sinBeta = tri->sinBeta;
    double *triZrot = tri->triZrot;
    double ***triXYrot = tri->triXYrot;

    // Variables needed for running
    bool startIn, endIn;
    double temp;
    double xStartRot, yStartRot, zStartRot;
    double xEndRot, yEndRot, zEndRot;

    // Rotate the endpoints
    temp = xStart * cosGamma[itriang] - yStart * sinGamma[itriang];
    xStartRot = temp * cosBeta[itriang] - zStart * sinBeta[itriang];
    yStartRot = xStart * sinGamma[itriang] + yStart * cosGamma[itriang];
    zStartRot = temp * sinBeta[itriang] + zStart * cosBeta[itriang];
    temp = xEnd * cosGamma[itriang] - yEnd * sinGamma[itriang];
    xEndRot = temp * cosBeta[itriang] - zEnd * sinBeta[itriang];
    yEndRot = xEnd * sinGamma[itriang] + yEnd * cosGamma[itriang];
    zEndRot = temp * sinBeta[itriang] + zEnd * cosBeta[itriang];
    //std::cout << "xStartRot (" << xStartRot << ", " << yStartRot << ", " << zStartRot << ")\n";
    //std::cout << "xEndRot   (" << xEndRot << ", " << yEndRot << ", " << zEndRot << ")\n";
    startIn = inside_triangle(&(triXYrot[itriang][0][0]), &(triXYrot[itriang][1][0]), xStartRot, yStartRot);
    endIn = inside_triangle(&(triXYrot[itriang][0][0]), &(triXYrot[itriang][1][0]), xEndRot, yEndRot);
    //std::cout << "start/end in (" << startIn << ", " << endIn << ")\n";

    //// TEST ROTATION UNDO
    //double xprime = xStartRot * cosBeta[itriang] + zStartRot * sinBeta[itriang];
    //double x2 = xprime * cosGamma[itriang] + yStartRot * sinGamma[itriang];
    //double y2 = -xprime * sinGamma[itriang] + yStartRot * cosGamma[itriang];
    //double z2 = -xStartRot * sinBeta[itriang] + zStartRot * cosBeta[itriang];
    //std::cout << "start undo (" << x2 << ", " << y2 << ", " << z2 << ")\n";
    //// END TEST

    if (startIn && endIn) {
        // if both endpoints are over triangle, then one must be closest
        // unless line passes through triangle
        if (b3dxor(zStartRot > triZrot[itriang], zEndRot > triZrot[itriang])) {
            //std::cout << "b3dxor triggered\n";
            *dist = 0;
            *t = (triZrot[itriang] - zStartRot) / (zEndRot - zStartRot);
            *xRot = (1.0 - *t) * xStartRot + *t * xEndRot;
            *yRot = (1.0 - *t) * yStartRot + *t * yEndRot;
        } else {
            //std::cout << "b3dxor not triggered\n";
            double distEnd = fabs(zStartRot - triZrot[itriang]);
            double distStart = fabs(zEndRot - triZrot[itriang]);
            if (distEnd < distStart) {
                *dist = distEnd;
                *xRot = xStartRot;
                *yRot = yStartRot;
                *t = 0.;
            } else {
                *dist = distStart;
                *xRot = xEndRot;
                *yRot = yEndRot;
                *t = 1.;
            }
        }
        return;
    }
    // Set dist to something absurd
    *dist = 1e10;

    // If one endpoint is over, it is a candidate
    if (startIn) {
        *xRot = xStartRot;
        *yRot = yStartRot;
        *dist = fabs(zStartRot - triZrot[itriang]);
        *t = 0.;
    }
    if (endIn) {
        *xRot = xEndRot;
        *yRot = yEndRot;
        *dist = fabs(zEndRot - triZrot[itriang]);
        *t = 1.;
    }

    // but still need to check each line segment
    double dmin = *dist * *dist;
    int ivMin = -1;
    int ivNextMin = -1;
    double t1AtMin, t2AtMin;
    for (int iv = 0; iv < 3; ++iv) {
        int ivNext = iv + 1;
        if (ivNext > 2) ivNext = 0;
        double t1, t2, dsqr;
        segment_dist(triXYrot[itriang][0][iv], triXYrot[itriang][1][iv], triZrot[itriang],
                     triXYrot[itriang][0][ivNext], triXYrot[itriang][1][ivNext], triZrot[itriang],
                     xStartRot, yStartRot, zStartRot,
                     xEndRot, yEndRot, zEndRot,
                     &t1, &t2, &dsqr);
        if (dsqr < dmin) {
            //std::cout << "  found new minimum distance " << dsqr << std::endl;
            ivMin = iv;
            ivNextMin = ivNext;
            dmin = dsqr;
            t1AtMin = t1;
            t2AtMin = t2;
            //std::cout << "  set t1 " << t1AtMin << ", t2 " << t2AtMin << std::endl;
        }
    }

    // If a segment was it, need squre root and rotated position
    if (ivMin >= 0) {
        *dist = sqrt(dmin);
        *xRot = (1.0 - t1AtMin) * triXYrot[itriang][0][ivMin] + t1AtMin *
            triXYrot[itriang][0][ivNextMin];
        *yRot = (1.0 - t1AtMin) * triXYrot[itriang][1][ivMin] + t1AtMin *
            triXYrot[itriang][1][ivNextMin];
        *t = t2AtMin;
    }
}

// Minimum distance from segment to a polygon
// INPUT
// (x, y, z) line segment
// tri triangle mesh
// RETURN: triangle number in mesh
// OUTPUT
// t fractional distance along segment
// (xRot, yRot) rotated coordinates of point on triangle mesh
// dist distance of closest approach
// rcontact lab frame coordinates of closest point of approach on
//     polygon
int segment_to_polygon(double xStart, double yStart, double zStart,
                       double xEnd, double yEnd, double zEnd,
                       triangle_mesh *tri, double *t,
                       double *xRot, double *yRot, double *dist,
                       double *rcontact) {
    // Set up shortcuts, because we can
    double *cosGamma = tri->cosGamma;
    double *sinGamma = tri->sinGamma;
    double *cosBeta = tri->cosBeta;
    double *sinBeta = tri->sinBeta;
    double *triZrot = tri->triZrot;
    double ***triXYrot = tri->triXYrot;

    // Set dist to something large
    int itriang = 0;
    *dist = 1e10;
    for (int itri = 0; itri < tri->numTriang; ++itri) {
        double tt, txRot, tyRot, tdist;
        segment_to_triangle(xStart, yStart, zStart,
                            xEnd, yEnd, zEnd,
                            tri, itri, &tt,
                            &txRot, &tyRot, &tdist);
        if (tdist < *dist) {
            *dist = tdist;
            *t = tt;
            *xRot = txRot;
            *yRot = tyRot;
            itriang = itri;
        }
    }

    // undo the rotation
    double xPrime = *xRot * cosBeta[itriang] + triZrot[itriang] * sinBeta[itriang];
    rcontact[0] = xPrime * cosGamma[itriang] + *yRot * sinGamma[itriang];
    rcontact[1] = -xPrime * sinGamma[itriang] + *yRot * cosGamma[itriang];
    rcontact[2] = -(*xRot) * sinBeta[itriang] + triZrot[itriang] * cosBeta[itriang];
    //std::cout << "Contact (" << rcontact[0] << ", "
    //                         << rcontact[1] << ", "
    //                         << rcontact[2] << ")\n";

    return itriang;
}
