// Contains the relevant informatino for triangle meshes
// Taken from the IMOD program

/**
 * @file triangle_mesh.h
 * @brief Header file for triangle mesh
 */

#ifndef _TRIANGLE_MESH_H_
#define _TRIANGLE_MESH_H_

// Triangle mesh structure (contains multiple triangles)
/// @brief Triangle mesh structure (contains multiple triangles)
struct _triangle_mesh {
    int numVerts;           ///< number of verticies in the mesh
    int numTriang;          ///< number of triangles in the mesh

    int **indVert;          ///< vertex indicies stored as [vertex012, itri]

    double *cosGamma;       ///< cosines of gamma [itri]
    double *sinGamma;       ///< sines of gamma [itri]
    double *cosBeta;        ///< cosines of beta [itri]
    double *sinBeta;        ///< sines of beta [itri]
    double *triZrot;        ///< rotated z coordiantes of triangle (vertices) [itri]
    double **verts;         ///< verticies [dimension][index]
    double ***triXYrot;     ///< rotated coordinates of triangle vertices [itri][dim][ivert]
};
typedef struct _triangle_mesh triangle_mesh;

// Initialization/Update routines
void init_triangle_mesh(triangle_mesh *tri);
void allocate_triangle_mesh(triangle_mesh *tri);
void deallocate_triangle_mesh(triangle_mesh *tri);
void create_basic_triangle(triangle_mesh *tri, double **verts);
void create_polygon(triangle_mesh *tri, std::vector<std::array<double, 3>> verts, std::vector<int> indicies);
void create_polygon_mesh(int nverts, std::vector<int> &indicies);
void update_polygon(triangle_mesh *tri);

// Print routines
void print_triangle(triangle_mesh *tri, int itri);
void print_polygon(triangle_mesh *tri);

// Helper routines
// Returns the XOR between two booleans a, b
inline bool b3dxor(bool a, bool b) {
    return ((a && !b) || (b && !a));
}
// Inlined version of distance calc for ease
// Function returns the parameter t for the point along the line
// closest to x, y, z
/**
 * @brief Inlined version of point distance to segment
 * @param (x) Point to find distance from
 * @param (a, b, c) Equation of line
 * @return parameter t for linear distance along line
 */
inline double tri_point_to_line(double xStart, double yStart, double zStart,
                                double a, double b, double c,
                                double x, double y, double z) {
    return (a * (x - xStart) + b * (y - yStart) + c * (z - zStart)) / (a*a + b*b + c*c);
}

// Run routines
/**
 * @brief Computes distance between two line segments
 * @param (x1, y1, z1) describes the start/end of segment 1
 * @param (x2, y2, z2) describes the start/end of segment 2
 * @param (t1, t2) linear distance along segments of closest approach
 * @param dist The squared distance of closest approach
 */
void segment_dist(double xStart1, double yStart1, double zStart1,
                  double xEnd1, double yEnd1, double zEnd1,
                  double xStart2, double yStart2, double zStart2,
                  double xEnd2, double yEnd2, double zEnd2,
                  double *t1, double *t2, double *dist);
/**
 * @brief Determines if xTest and yTest lie within the triangle
 * @param (bx, by) triangle describing boundary
 * @param (xTest, yTest) test point
 */
bool inside_triangle(double *bx, double *by, double xTest, double yTest);
/**
 * @brief Computes the distance from a segment to a triangle
 * @param (x,y,z) start/end of segment
 * @param (tri, itriang) triangles in mesh to calculate distance to
 * @param t fractional distance along line of closest approach
 * @param (xRot, yRot) Rotated point of closest approach to triangle
 * @param dist The distance of closest approach
 */
void segment_to_triangle(double xStart, double yStart, double zStart,
                         double xEnd, double yEnd, double zEnd,
                         triangle_mesh *tri, int itriang, double *t,
                         double *xRot, double *yRot, double *dist);
/**
 * @brief Minimum distance from segment to a polygon
 * @param (x,y,z) start/end of segment
 * @param tri triangle mesh
 * @param t fractional distance along line of closest approach
 * @param (xRot, yRot) Rotated point of closest approach to triangle
 * @param dist The distance of closest approach
 * @param rcontact Lab frame coordinate of closest point of approach on polygon
 */
int segment_to_polygon(double xStart, double yStart, double zStart,
                       double xEnd, double yEnd, double zEnd,
                       triangle_mesh *tri, double *t,
                       double *xRot, double *yRot, double *dist,
                       double *rcontact);

#endif
