/* This routine computes the inverse unit cell matrix, the unit cell volume,
   and various other related quantities from the unit cell matrix.

   Input: number of spatial dimensions (n_dim)
          unit cell matrix (h)

   Output: inverse unit cell matrix (h_inv)
           direct lattice vectors (a)
           reciprocal lattice vectors (b)
           perpendicular distances between opposite unit cell faces (a_perp)
           pointer to unit cell volume (volume) */

#include "bob.h"

void unit_cell_dimensions(int n_dim, double **h,
                          double **h_inv, double **a, double **b, double *a_perp, double *volume)
{
    int i, j;
    double b_mag2, b_mag;

    /* Compute inverse unit cell matrix. */
    matrix_inversion(n_dim, h, h_inv);

    /* Compute unit cell volume. */
    *volume = determinant(n_dim, h);

    /* Set up direct and reciprocal lattice vectors. */
    for (i = 0; i < n_dim; ++i)
        for (j = 0; j < n_dim; ++j) {
            a[i][j] = h[j][i];
            b[i][j] = h_inv[i][j];
        }

    /* Compute perpendicular distances between opposite unit cell faces. */
    for (i = 0; i < n_dim; ++i) {
        b_mag2 = 0.0;
        for (j = 0; j < n_dim; ++j)
            b_mag2 += SQR(b[i][j]);
        b_mag = sqrt(b_mag2);
        a_perp[i] = 1.0 / b_mag;
    }

    return;
}
