// Implementaiton of unit test of lookup tables in multiple ways

#include "bob.h"
#include "correlation_data.h"
#include "kinetochore.h"
#include "unit_test_af_mt_harmonic_potential.h"

#include <fstream>
#include <iostream>

// Run the tests and all their variants
void UnitTestAFMTHarmonicPotential::RunTests() {
    std::cout << "****************\n";
    std::cout << "Unit Test AF MT Harmonic Potential run tests\n";
    std::cout << "****************\n";

    for (auto &kv : tests_) {
        std::cout << "----------------\n";
        std::cout << "Test : " << kv.first << std::endl;
        std::cout << "----------------\n";

        for (int iv = 0; iv < node_["Tests"][kv.first].size(); ++iv) {
            std::cout << "^^^^^^^^^^^^^^^^\n";
            std::cout << "  Variant : " << iv << std::endl;
            std::cout << "^^^^^^^^^^^^^^^^\n";
            var_subnode_ = node_["Tests"][kv.first][iv];
            iv_ = iv;
            auto result = kv.second();

            if (!result) {
                std::cout << "Test : " << kv.first << " failed, check output!\n";
                exit(1);
            }
        }
    }
}

// Set the possible tests to run
void UnitTestAFMTHarmonicPotential::SetTests() {
    node_ = YAML::LoadFile(filename_);

    // Check against names to bind method calls
    if (node_["Tests"]["TestPotentialandUpdate"]) {
        tests_["TestPotentialandUpdate"] = std::bind(&UnitTestAFMTHarmonicPotential::TestPotentialandUpdate, this);
    }
    if (node_["Tests"]["TestRandomSetup"]) {
        tests_["TestRandomSetup"] = std::bind(&UnitTestAFMTHarmonicPotential::TestRandomSetup, this);
    }
}

// Initialize the system of chromosomes (with other stuff toooo)
void UnitTestAFMTHarmonicPotential::InitSystem(system_parameters *parameters,
                                               system_properties *properties) {

    // Generic parameter initialization
    parse_parameters_node(var_subnode_["params"], parameters);
    gsl_rng_env_setup();
    properties->rng.T = gsl_rng_default;
    properties->rng.r = gsl_rng_alloc(properties->rng.T);
    gsl_rng_set(properties->rng.r, parameters->seed);

    // Generate the unit cell
    int ndim = parameters->n_dim;
    properties->unit_cell.h = (double **) allocate_2d_array(ndim, ndim, sizeof(double));
    for (int i = 0; i < ndim; ++i) {
        properties->unit_cell.h[i][i] = 100;
    }

    // Get the microtubule information
    {
        int nmts = (int)var_subnode_["microtubules"].size();
        init_unit_cell_structure(parameters, properties);
        int nsites = 2*nmts;
        parameters->n_spheros = properties->bonds.n_bonds = nmts;
        properties->sites.n_sites = nsites;
        properties->sites.v = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
        properties->sites.r = (double **) allocate_2d_array(nsites, ndim, sizeof(double));
        init_site_structure_sphero(parameters, properties);
        init_bond_structure_sphero(parameters, properties);
        for (int imt = 0; imt < nmts; ++imt) {
            for (int i = 0; i < ndim; ++i) {
                properties->bonds.r_bond[imt][i] = var_subnode_["microtubules"][imt]["r"][i].as<double>();
                properties->bonds.u_bond[imt][i] = var_subnode_["microtubules"][imt]["u"][i].as<double>();
            }
            properties->bonds.length[imt] = var_subnode_["microtubules"][imt]["l"].as<double>();
        }
        // Print the bond information
        for (int imt = 0; imt < nmts; ++imt) {
            std::cout << "New Microtubule:\n";
            std::cout << "  r(" << properties->bonds.r_bond[imt][0] << ", "
                                << properties->bonds.r_bond[imt][1] << ", "
                                << properties->bonds.r_bond[imt][2] << ")\n";
            std::cout << "  u(" << properties->bonds.u_bond[imt][0] << ", "
                                << properties->bonds.u_bond[imt][1] << ", "
                                << properties->bonds.u_bond[imt][2] << ")\n";
            std::cout << "  l " << properties->bonds.length[imt] << std::endl;
        }
        // Create some force stuff
        f_comp_ = (double **) allocate_2d_array(nmts, ndim, sizeof(double));
        t_comp_ = (double **) allocate_2d_array(nmts, ndim, sizeof(double));
        virial_comp_ = (double **) allocate_2d_array(ndim, ndim, sizeof(double));
        calc_matrix_ = (int *) allocate_1d_array(nmts, sizeof(int));

        properties->anchors.n_anchors = 0;
        init_dynamic_instability(parameters, properties);
    }
}

// Generate the invert distribution of attachments along MT
bool UnitTestAFMTHarmonicPotential::TestPotentialandUpdate() {
    bool success = true;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);

    // Create the chromosomes
    // First, create a node based on the basic information
    // Then override what we want for this particular parameter set
    std::string c_file = parameters.chromosome_config;
    YAML::Node chromosome_node = YAML::LoadFile(c_file.c_str());

    // Override the values
    chromosome_node["chromosomes"]["properties"]["AF_spring"]                = var_subnode_["chromosome"]["AF_spring"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_spring_angular"]        = var_subnode_["chromosome"]["AF_spring_angular"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_number_complexes"]      = var_subnode_["chromosome"]["AF_number_complexes"].as<int>();
    chromosome_node["chromosomes"]["properties"]["AF_rest_length"]           = var_subnode_["chromosome"]["AF_rest_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_characteristic_length"] = var_subnode_["chromosome"]["AF_characteristic_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["nkmc"]                     = var_subnode_["chromosome"]["nkmc"].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][0]                 = var_subnode_["chromosome"]["rA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][1]                 = var_subnode_["chromosome"]["rA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][2]                 = var_subnode_["chromosome"]["rA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][0]                 = var_subnode_["chromosome"]["uA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][1]                 = var_subnode_["chromosome"]["uA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][2]                 = var_subnode_["chromosome"]["uA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][0]                 = var_subnode_["chromosome"]["uB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][1]                 = var_subnode_["chromosome"]["uB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][2]                 = var_subnode_["chromosome"]["uB"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][0]                 = var_subnode_["chromosome"]["rB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][1]                 = var_subnode_["chromosome"]["rB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][2]                 = var_subnode_["chromosome"]["rB"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["special"]               = var_subnode_["chromosome"]["special"].as<std::string>();

    properties.chromosomes.Init(&parameters,
                                &properties,
                                &chromosome_node);

    double uret = af_mt_harmonic_potential(&parameters,
                                           &properties,
                                           f_comp_,
                                           virial_comp_,
                                           t_comp_,
                                           calc_matrix_);

    success = TestForceTorqueBalance(&parameters, &properties, true);

    return success;
}

bool UnitTestAFMTHarmonicPotential::TestRandomSetup() {
    bool success = true;

    system_parameters parameters;
    system_properties properties;

    // Initialize the system
    InitSystem(&parameters, &properties);

    // Create the chromosomes
    // First, create a node based on the basic information
    // Then override what we want for this particular parameter set
    std::string c_file = parameters.chromosome_config;
    YAML::Node chromosome_node = YAML::LoadFile(c_file.c_str());

    // Override the values
    chromosome_node["chromosomes"]["properties"]["AF_spring"]                = var_subnode_["chromosome"]["AF_spring"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_spring_angular"]        = var_subnode_["chromosome"]["AF_spring_angular"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_number_complexes"]      = var_subnode_["chromosome"]["AF_number_complexes"].as<int>();
    chromosome_node["chromosomes"]["properties"]["AF_rest_length"]           = var_subnode_["chromosome"]["AF_rest_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["AF_characteristic_length"] = var_subnode_["chromosome"]["AF_characteristic_length"].as<double>();
    chromosome_node["chromosomes"]["properties"]["nkmc"]                     = var_subnode_["chromosome"]["nkmc"].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][0]                 = var_subnode_["chromosome"]["rA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][1]                 = var_subnode_["chromosome"]["rA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rA"][2]                 = var_subnode_["chromosome"]["rA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][0]                 = var_subnode_["chromosome"]["uA"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][1]                 = var_subnode_["chromosome"]["uA"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uA"][2]                 = var_subnode_["chromosome"]["uA"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][0]                 = var_subnode_["chromosome"]["uB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][1]                 = var_subnode_["chromosome"]["uB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["uB"][2]                 = var_subnode_["chromosome"]["uB"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][0]                 = var_subnode_["chromosome"]["rB"][0].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][1]                 = var_subnode_["chromosome"]["rB"][1].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["rB"][2]                 = var_subnode_["chromosome"]["rB"][2].as<double>();
    chromosome_node["chromosomes"]["chromosome"][0]["special"]               = var_subnode_["chromosome"]["special"].as<std::string>();

    properties.chromosomes.Init(&parameters,
                                &properties,
                                &chromosome_node);

    int n = parameters.n_steps;
    int ndim = parameters.n_dim;
    for (int itest = 0; itest < n; ++itest) {
        // Rezero forces on the kinetochore
        for (int i = 0; i < ndim; ++i) {
            properties.chromosomes.f_[0][i] = 0.0;
            properties.chromosomes.t_[0][i] = 0.0;
        }

        // Randomly set the microtubule to something 
        for (int i = 0; i < parameters.n_dim; ++i) {
            properties.bonds.r_bond[0][i] = (gsl_rng_uniform(properties.rng.r) - 0.5) * 20.0;
        }
        generate_random_unit_vector(parameters.n_dim, properties.bonds.u_bond[0], properties.rng.r);
        //std::cout << "           DERP\n";
        double uret = af_mt_harmonic_potential(&parameters,
                                               &properties,
                                               f_comp_,
                                               virial_comp_,
                                               t_comp_,
                                               calc_matrix_);
        success = success && TestForceTorqueBalance(&parameters, &properties, false);
        if (!success)
            break;
    }
    return success;
}

bool UnitTestAFMTHarmonicPotential::TestForceTorqueBalance(system_parameters *parameters,
                                                           system_properties *properties,
                                                           bool print_info) {
    // tA + tB + r x f = 0
    bool success = true;

    double r[3] = {0.0};
    double fA[3] = {0.0};
    double tA[3] = {0.0};
    double fmt[3] = {0.0};
    double tmt[3] = {0.0};

    int ndim = parameters->n_dim;
    for (int i = 0; i < ndim; ++i) {
        r[i] = properties->chromosomes.kinetochores_[0].r_cross_[0][i];

        fA[i] = properties->chromosomes.f_[0][i];
        tA[i] = properties->chromosomes.t_[0][i];

        fmt[i] = f_comp_[0][i];
        tmt[i] = t_comp_[0][i];
    }

    if (print_info) {
        std::cout << "force  KC("
            << fA[0] << ", "
            << fA[1] << ", "
            << fA[2] << ")\n";
        std::cout << "torque KC("
            << tA[0] << ", "
            << tA[1] << ", "
            << tA[2] << ")\n";
        std::cout << "force  MT("
            << fmt[0] << ", "
            << fmt[1] << ", "
            << fmt[2] << ")\n";
        std::cout << "torque MT("
            << tmt[0] << ", "
            << tmt[1] << ", "
            << tmt[2] << ")\n";
    }

    double rcrossf[3] = {0.0};
    cross_product(r, fmt, rcrossf, ndim);

    double finalans[3] = {0.0};
    for (int i = 0; i < ndim; ++i) {
        finalans[i] = tA[i] + rcrossf[i];
    }
    if (print_info) {
        std::cout << "balance ("
            << finalans[0] << ", "
            << finalans[1] << ", "
            << finalans[2] << ")\n";
    }
    if (fabs(finalans[0]) > 1e-10 ||
        fabs(finalans[1]) > 1e-10 ||
        fabs(finalans[2]) > 1e-10) {
        std::cout << "Force balance didin't work correctly, failing test!\n";
        success = false;
    }

    return success;
}
