/* This routine calculates scaled and real positions of sites from bond positions, bond orientations, and bond lengths.
 
 Input: number of spatial dimensions (n_dim)
 number of periodic dimensions (n_periodic)
 number of bonds (n_bonds)
 number of sites (n_sites)
 unit cell matrix (h)
 inverse unit cell matrix (h_inv)
 array of labels of first bond endpoint (bond_site_1)
 array of labels of second bond endpoint (bond_site_2)
 array of real bond positions (r_bond)
 array of bond orientation vectors (u_bond)
 array of bond lengths (length)
 
 Output: array of real site positions (r)
 array of scaled site positions (s) */

#include "bob.h"

void update_bond_site_positions(int n_dim, int n_periodic, int n_bonds, int n_sites, double **h,
                                double **h_inv, int *bond_site_1, int *bond_site_2, double **r_bond,
                                double **u_bond, double *length, double **r, double **s)
{
    int i_bond, i, site_1, site_2;
    double half_length;

    /* Loop over bonds and compute bond site positions. */
    for (i_bond = 0; i_bond < n_bonds; ++i_bond) {
        site_1 = bond_site_1[i_bond];
        site_2 = bond_site_2[i_bond];
        half_length = 0.5 * length[i_bond];
        for (i = 0; i < n_dim; ++i) {
            r[site_1][i] = r_bond[i_bond][i] - half_length * u_bond[i_bond][i];
            r[site_2][i] = r_bond[i_bond][i] + half_length * u_bond[i_bond][i];
        }
    }

    /* Compute scaled site positions and apply periodic boundary conditions if necessary. */
    if (n_periodic > 0)
        periodic_boundary_conditions(n_periodic, n_sites, h, h_inv, r, s);

    return;
}
