/* This routine updates the neighbor lists for a set of objects in any number of
   spatial dimensions and for any type of boundary conditions (free, periodic, or mixed).
   At present, the routine only handles spherocylinder "objects".

   input: number of spatial dimensions (n_dim)
   number of periodic dimensions (n_periodic)
   unit cell matrix (h)
   neighbor list skin (skin)
   number of objects (n_obj)
   array of real object positions (r)
   array of scaled object positions (s)
   array of object directors (u)
   array of object lengths (length)
   array of labels of molecules to which objects belong (mol_label)
   array of intramolecular object labels (intra_label)
   array of object type indices (type_index)
   molecular species array (mol_species)
   array of pair interaction cutoff distances (r_cutoff)
   array of intramolecular exclusions (excluded)

   output: dynamic array of neighbor lists (neighbs) */

#include "bob.h"
#include <vector>

void update_neighbor_lists_sphero_all_pairs(int n_dim, int n_periodic, double **h, double skin,
                                            double diameter, int n_bonds, double **r, double **s,
                                            double **u, double *length,
                                            nl_list *neighbs, int nl_twoway_flag) {
    /* Purge neighbor lists. */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond)
        neighbs[i_bond].clear();

    /* Loop over first bond. */
    for (int i_bond = 0; i_bond < n_bonds - 1; ++i_bond) {

        /* Loop over second bond. */
        for (int j_bond = i_bond + 1; j_bond < n_bonds; ++j_bond) {
            /* Add an entry to the neighbor lists for bond i_bond and
               j_bond if the minimum distance between the two bonds
               is smaller than (r_cutoff[i_type][j_type] + skin). */
            if (sphero_overlap(n_dim, n_periodic, h, skin, diameter,
                               r[i_bond], s[i_bond], u[i_bond], length[i_bond],
                               r[j_bond], s[j_bond], u[j_bond], length[j_bond])) {

                nl_entry new_neighb;
                new_neighb.label = j_bond;
                neighbs[i_bond].push_back(new_neighb);

                if (nl_twoway_flag) {
                    new_neighb.label = i_bond;
                    neighbs[j_bond].push_back(new_neighb);
                }
            }
        }
    }

    return;
}

/* OpenMP version for use with small numbers of particles */
void update_neighbor_lists_sphero_all_pairs_mp(int n_dim, int n_periodic, double **h, double skin,
                                               double diameter, int n_bonds, double **r, double **s,
                                               double **u, double *length,
                                               nl_list *neighbs, int nl_twoway_flag) {
    /* Purge neighbor lists. */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond)
        neighbs[i_bond].clear();

    /* Loop over first bond. */
    #ifdef ENABLE_OPENMP
    #pragma omp parallel
    #endif
    {
        #ifdef ENABLE_OPENMP
        #pragma omp for schedule(runtime) nowait
        #endif
        for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
            int start_bond = i_bond + 1;
            if (nl_twoway_flag)
                start_bond = 0;
            /* Loop over second bond. */
            for (int j_bond = start_bond; j_bond < n_bonds; ++j_bond) {
                if (j_bond != i_bond) {
                    /* Add an entry to the neighbor lists for bond i_bond and
                       j_bond if the minimum distance between the two bonds
                       is smaller than (r_cutoff[i_type][j_type] + skin). */
                    if (sphero_overlap(n_dim, n_periodic, h, skin, diameter,
                                       r[i_bond], s[i_bond], u[i_bond], length[i_bond],
                                       r[j_bond], s[j_bond], u[j_bond], length[j_bond])) {
                        nl_entry new_neighb;
                        new_neighb.label = j_bond;
                        neighbs[i_bond].push_back(new_neighb);
                    }
                }
            }
        }
    } /* end of omp parallel section */

    return;
}
