/* This routine updates the neighbor lists for a set of objects in any number of
   spatial dimensions and for any type of boundary conditions (free, periodic, or mixed),
   using cell lists. At present, the routine only handles spherocylinder "objects".

   input: number of spatial dimensions (n_dim)
          number of periodic dimensions (n_periodic)
          unit cell matrix (h)
          neighbor list skin (skin)
          interaction distance (r_cutoff)
          number of spherocylinders (n_bonds)
          array of real object positions (r)
          array of scaled object positions (s)
          array of object directors (u)
          array of object lengths (length)
          pointer to cell list object
          neighbor list array
          array of intramolecular exclusions (excluded)
          pointer to cell list structure (cells)

   output: dynamic array neighbor list (neighbs) 
*/

#include "bob.h"

void update_neighbor_lists_sphero_cells(int n_dim, int n_periodic, double **h,
                                        double skin, double r_cutoff, 
                                        int n_bonds, double **r, double **s, double **u,
                                        double *length, cell_list *cells,
                                        nl_list *neighbs,
                                        int nl_twoway_flag) {

    /* Purge neighbor lists. */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond)
        neighbs[i_bond].clear();

    /* Loop over objects. */
    for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {

        /* Get label of cell containing object. */
        int i_linear = cells->cell[i_bond];

        /* Loop over cells in neighborhood of selected object. These cells are referenced via an
           array of labels of cells in the search domain of the cell containing the selected object. */
        for (int i_search = 0; i_search < cells->n_search_half[i_linear]; ++i_search) {

            /* Get cell label. */
            int j_linear = cells->search_cell_half[i_linear][i_search];

            /* Get label of next object in cell (for "home" cell) or of first object in cell (otherwise). */
            int j_bond;
            if (j_linear == i_linear)
                j_bond = cells->next[i_bond];
            else
                j_bond = cells->head[j_linear];

            /* Loop over objects in cell. */
            while (j_bond != -1) {

                /* Add an entry to the neighbor lists for object i_bond and 
                   j_bond if the minimum distance between the two objects  
                   is smaller than (r_cutoff + skin).
                   The sphero_overlap routine is called with the number of
                   periodic dimensions set to zero, because periodic boundary
                   conditions are already taken into account via r_shift_i. */
                if (sphero_overlap(n_dim, n_periodic, h, skin, r_cutoff,
                                   r[i_bond], s[i_bond], u[i_bond], length[i_bond],
                                   r[j_bond], s[j_bond], u[j_bond], length[j_bond])) {
                    
                    int bond_1 = MIN(i_bond, j_bond);
                    int bond_2 = MAX(i_bond, j_bond);
                    nl_entry new_neighb;
                    new_neighb.label = bond_2;
                    neighbs[bond_1].push_back(new_neighb);

                    if (nl_twoway_flag) {
                        new_neighb.label = bond_1;
                        neighbs[bond_2].push_back(new_neighb);
                    }
                }

                /* Get label of next object in cell. */
                j_bond = cells->next[j_bond];
            }
        }
    }

    return;
}
