/* This routine updates the neighbor lists for a set of objects in any number of
   spatial dimensions and for any type of boundary conditions (free, periodic, or mixed),
   using cell lists. At present, the routine only handles spherocylinder "objects".

   input: number of spatial dimensions (n_dim)
   number of periodic dimensions (n_periodic)
   unit cell matrix (h)
   neighbor list skin (skin)
   number of objects (n_obj)
   array of real object positions (r)
   array of scaled object positions (s)
   array of object directors (u)
   array of object lengths (length)
   array of labels of molecules to which objects belong (mol_label)
   array of intramolecular object labels (intra_label)
   array of object type indices (type_index)
   molecular species array (mol_species)
   array of pair interaction cutoff distances (r_cutoff)
   array of intramolecular exclusions (excluded)
   pointer to cell list structure (cells)

   output: dynamic array version of neighbor list (neighbs) */

#include "bob.h"

void update_neighbor_lists_sphero_cells_mp(int n_dim, int n_periodic, double **h,
                                           double skin, double r_cutoff,
                                           int n_bonds,
                                           double **r, double **s, double **u, double *length,
                                           cell_list *cells,
                                           nl_list *neighbs,
                                           int nl_twoway_flag) {
#pragma omp parallel 
    {
        /* Loop over objects. */
#pragma omp for schedule(runtime) nowait
        for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
            /* Purge neighbor list. */
            neighbs[i_bond].clear();

            /* Get label of cell containing object. */
            int i_linear = cells->cell[i_bond];
            /* Loop over cells in neighborhood of selected object. These cells
               are referenced via an array of labels of cells in the search
               domain of the cell containing the selected object. */
            for (int i_search = 0; i_search < cells->n_search_full[i_linear]; ++i_search) {
                /* Get cell label. */
                int j_linear = cells->search_cell_full[i_linear][i_search];

                /* Get label of next object in cell (for "home" cell) or of
                   first object in cell (otherwise). */
                int j_bond = cells->head[j_linear];

                /* Loop over objects in cell. */
                while (j_bond != -1) {
                    if (nl_twoway_flag || j_bond > i_bond) {
                        /* Add an entry to the neighbor lists for object i_bond
                           and j_bond if the minimum distance between the two
                           objects is smaller than (r_cutoff + skin). */
                        if (sphero_overlap(n_dim, n_periodic, h, skin, r_cutoff,
                                           r[i_bond], s[i_bond], u[i_bond], length[i_bond],
                                           r[j_bond], s[j_bond], u[j_bond], length[j_bond])) {
                            nl_entry new_neighb;
                            new_neighb.label = j_bond;
                            neighbs[i_bond].push_back(new_neighb);
                        }

                    }
                    /* Get label of next object in cell. */
                    j_bond = cells->next[j_bond];
                }
            }
        }
    }

    return;
}
