// Vesicle entries

#include "bob.h"
#include "vesicle.h"
#include "lookup_table.h"
#include "minimum_distance.h"

#include <iostream>

#define SKIN_ 8.0

Vesicle::Vesicle() : parent_(NULL), idx_(-1) {

}

void Vesicle::Init(VesicleManagement *parent,
                   int idx,
                   double *r,
                   double *u,
                   double *v,
                   double *w) {

    parent_ = parent;
    idx_ = idx;

    ndim_   = parent_->ndim_;
    nsites_ = parent_->nmotors_;
    nbonds_ = parent_->properties_->bonds.n_bonds;

    r_ = r;
    u_ = u;
    v_ = v;
    w_ = w;

    CreateRefVectors();

    CreateBindingSites();

    Allocate();
}

// Print information about current frame
void Vesicle::PrintFrame() {
    std::cout << "Vesicle[" << idx_ << "]\n";

    // KMC/site stuff
    std::cout << "    nbound: " << parent_->n_bound_[idx_] << ", nexptot: " << n_exp_tot_ << std::endl;
    for (int isite = 0; isite < nsites_; ++isite) {
        std::cout << "    site[" << isite << "] attached: " << attach_[isite] << ", nexp: " << n_exp_[isite] << std::endl;
    }
}

void Vesicle::CreateRefVectors() {
    // Make sure that u is actually a unit vector
    double norm_factor = 1.0/sqrt(dot_product(ndim_, u_, u_));
    for (int i = 0; i < ndim_; ++i) {
        u_[i] *= norm_factor;
    }

    double vect1[3] = { 1.0, 0, 0.0};
    double vect2[3] = { 0.0, 1.0, 0.0};
    if (1.0 - ABS(u_[0]) > 1e-2)
        cross_product(u_, vect1, v_, 3);
    else
        cross_product(u_, vect2, v_, 3);

    norm_factor = sqrt(1.0/dot_product(3, v_, v_));
    for (int i = 0; i < 3; ++i)
        v_[i] *= norm_factor;
    
    cross_product(u_, v_, w_, 3);
}

// Create binding sites on the vesicle
void Vesicle::CreateBindingSites() {
    double rv = 0.5 * parent_->v_diameter_;
    double pos_vec[3] = {0.0};

    for (int im = 0; im < nsites_; ++im) {
        // Known location insert
        if (parent_->sites_specified_) {
            pos_vec[0] = parent_->site_specified_[0];
            pos_vec[1] = parent_->site_specified_[1];
            pos_vec[2] = parent_->site_specified_[2];
        } else {
            generate_random_unit_vector(ndim_, pos_vec, parent_->properties_->rng.r);
        }

        // Anchor position is relative in the frame of the vesicle
        // So need to update appropriately
        al_entry new_anchor;
        new_anchor.label = idx_;
        for (int i = 0; i < ndim_; ++i) {
            new_anchor.pos[i] = r_[i] + rv * pos_vec[i];
        }
        double r_rel[3] = {0.0};
        for (int i = 0; i < ndim_; ++i) {
            r_rel[i] = new_anchor.pos[i] - r_[i];
        }
        double u_proj = dot_product(ndim_, r_rel, u_);
        double v_proj = dot_product(ndim_, r_rel, v_);
        double w_proj = dot_product(ndim_, r_rel, w_);
        new_anchor.pos_rel[0] = u_proj;
        new_anchor.pos_rel[1] = v_proj;
        new_anchor.pos_rel[2] = w_proj;

        parent_->anchors_[idx_].push_back(new_anchor);
    }
}

void Vesicle::Allocate() {
    // Allocate space
    neighbors_ = new nl_list[nsites_];
    dr_tot_ = (double**) allocate_2d_array(nsites_, ndim_, sizeof(double));
    n_exp_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    attach_ = (int*) allocate_1d_array(nsites_, sizeof(int));
    cross_pos_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    r_cross_ = (double**) allocate_2d_array(nsites_, ndim_, sizeof(double));
    upot_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    feff_ = (double*) allocate_1d_array(nsites_, sizeof(double));
    flink_ = (double**) allocate_2d_array(nsites_, ndim_, sizeof(double));

    // Attach to any memory locations we need
    // Attach to the position in the anchor array
    anchors_ = &(parent_->anchors_[idx_]);

    // Set any defaults
    for (int isite = 0; isite < nsites_; ++isite) {
        attach_[isite] = -1;
    }
}

// Update the neighbor list
void Vesicle::UpdateNeighbors(int ndim,
                                  int nperiodic,
                                  int nbonds,
                                  double **h,
                                  double **rbond,
                                  double **sbond,
                                  double **ubond,
                                  double *lbond,
                                  double rcutoff2) {
    // Loop over sites
    for (int isite = 0; isite < nsites_; ++isite) {
        dr_tot_[isite][0] = 0.0;
        dr_tot_[isite][1] = 0.0;
        dr_tot_[isite][2] = 0.0;
        neighbors_[isite].clear();
        if (attach_[isite] != -1)
            continue;

        // Loop over bonds
        for (int ibond = 0; ibond < nbonds_; ++ibond) {
            nl_entry neighb;
            double *s_ = NULL;
            double mu, r_min[3];
            min_distance_sphere_sphero(ndim,
                                       nperiodic,
                                       h,
                                       (*anchors_)[isite].pos,
                                       s_,
                                       rbond[ibond],
                                       sbond[ibond],
                                       ubond[ibond],
                                       lbond[ibond],
                                       r_min,
                                       &neighb.r_min_mag2,
                                       &mu);
            if (neighb.r_min_mag2 < rcutoff2) {
                nl_entry new_neighbor;
                new_neighbor.label = ibond;
                new_neighbor.value = 0.0;
                new_neighbor.duplicate_ = false;

                neighbors_[isite].push_back(new_neighbor);
            }
        }
    }

    //// DEBUG, print the neighbor list
    //for (int isite = 0; isite < nsites_; ++isite) {
    //    for (auto p = neighbors_[isite].begin(); p != neighbors_[isite].end(); ++p) {
    //        std::cout << " neighbor: " << p->label << std::endl;
    //    }
    //}
}

// Clear the neighbor list
void Vesicle::ClearNeighbors(int isite) {
    n_exp_tot_ -= n_exp_[isite];
    n_exp_[isite] = 0.0;
    neighbors_[isite].clear();
}

// Update the partition function for the sites
void Vesicle::Update_1_2(int ndim,
                         int nperiodic,
                         int nbonds,
                         double **h,
                         double **rbond,
                         double **sbond,
                         double **ubond,
                         double *lbond) {
    // Already have neighbor list
    n_exp_tot_ = 0.0;
    for (int isite = 0; isite < nsites_; ++isite) {
        //std::cout << "  af[" << isite << "]\n";
        n_exp_[isite] = 0.0;
        if (attach_[isite] != -1) {
            continue;
        }

        // Each site has it's own neighbor list and everything!
        for (nl_list::iterator p = neighbors_[isite].begin(); p != neighbors_[isite].end(); ++p) {
            double dr[3] = {0.0};
            for (int i = 0; i < ndim_; ++i) {
                dr[i] = (*anchors_)[isite].pos[i] - rbond[p->label][i];
            }
            double binding_affinity = parent_->motor_eps_eff_ * parent_->motor_on_rate_;
            p->value = binding_affinity * Stage_1_2_Probability(ndim_,
                                                                dr,
                                                                ubond[p->label],
                                                                lbond[p->label]);
            n_exp_[isite] += p->value;
            n_exp_tot_ += p->value;
        }
    }
}

// Actualy calculation of the partition function
double Vesicle::Stage_1_2_Probability(int n_dim,
                                      double *dr,
                                      double *uline,
                                      double length2) {
    double dr_cross_line[3];
    for (int i = 0; i < n_dim; ++i) {
        dr_cross_line[i] = dr[i];
    }
    double mu0 = -dot_product(n_dim, dr_cross_line, uline);

    double r_min_mag2 = 0.0;
    for (int i = 0; i < n_dim; ++i) {
        double dri = uline[i] * mu0 + dr_cross_line[i];
        r_min_mag2 += SQR(dri);
    }
    if (r_min_mag2 > parent_->rcutoff2_) {
        return 0.0;
    }

    if (parent_->motor_r0_ == 0.0 && parent_->motor_xc_ == 0.0) {
        std::cout << "Don't have zero rest length on chromosomes, exiting!\n";
        exit(1);
    } else {
        // Does this regardless of attachment state
        double lim0 = -mu0 - 0.5 * length2;
        double lim1 = -mu0 + 0.5 * length2;
        double r_min_mag = sqrt(r_min_mag2);
        double x[2] = {fabs(lim0), r_min_mag};
        double term0 = parent_->n_exp_lookup_->Lookup(x) * ((lim0 < 0) ? -1.0 : 1.0);
        x[0] = fabs(lim1);
        double term1 = parent_->n_exp_lookup_->Lookup(x) * ((lim1 < 0) ? -1.0 : 1.0);
        double nexp = (term1 - term0);

        if (nexp > 1E-3) {
            return nexp;
        }
        else
            return 0.0;
    }
    return 0.0;
}

// Insert the vesicles (motors) according to their probabilities
bool Vesicle::Insert_1_2(system_parameters *parameters,
                         system_properties *properties) {
    double ran_loc = gsl_rng_uniform(properties->rng.r) * n_exp_tot_;
    // Loop over sites to pick which one
    double loc = 0.0;
    for (int isite = 0; isite < nsites_; ++isite) {
        loc += n_exp_[isite]; 
        if (loc > ran_loc) {
            loc -= n_exp_[isite];

            // Look at my neighbors for which one to fall onto
            for (nl_list::iterator nl = neighbors_[isite].begin(); nl != neighbors_[isite].end(); ++nl) {
                loc += nl->value;

                if (loc > ran_loc) {

                    bool didconvert = Convert_1_2(parameters,
                                                  properties,
                                                  nl->label,
                                                  isite);

                    if (!didconvert) {
                        std::cout << "Obviously failed, what else can we look at?\n";
                        std::cout << "   loc: " << loc << ", ran_loc: " << ran_loc << ", nexp[" << isite << "]: "
                            << n_exp_[isite] << ", nexptot: " << n_exp_tot_ << std::endl;
                    }
                    return didconvert;
                }
            }
        }
    }

    // If we make it here, something has gone wrong with attaching a trial
    return false;
}

// Convert this to an attachment
bool Vesicle::Convert_1_2(system_parameters *parameters,
                              system_properties *properties,
                              int ibond,
                              int isite) {
    bool success = false;
    ClearNeighbors(isite); 
    attach_[isite] = ibond;

    double **rbond = properties->bonds.r_bond;
    double **ubond = properties->bonds.u_bond;
    double *lbond  = properties->bonds.length;
    double s_1[3] = {0.0};

    // No tip enhancement, so just find the position
    //cross_pos_[isite] = RandomPosition(parameters->n_dim,
    //                                   parameters->n_periodic,
    //                                   properties->unit_cell.h,
    //                                   (*anchors_)[isite].pos,
    //                                   s_1,
    //                                   properties->bonds.r_bond[ibond],
    //                                   properties->bonds.s_bond[ibond],
    //                                   properties->bonds.u_bond[ibond],
    //                                   properties->bonds.length[ibond],
    //                                   parent_->motor_k_,
    //                                   properties->rng.r);
    if (cross_pos_[isite] != cross_pos_[isite]) {
        std::cout << "NaN encountered in KC Convert_1_2\n";
        std::cout << "  bond: " << ibond << ", site: " << isite << std::endl;
        // Instead of exiting, just don't attach and keep going (slightly unexpected behavior)
        attach_[isite] = -1;
        //exit(1);
        success = false;
    } else {
        //std::cout << "we good!\n";
        // Calculate the force/energy for later use
        double f[3] = {0.0};
        //CalcForce(parameters->n_dim,
        //          parameters->n_periodic,
        //          properties->unit_cell.h,
        //          isite,
        //          properties->bonds.r_bond[ibond],
        //          properties->bonds.s_bond[ibond],
        //          properties->bonds.u_bond[ibond],
        //          properties->bonds.length[ibond],
        //          f);
        success = true;
    }
    return success;
}

// Update the positions of everything via stepping
void Vesicle::Step(int ndim,
                   int nperiodic,
                   int nbonds,
                   double **h,
                   double **rbond,
                   double **sbond,
                   double **ubond,
                   double *lbond,
                   double delta,
                   gsl_rng *r) {
    // Check if we need to update the sites
    bool update_sites = false;
    for (int isite = 0; isite < nsites_; ++isite) {
        if (dot_product(ndim, dr_tot_[isite], dr_tot_[isite]) > 0.25 * SQR(SKIN_)) {
            update_sites = true;
        }
    }

    if (update_sites) {
        //std::cout << "Tripped update on sites!\n";
        double rcutoff_2 = SQR(parent_->rcutoff_ + SKIN_);
        UpdateNeighbors(ndim,
                        nperiodic,
                        nbonds,
                        h,
                        rbond,
                        sbond,
                        ubond,
                        lbond,
                        rcutoff_2);
    }

    // Now we walk and diffuse
    // Here is the walking
    if (parent_->motor_velocity_ != 0.0) {
        for (int isite = 0; isite < nsites_; ++isite) {
            if (attach_[isite] == -1)
                continue;
            int ibond = attach_[isite];
            double ui_dot_f = dot_product(ndim, ubond[ibond], flink_[isite]);
            double f_mag_j = -ui_dot_f; // following convention from croslsinks
            double velocity = parent_->motor_velocity_;
            if (f_mag_j*velocity > 0.0)
                f_mag_j = 0.0;
            else {
                // Assume parallel stall type
                f_mag_j = ABS(f_mag_j);
            }

            if (f_mag_j < parent_->motor_stall_force_)
                velocity *= 1.0 - f_mag_j/parent_->motor_stall_force_;
            else
                velocity = 0.0;

            //std::cout << "site[" << isite << "] adj vel: " << velocity << std::endl;
            cross_pos_[isite] += delta * velocity;
        }
    }
    // Here is the diffusion
    if (parent_->motor_diffusion_ > 0.0) {
        for (int isite = 0; isite < nsites_; ++isite) {
            if (attach_[isite] == -1)
                continue;
            int ibond = attach_[isite];
            //std::cout << "initial crosspos[" << isite << "]: " << cross_pos_[isite] << std::endl;
            // Check the tip enhancement and reduce the set size accordingly
            double dstab = lbond[ibond] - cross_pos_[isite];
            double diffusion_const = parent_->motor_diffusion_;
            // random 1d diffusion
            cross_pos_[isite] += sqrt(2.0 * diffusion_const * delta) *
                gsl_ran_gaussian_ziggurat(r, 1.0);
            // Force dependent diffusion
            double ui_dot_f = -dot_product(ndim, ubond[ibond], flink_[isite]);
            //std::cout << "force diffusion[" << isite << "] with bond[" << ibond << "]\n";
            //std::cout << "  ubond(" << ubond[ibond][0] << ", " << ubond[ibond][1] << ", " << ubond[ibond][2] << ")\n";
            //std::cout << "  flink(" << flink_[isite][0] << ", " << flink_[isite][1] << ", " << flink_[isite][2] << ")\n";
            //std::cout << "  ui_dot_f: " << ui_dot_f << std::endl;
            cross_pos_[isite] += ui_dot_f * diffusion_const * delta; // Force in the direction of diffusion
            // Tip enhancement diffusion
        }
    }

    // No matter what, check if we've gone off the end and reset this
    for (int isite = 0; isite < nsites_; ++isite) {
        int ibond = attach_[isite];
        if (ibond == -1)
            continue;
        
        if (cross_pos_[isite] > lbond[ibond]) {
            cross_pos_[isite] = lbond[ibond];
        }
        if (cross_pos_[isite] < 0.0) {
            cross_pos_[isite] = 0.0;
        }
    }
}
