#include "bob.h"
#include "lookup_table.h"
#include "macros.h"
#include "probabilities.h"
#include "vesicle.h"
#include "vesicle_management.h"

#include <gsl/gsl_integration.h>

#include <iostream>
#include <iomanip>

#define SKIN_ 8.0

VesicleManagement::VesicleManagement() {
}

void VesicleManagement::Init(system_parameters *parameters,
                             system_properties *properties,
                             const char* vesicle_file) {
    // Load the file and then continue
    YAML::Node node = YAML::LoadFile(vesicle_file);
    Init(parameters, properties, &node);
}

void VesicleManagement::Init(system_parameters *parameters,
                             system_properties *properties,
                             YAML::Node* node) {
    node_       = *node;
    parameters_ = parameters;
    properties_ = properties;

    nbonds_ = properties->bonds.n_bonds;
    ndim_   = parameters->n_dim;

    std::cout << "DO NOT UNDER ANY CIRCUMSTANCES USE THIS YET, IT IS STILL NOT FULLY WORKING!\n";
    exit(1);

    ParseParams();

    if (!enabled_) {
        nvesicles_ = 0;
        return;
    }

    // Calculate the cutoff stuff
    double max_length = 0.0;
    if (parameters->dynamic_instability_flag) {
        max_length = (parameters->n_periodic == 0) ? properties->unit_cell.h[0][0] :
            0.5 * properties->unit_cell.h[0][0];
    } else {
        max_length = 100.0; // FIXME XXX Set to something real from the bonds
    }
    CalcCutoff(max_length);

    // Build the integration tables
    BuildTables();

    // Initialize the neighbor lists
    InitNeighborLists();

    Print();
}

void VesicleManagement::ParseParams() {
    nvesicles_ = node_["vesicles"]["vesicle"].size();

    std::cout << "Number of vesicles: " << nvesicles_ << std::endl;

    if (nvesicles_ == 0) {
        std::cout << "No vesicles present, return before checking info\n";
        enabled_ = false;
        return;
    }
    enabled_ = true;

    try {
        v_diameter_ = node_["vesicles"]["properties"]["diameter"].as<double>();
        if (node_["vesicles"]["properties"]["translational_drag"]) {
            gamma_t_ = node_["vesicles"]["properties"]["translational_drag"].as<double>();
            gamma_r_ = node_["vesicles"]["properties"]["rotational_drag"].as<double>();
        } else {
            // Stoke's formula with eta = 1/3pi for a sphere to make units nice
            double v_rad = 0.5 * v_diameter_;
            gamma_t_ = 2 * v_rad;
            gamma_r_ = (8./3.) *v_rad*v_rad*v_rad;
        }
        nmotors_ = node_["vesicles"]["properties"]["motor_number"].as<int>();
        motor_k_ = node_["vesicles"]["properties"]["motor_spring"].as<double>();
        motor_r0_ = node_["vesicles"]["properties"]["motor_rest_length"].as<double>() + 0.5;
        motor_xc_ = node_["vesicles"]["properties"]["motor_characteristic_length"].as<double>();
        motor_eps_eff_ = node_["vesicles"]["properties"]["motor_concentration"].as<double>();
        motor_on_rate_ = node_["vesicles"]["properties"]["motor_on_rate"].as<double>();
        motor_diffusion_ = node_["vesicles"]["properties"]["motor_diffusion"].as<double>();
        motor_velocity_ = node_["vesicles"]["properties"]["motor_velocity"].as<double>();
        motor_stall_force_ = node_["vesicles"]["properties"]["motor_stall_force"].as<double>();


        // If everything is good, run the allocate command
        Allocate();


        // Now have to get the color information to load
        if (node_["vesicles"]["properties"]["replicate"]) {
            // Fake the initiailization of the first one over and over
            for (int iv = 0; iv < nvesicles_; ++iv) {
                color_[iv][0] = node_["vesicles"]["vesicle"][0]["color"][0].as<float>();
                color_[iv][1] = node_["vesicles"]["vesicle"][0]["color"][1].as<float>();
                color_[iv][2] = node_["vesicles"]["vesicle"][0]["color"][2].as<float>();
                color_[iv][3] = node_["vesicles"]["vesicle"][0]["color"][3].as<float>();
                CreateVesicle(iv, true);
            }
        } else {
            // initialize each
            for (int iv = 0; iv < nvesicles_; ++iv) {
                color_[iv][0] = node_["vesicles"]["vesicle"][iv]["color"][0].as<float>();
                color_[iv][1] = node_["vesicles"]["vesicle"][iv]["color"][1].as<float>();
                color_[iv][2] = node_["vesicles"]["vesicle"][iv]["color"][2].as<float>();
                color_[iv][3] = node_["vesicles"]["vesicle"][iv]["color"][3].as<float>();
                for (int icolor = 0; icolor < 4; ++icolor) {
                    if (node_["vesicles"]["vesicle"][iv]["color_motor"][0]) {
                        color_motor_[iv][icolor] = node_["vesicles"]["vesicle"][iv]["color_motor"][icolor].as<float>();
                    } else {
                        color_motor_[iv][icolor] = color_[iv][icolor];
                    }
                }
                CreateVesicle(iv);
            }
        }
    } catch (std::exception &ex) {
        std::cerr << "Exception thrown during parsing of Vesicle YAML file, check: \n";
        std::cerr << "   " << ex.what() << std::endl;
        enabled_ = false;
        return;
    }
}

// Allocate space for the arrays
void VesicleManagement::Allocate() {
    vesicles_ = new Vesicle[nvesicles_];
    color_ = (float**) allocate_2d_array(nvesicles_, 4, sizeof(float));
    color_motor_ = (float**) allocate_2d_array(nvesicles_, 4, sizeof(float));
    f_ = (double**) allocate_2d_array(nvesicles_, ndim_, sizeof(double));
    t_ = (double**) allocate_2d_array(nvesicles_, ndim_, sizeof(double));
    r_ = (double**) allocate_2d_array(nvesicles_, ndim_, sizeof(double));
    u_ = (double**) allocate_2d_array(nvesicles_, ndim_, sizeof(double));
    v_ = (double**) allocate_2d_array(nvesicles_, ndim_, sizeof(double));
    w_ = (double**) allocate_2d_array(nvesicles_, ndim_, sizeof(double));
    anchors_ = new al_list[nvesicles_];
    n_exp_ = (double*) allocate_1d_array(nvesicles_, sizeof(double));
    n_bound_ = (int*) allocate_1d_array(nvesicles_, sizeof(int));
    n_exp_lookup_ = (LookupTable*) allocate_1d_array(1, sizeof(LookupTable));

    // Any initial conditions
    for (int iv = 0; iv < nvesicles_; ++iv) {
        n_bound_[iv] = 0;
    }
}

// Create a single vesicle
void VesicleManagement::CreateVesicle(int iv,
                                      bool replicate) {
    YAML::Node node;
    if (replicate) {
        node = node_["vesicles"]["vesicle"][0];
    } else {
        node = node_["vesicles"]["vesicle"][iv];
    }

    std::string insertion_type = node["insertion_type"].as<std::string>();

    if (insertion_type == "xyz") {
        for (int i = 0; i < ndim_; ++i) {
            r_[iv][i] = node["r"][i].as<double>();
            u_[iv][i] = node["u"][i].as<double>();
        }
    } else {
        std::cerr << "No vesicle insertion of type " << insertion_type << std::endl;
        exit(1);
    }
    // Check if we have specified sites
    if (node["site"]) {
        sites_specified_ = true;
        site_specified_[0] = node["site"][0].as<double>();
        site_specified_[1] = node["site"][1].as<double>();
        site_specified_[2] = node["site"][2].as<double>();
    }

    // Initialize the vesicle
    vesicles_[iv].Init(this,
                       iv,
                       r_[iv],
                       u_[iv],
                       v_[iv],
                       w_[iv]);
}

// Calculate the cutoff for the tables based on the microtubule lengths
void VesicleManagement::CalcCutoff(double max_length) {
    if (print_info_) {
        std::cout << "   Calculating Vesicle 1->2 cutoff\n";
    }
    for (int ibond = 0; ibond < nbonds_; ++ibond) {
        max_length = MAX(properties_->bonds.length[ibond], max_length);
    }
    const double eps = 1E-3; // Tolerance for number of binding sites between pair
    const double temp = 1.0;
    double eps_eff = std::max(motor_eps_eff_, motor_eps_eff_);
    double epseffprime = eps_eff * exp(0.5 * motor_k_ * motor_xc_ * motor_xc_);
    double rc_0 = sqrt(2.0 / (motor_k_) *
                       temp *
                       log(epseffprime * max_length / eps *sqrt(2.0 * temp / motor_k_)));
    rcutoff_ = rc_0 + motor_r0_ + motor_xc_;
    rcutoff2_ = rcutoff_*rcutoff_;
}

// Biuld the integraiton tables
void VesicleManagement::BuildTables() {
    if (motor_r0_ != 0.0) {
        std::vector<double> x[2];
        double bin_size = 0.05;
        double alpha = motor_k_ / 2.;
        double eps = 1e-5;
        double a_cutoff = exp(alpha*motor_xc_*motor_xc_)/sqrt(alpha) * my_erfinv(1 - 4.0*sqrt(alpha/M_PI)*eps) +
            motor_r0_ + motor_xc_;
        double y_cutoff = rcutoff_;

        kc_params params;
        params.alpha = alpha;
        params.r0 = motor_r0_;
        params.xc = motor_xc_;

        for (double a = 0.0; a <= a_cutoff; a += bin_size)
            x[0].push_back(a);

        for (double y0 = 0.0; y0 <= y_cutoff; y0 += bin_size)
            x[1].push_back(y0);

        n_exp_lookup_->Init(2, x, &prob_1_2_fdep, &params);
    } else {
        std::cerr << "Tables cannot be built because you have a zero rest length!\n";
        exit(1);
    }
}

// Build the original neighbor lists
void VesicleManagement::InitNeighborLists() {
    for (int iv = 0; iv < nvesicles_; ++iv) {
        Vesicle *v = &(vesicles_[iv]);
        double r_cutoff2 = SQR(rcutoff_ + SKIN_);
        v->UpdateNeighbors(parameters_->n_dim,
                           parameters_->n_periodic,
                           properties_->bonds.n_bonds,
                           properties_->unit_cell.h,
                           properties_->bonds.r_bond,
                           properties_->bonds.s_bond,
                           properties_->bonds.u_bond,
                           properties_->bonds.length,
                           r_cutoff2);
    }
}


// Output Functions
void VesicleManagement::Print() {
    if (!print_info_) {
        return;
    }
    std::cout << "Vesicle Common Properties\n";
    std::cout << "   --- Vesicle Properties ---" << std::endl;
    std::cout << "   vesicle diameter: " << v_diameter_ << std::endl;
    std::cout << "   --- Vesicle Motor Properties ---" << std::endl;
    std::cout << "   Number of Attached Motors: " << nmotors_ << std::endl;
    std::cout << "   Motor spring: " << motor_k_ << std::endl;
    std::cout << "   Motor rest length (real): " << motor_r0_ - 0.5 << " (simadj: " << motor_r0_ << ")\n";
    std::cout << "   Motor characteristic length: " << motor_xc_ << std::endl;
    std::cout << "   Motor eps eff: " << motor_eps_eff_ << std::endl;
    std::cout << "   Motor on rate: " << motor_on_rate_ << std::endl;
    std::cout << "   Motor diffusion: " << motor_diffusion_ << std::endl;
    std::cout << "   Motor rcutoff: " << rcutoff_ << std::endl;
    std::cout << "   Motor velocity: " << motor_velocity_ << std::endl;
    std::cout << "   Motor stall force: " << motor_stall_force_<< std::endl;
    for (int iv = 0; iv < nvesicles_; ++iv) {
        std::cout << "Vesicle " << iv << ":" << std::endl;
        PrintVesicle(iv);
    }
}

void VesicleManagement::PrintVesicle(int iv) {
    std::cout << "      r = [" << vesicles_[iv].r_[0] << ", "
                               << vesicles_[iv].r_[1] << ", "
                               << vesicles_[iv].r_[2] << "]\n";
    std::cout << "      u = [" << vesicles_[iv].u_[0] << ", "
                               << vesicles_[iv].u_[1] << ", "
                               << vesicles_[iv].u_[2] << "]\n";
    std::cout << "      v = [" << vesicles_[iv].v_[0] << ", "
                               << vesicles_[iv].v_[1] << ", "
                               << vesicles_[iv].v_[2] << "]\n";
    std::cout << "      w = [" << vesicles_[iv].w_[0] << ", "
                               << vesicles_[iv].w_[1] << ", "
                               << vesicles_[iv].w_[2] << "]\n";
    for (auto p =  anchors_[iv].begin();
              p != anchors_[iv].end();
              ++p) {
        std::cout << "         motor = [" << p->pos[0] << ", "
                                          << p->pos[1] << ", "
                                          << p->pos[2] << "]\n";
    }
}

// Print information on the current frame
void VesicleManagement::PrintFrame() {
    std::cout << "****************\n";
    std::cout << "Vesicles Print Frame: " << properties_->i_current_step << std::endl;
    for (int iv = 0; iv < nvesicles_; ++iv) {
        Vesicle *v = &(vesicles_[iv]);
        v->PrintFrame();
    }
}

void VesicleManagement::UpdatePositions() {

}


// KMC functions
void VesicleManagement::PrepKMC(system_parameters *parameters,
                                system_properties *properties){
    Update_1_2_Probability(parameters, properties);
}

void VesicleManagement::StepKMC(system_parameters *parameters,
                                system_properties *properties) {
    if (!enabled_) {
        return;
    }
    PrepKMC(parameters, properties);
    RunKMC(parameters, properties);
}

void VesicleManagement::RunKMC(system_parameters *parameters,
                               system_properties *properties){

    // Randomly decide to do detach->attach or reverse
    int g[2] = {0, 1};
    for (int i = 0; i < 2; ++i) {
        int j = gsl_rng_uniform_int(properties->rng.r, 2);
        int swap = g[i];
        g[i] = g[j];
        g[j] = swap;
    }

    for (int i = 0; i < 2; ++i) {
        switch (g[i]) {
            case 0:
                KMC_1_2(parameters, properties);
                break;
            case 1:
                KMC_2_1(parameters, properties);
                break;
        }
    }

    // Check if the vesicles need to rebuild their neighbor list
    for (int iv = 0; iv < nvesicles_; ++iv) {
        Vesicle *v = &(vesicles_[iv]);
        if (v->trip_update_on_removal_) {
            v->trip_update_on_removal_ = false;
            double r_cutoff2 = SQR(rcutoff_ + SKIN_);
            v->UpdateNeighbors(parameters->n_dim,
                               parameters->n_periodic,
                               properties->bonds.n_bonds,
                               properties->unit_cell.h,
                               properties->bonds.r_bond,
                               properties->bonds.s_bond,
                               properties->bonds.u_bond,
                               properties->bonds.length,
                               r_cutoff2); 
        }
    }

    // DEBUG
    //for (int ikc = 0; ikc < nkcs_; ++ikc) {
    //    std::cout << "KC[" << ikc << "], nbound: " << n_bound_[ikc] << std::endl;
    //}
}

void VesicleManagement::Update_1_2_Probability(system_parameters *parameters,
                                system_properties *properties){
    // Zero out everything
    for (int im = 0; im < nmotors_; ++im) {
        n_exp_[im] = 0.0;
    }

    // Loop over the vesicles
    for (int iv = 0; iv < nvesicles_; ++iv) {
        Vesicle *viter = &(vesicles_[iv]);
        viter->Step(ndim_,
                    parameters->n_periodic,
                    properties->bonds.n_bonds,
                    properties->unit_cell.h,
                    properties->bonds.r_bond,
                    properties->bonds.s_bond,
                    properties->bonds.u_bond,
                    properties->bonds.length,
                    parameters->delta,
                    properties->rng.r);
        viter->Update_1_2(ndim_,
                          parameters_->n_periodic,
                          properties->bonds.n_bonds,
                          properties->unit_cell.h,
                          properties->bonds.r_bond,
                          properties->bonds.s_bond,
                          properties->bonds.u_bond,
                          properties->bonds.length);
        n_exp_[iv] += viter->n_exp_tot_;
    } // loop over vesicles
}

void VesicleManagement::KMC_1_2(system_parameters *parameters,
                                system_properties *properties){
    for (int iv = 0; iv < nvesicles_; ++iv) {
        // How many attach?
        int Ntot = gsl_ran_poisson(properties->rng.r,
                                   n_exp_[iv] * parameters->delta);
        // Now do the attachment
        for (int itrial = 0; itrial < Ntot; ++itrial) {
            Vesicle *v = &(vesicles_[iv]);
            bool successful_insert = v->Insert_1_2(parameters,
                                                   properties);
            if (successful_insert) {
                n_bound_[iv]++;
                //PrintFrame();
            }
        }
    }
}

void VesicleManagement::KMC_2_1(system_parameters *parameters,
                               system_properties *properties){

}

void VesicleManagement::KMC_2_1_FIndep(system_parameters *parameters,
                               system_properties *properties){

}

void VesicleManagement::KMC_2_1_FDep(system_parameters *parameters,
                               system_properties *properties){

}

void VesicleManagement::WriteState(system_parameters *parameters,
                                    system_properties *properties,
                                    const char* dumpfile){

}

void VesicleManagement::ReadState(system_parameters *parameters,
                               system_properties *properties){

}
void VesicleManagement::Restart(system_parameters *parameters,
                                 system_properties *properties,
                                 const std::string& restart_name){

}

// Helper functions
void VesicleManagement::CloseAll(){

}

void VesicleManagement::ConsistencyCheck(){

}
