#ifndef _VESICLE_MANAGEMENT_H_
#define _VESICLE_MANAGEMENT_H_

#include <vector>
#include <yaml-cpp/yaml.h>

class Vesicle;

#ifndef _PARAMETERS_H
typedef struct system_parameters system_parameters;
#endif
#ifndef _SYSTEM_PROPERTIES_H
typedef struct system_properties system_properties;
#endif

#include "anchor_list.h"
#include "helpers.h"
#include "neighbors.h"
#include <gsl/gsl_rng.h>

#ifndef _LOOKUP_TABLE_H
class LookupTable;
#endif

class VesicleManagement {
  private:

    std::string filename_ = "vesicles.posit";
    FILE *outfile_ = nullptr;
    FILE *infile_ = nullptr;
    YAML::Node node_;

  public:

    bool sites_specified_ = false;
    bool write_posit_ = false;
    bool thermal_diffusion_ = true;
    bool enabled_ = false;
    bool print_info_ = true;

    int ndim_ = 3;
    int nbonds_ = 0;
    int nvesicles_ = 0; // how many vesicles are there?
    int nmotors_ = 0; // how many motors per vesicle?

    double v_diameter_ = 0.0;
    double gamma_t_ = 0.0;
    double gamma_r_ = 0.0;
    double motor_k_ = 0.0;
    double motor_r0_ = 0.0;
    double motor_xc_ = 0.0;
    double motor_eps_eff_ = 0.0;
    double motor_on_rate_ = 0.0;
    double motor_diffusion_ = 0.0;
    double motor_velocity_ = 0.0;
    double motor_stall_force_ = 0.0;

    double rcutoff_ = 0.0;
    double rcutoff2_ = 0.0;
    double site_specified_[3] = {0.0};

    int *n_bound_;
    double *n_exp_;

    float **color_;
    float **color_motor_;
    double **f_;
    double **t_;
    double **r_;
    double **u_;
    double **v_;
    double **w_;

    Vesicle *vesicles_;
    al_list *anchors_;
    LookupTable *n_exp_lookup_;

    system_parameters *parameters_ = nullptr;
    system_properties *properties_ = nullptr;

  private:

    void CalcCutoff(double max_length);
    void BuildTables();
    void InitNeighborLists();

  public:

    VesicleManagement();
    void Init(system_parameters *parameters,
              system_properties *properties,
              const char* vesicle_file);
    void Init(system_parameters *parameters,
              system_properties *properties,
              YAML::Node* node);

    void Allocate();
    void ParseParams();
    void CreateVesicle(int iv,
                       bool replicate = false);

    // Output functions
    void Print();
    void PrintVesicle(int iv);
    void PrintFrame();
    void UpdatePositions();
    void ChangeFilename(const std::string &filename) {
        filename_ = filename;
    }

    // KMC functions
    void PrepKMC(system_parameters *parameters,
                 system_properties *properties);
    void StepKMC(system_parameters *parameters,
                 system_properties *properties);
    void RunKMC(system_parameters *parameters,
                system_properties *properties);
    void Update_1_2_Probability(system_parameters *parameters,
                                system_properties *properties);
    void KMC_1_2(system_parameters *parameters,
                 system_properties *properties);
    void KMC_2_1(system_parameters *parameters,
                 system_properties *properties);
    void KMC_2_1_FIndep(system_parameters *parameters,
                        system_properties *properties);
    void KMC_2_1_FDep(system_parameters *parameters,
                      system_properties *properties);
    void WriteState(system_parameters *parameters,
                    system_properties *properties,
                    const char* dumpfile = nullptr);
    void ReadState(system_parameters *parameters,
                   system_properties *properties);
    void Restart(system_parameters *parameters,
                 system_properties *properties,
                 const std::string& restart_name);

    // Helper functions
    void CloseAll();
    void ConsistencyCheck();

};

#endif
