/* This routine writes the unit cell matrix and site positions to an output file.
   It's unique in that the number of sites is also output, not readable by standard reader program.

   Input: number of spatial dimensions (n_dim)
          number of sites (n_sites)
          unit cell matrix (h)
          array of site positions (r)
          array of anchor positions (ra)
          array of anchor orientations (ua)
          pointer to output file (f_posit)

   Output: the unit cell matrix and site positions are written to the output file 
           timestamp
           sites
           anchors positions
           anchors orientations
 */

#include "bob.h"

void write_positions_spb_dynamics(int n_dim, int n_sites, int n_bonds, int n_anchors,
                                  double **h, double **r, double **ra, double **ua,
                                  double time, FILE * f_posit)
{
    int i, i_site, i_anchor, j;
    float time_float, h_float;
    static int first_call = 1;
    static float **r_float;
    static float **ra_float;
    static float **ua_float;
    
    if (first_call) {
        r_float = (float**) allocate_2d_array(n_sites, n_dim, sizeof(float));
        ra_float = (float**) allocate_2d_array(n_anchors, n_dim, sizeof(float));
        ua_float = (float**) allocate_2d_array(n_anchors, n_dim, sizeof(float));

        first_call = 0;
    }

    /* Write unit cell matrix to output file. */
    for (i = 0; i < n_dim; ++i) {
        for (j = 0; j < n_dim; ++j) {
            h_float = (float) h[i][j];
            fwrite(&h_float, sizeof(float), 1, f_posit);
        }
    }

    for (i_site = 0; i_site < n_sites; ++i_site) {
        for (i = 0;  i < n_dim; ++i)
            r_float[i_site][i] = (float) r[i_site][i];
    }
    
    time_float = (float) time;
    fwrite(&time_float, sizeof(float), 1, f_posit);

    /* Write site positions to output file. */
    fwrite(r_float[0], sizeof(float), n_dim * n_sites, f_posit);

    /* Write anchor positions and orientations to file */
    for (i_anchor = 0; i_anchor < n_anchors; ++i_anchor) {
        for (i = 0;  i < n_dim; ++i) {
            ra_float[i_anchor][i] = (float) ra[i_anchor][i];
            ua_float[i_anchor][i] = (float) ua[i_anchor][i];
        }
    }

    fwrite(ra_float[0], sizeof(float), n_dim * n_anchors, f_posit);
    fwrite(ua_float[0], sizeof(float), n_dim * n_anchors, f_posit);

    return;
}
