// Write various thermodynamic quantities from the spindle

#include "bob.h"

// Header:
//      ndim    int
//      ncomp   int
//      h       ndim x ndim float
// File
//      istep   int
//      u       float (total energy)
//      ucomp   ncomp (each potential's energy)
//      force   force (each potential's force on microtubules (total) contribution)
//      torque  torque (each potential's torque on microtubules (total) contribution)
//      virial  virial (each potential's virial contribution)

#include <iostream>

std::string get_potential_name(system_potential *potential, int icomp) {
    std::string potname;

    // Huge ugly if/else block for potential naming, ugh 
    if (potential->pot_func[icomp] == brownian_sphero_neighbor_lists_mp) {
        potname = "brownian_sphero_neighbor_lists_mp";
    } else if (potential->pot_func[icomp] == crosslink_interaction_bd_mp) {
        potname = "crosslink_interaction_bd_mp";
    } else if (potential->pot_func[icomp] == anchor_potential_bd) {
        potname = "anchor_potential_bd";
    } else if (potential->pot_func[icomp] == anchor_wca_potential_bd) {
        potname = "anchor_wca_potential_bd";
    } else if (potential->pot_func[icomp] == spb_spring_potential_angular) {
        potname = "spb_spring_potential_angular";
    } else if (potential->pot_func[icomp] == chromosome_chromatin_potential) {
        potname = "chromosome_chromatin_potential";
    } else if (potential->pot_func[icomp] == kc_wall_wca_potential) {
        potname = "kc_wall_wca_potential";
    } else if (potential->pot_func[icomp] == chromosome_mt_soft_gaussian_potential_allpairs) {
        potname = "chromosome_mt_soft_gaussian_potential_allpairs";
    } else if (potential->pot_func[icomp] == af_mt_harmonic_potential) {
        potname = "af_mt_harmonic_potential";
    } else if (potential->pot_func[icomp] == af_mt_quartic_potential) {
        potname = "af_mt_quartic_potential";
    } else if (potential->pot_func[icomp] == kinetochoremesh_mt_wca_potential_allpairs) {
        potname = "kinetochoremesh_mt_wca_potential_allpairs";
    } else if (potential->pot_func[icomp] == wca_sphero_wall_potential_bd) {
        potname = "wca_sphero_wall_potential_bd";
    } else if (potential->pot_func[icomp] == NM_sphero_wall_potential_bd) {
        potname = "NM_sphero_wall_potential_bd";
    } else if (potential->pot_func[icomp] == anchor_lj_potential_bd) {
        potname = "anchor_lj_potential_bd";
    } else if (potential->pot_func[icomp] == kinetochorepoint_mt_wca_potential_allpairs) {
        potname = "kinetochorepoint_mt_wca_potential_allpairs";
    }

    return potname;
}

void write_thermo_spindle(system_parameters *parameters,
                          system_properties *properties,
                          system_potential  *potential,
                          FILE *f_thermo,
                          FILE *f_pole,
                          FILE *f_forces) {
    static bool first_call_thermo_spindle = true;

    // Set up shortcuts
    int ndim = parameters->n_dim;
    int ncomp = potential->n_comp;
    int ntypes = properties->crosslinks.n_types_;
    int nanchors = properties->anchors.n_anchors;
    int nbonds = properties->bonds.n_bonds;
    double **h = properties->unit_cell.h;

    al_list *anchor_list = properties->anchors.anchor_list;
    double r_cutoff = pow(2,1.0/6.0);
    double conf_rad = 0.5 * properties->unit_cell.h[0][0] - r_cutoff + 0.5;
    double conf_rad2 = SQR(conf_rad);

    // Write out various pieces of information
    if (first_call_thermo_spindle) {
        first_call_thermo_spindle = false;
        // New methods of writing things out in plaintext
        // First, write out the header information for the total info
        fprintf(f_thermo, "HEADERINFO\n");
        fprintf(f_forces, "HEADERINFO\n");
        fprintf(f_thermo, "ndim %d\n", ndim);
        fprintf(f_forces, "ndim %d\n", ndim);
        fprintf(f_thermo, "ncomp %d\n", ncomp);
        fprintf(f_thermo, "ntypes %d\n", ntypes);
        fprintf(f_forces, "ntypes %d\n", ntypes);
        fprintf(f_thermo, "nanchors %d\n", nanchors);
        fprintf(f_forces, "nanchors %d\n", nanchors);

        // Write out the information on which potential is which, and this will also
        // tell us which of the potentaisl is broken up into the crosslinks
        for (int icomp = 0; icomp < ncomp; ++icomp) {
            std::string potname = get_potential_name(potential, icomp);
            if (potential->pot_func[icomp] == crosslink_interaction_bd_mp) {
                for (int itype = 0; itype < ntypes; ++itype) {
                    fprintf(f_thermo, "%d %d %s_%d\n", icomp, itype, potname.c_str(), itype);
                }
            } else {
                fprintf(f_thermo, "%d %s\n", icomp, potname.c_str());
            }
        }
        fprintf(f_thermo, "ENDHEADERINFOEND\n");
        fprintf(f_forces, "ENDHEADERINFOEND\n");

        // Now write specifics
        fprintf(f_thermo, "step ");
        for (int i = 0; i < ndim; ++i) {
            for (int j = 0; j < ndim; ++j) {
                fprintf(f_thermo, "h_%d%d ", i, j);
            }
        }

        // The force on each SPB
        for (int ianchor = 0; ianchor < nanchors; ++ianchor) {
            for (int i = 0; i < ndim; ++i) {
                fprintf(f_thermo, "spb_%d_%d ", ianchor, i);
                fprintf(f_pole, "spb_%d_%d ", ianchor, i);
            }
        }

        // Now ALL the virial contributions
        for (int icomp = 0; icomp < ncomp; ++icomp) {
            if (potential->pot_func[icomp] == crosslink_interaction_bd_mp) {
                for (int itype = 0; itype < ntypes; ++itype) {
                    for (int i = 0; i < ndim; ++i) {
                        for (int j = 0; j < ndim; ++j) {
                            fprintf(f_thermo, "virial_%d_%d_%d_%d ", icomp, itype, i, j);
                        }
                    }
                }
            } else {
                for (int i = 0; i < ndim; ++i) {
                    for (int j = 0; j < ndim; ++j) {
                        fprintf(f_thermo, "virial_%d_%d_%d ", icomp, i, j);
                    }
                }
            }
        }

        // Write out the format of the forces file, just to be safe
        // Crosslinks
        for (int ipole = 0; ipole < nanchors; ++ipole) {
            for (int itype = 0; itype < ntypes; ++itype) {
                for (int i = 0; i < ndim; ++i) {
                    fprintf(f_forces, "xlink_%d_%d_%d ", ipole, itype, i);
                }
            }
        }

        // AF by pole
        for (int ipole = 0; ipole < nanchors; ++ipole) {
            for (int i = 0; i < ndim; ++i) {
                fprintf(f_forces, "af_%d_%d ", ipole, i);
            }
        }

        // Chromosome by chromosome!
        for (int ic = 0; ic < properties->chromosomes.nchromosomes_; ++ic) {
            for (int i = 0; i < ndim; ++i) {
                fprintf(f_forces, "interkc_%d_%d ", ic, i);
            }
        }

        fprintf(f_thermo, "\n");
        fprintf(f_pole, "\n");
        fprintf(f_forces, "\n");
    }

    // Now, write out the actual information for all the steps, etc
    // Current step
    fprintf(f_thermo, "%d ", properties->i_current_step);

    // Current h
    for (int i = 0; i < ndim; ++i) {
        for (int j = 0; j < ndim; ++j) {
            fprintf(f_thermo, "%g ", h[i][j]);
        }
    }

    // SPB forces
    for (int ianchor = 0; ianchor < nanchors; ++ianchor) {
        for (int i = 0; i < ndim; ++i) {
            fprintf(f_thermo, "%g ", properties->anchors.f_anchor[ianchor][i]);
        }
    }

    // Now ALL the virial contributions
    for (int icomp = 0; icomp < ncomp; ++icomp) {
        if (potential->pot_func[icomp] == crosslink_interaction_bd_mp) {
            for (int itype = 0; itype < ntypes; ++itype) {
                for (int i = 0; i < ndim; ++i) {
                    for (int j = 0; j < ndim; ++j) {
                        fprintf(f_thermo, "%g ", properties->thermo.virial_xlink[itype][i][j]);
                    }
                }
            }
        } else {
            for (int i = 0; i < ndim; ++i) {
                for (int j = 0; j < ndim; ++j) {
                    fprintf(f_thermo, "%g ", potential->virial_comp[icomp][i][j]);
                }
            }
        }
    }


    fprintf(f_thermo, "\n");
    fflush(f_thermo);

    // Now look at the pole forces information. This is read out from the SPBs, as well
    // as the NE force on the MTs. This is to reconstruct the 'true' force on the spindle pole via
    // fpole = -fspb - sum(fmt,ne)

    // Quickly construct the MT minus-end force
    // FIXME: Always assumes we are using a WCA potential that affects MT minus-ends
    for (int ianchor = 0; ianchor < nanchors; ++ianchor) {
        // Grab the SPB force, take the negative because pole
        double fpole[3] = {0.0};
        for (int idim = 0; idim < ndim; ++idim) {
            fpole[idim] = -properties->anchors.f_anchor[ianchor][idim];
        }

        // Get the bonds associted with this SPB
        for (al_list::iterator p = anchor_list[ianchor].begin();
             p < anchor_list[ianchor].end();
             p++) {
            int ibond = p->label;

            // Unfortunatley, have to recalculate the minus-end force on the sPB
            double f_cutoff = 0.1 / parameters->delta * properties->bonds.gamma_par[ibond];
            
            double r_mag2_0 = 0.0;
            double r_contact_0[3];
            for (int i = 0; i < parameters->n_dim; ++i) {
                r_contact_0[i] = -0.5 * properties->bonds.length[ibond] * properties->bonds.u_bond[ibond][i];
                r_mag2_0 += SQR(properties->bonds.r_bond[ibond][i] + r_contact_0[i]);
            }

            double f[3] = {0.0};

            //Minus end of the MT touching the wall
            if (r_mag2_0 > conf_rad2 && !parameters->force_poly_flag) {
                double r_mag = sqrt(r_mag2_0);
                
                double delta_r = -r_mag + conf_rad + r_cutoff;
                double rho2 = 1.0 / SQR(delta_r);
                double rho6 = CUBE(rho2);
                double rho12 = SQR(rho6);
                double factor = 24.0 * (2.0 * rho12 - rho6) * rho2;
                
                if (factor * delta_r > f_cutoff) {
                    //std::cout << "NOTE tripping fcutoff in single site, resetting force factor, original " << factor << " to ";
                    factor = f_cutoff / delta_r;
                    //std::cout << factor << std::endl;
                }
                for (int i = 0; i < parameters->n_dim; ++i) {
                    f[i] = -factor * delta_r *
                        (properties->bonds.r_bond[ibond][i] + r_contact_0[i]) / r_mag;
                }
            } // The MT was interacting with the minus end

            // Subtract off contribution
            for (int i = 0; i < ndim; ++i) {
                fpole[i] -= f[i];
            }
        } // Loop over anchor bonds

        // Now write out the force on the pole!
        for (int i = 0; i < ndim; ++i) {
            fprintf(f_pole, "%g ", fpole[i]);
        }
    } // Loop over anchors
    fprintf(f_pole, "\n");
    fflush(f_pole);


    // Write out the forces for crosslinkers, AFs, and inter-kinetochores
    // Crosslinkers
    auto xlink_forces = properties->crosslinks.GenerateForces(); 
    for (int ipole = 0; ipole < properties->anchors.n_anchors; ++ipole) {
        for (int itype = 0; itype < properties->crosslinks.n_types_; ++itype) {
            for (int i = 0; i < ndim; ++i) {
                fprintf(f_forces, "%g ", xlink_forces[ipole][itype][i]);
            }
        }
    }

    // AFs
    auto af_forces = properties->chromosomes.GenerateAFForces();
    for (int ipole = 0; ipole < properties->anchors.n_anchors; ++ipole) {
        for (int i = 0; i < ndim; ++i) {
            fprintf(f_forces, "%g ", af_forces[ipole][i]);
        }
    }

    // Inter chromosome stuff
    auto kc_forces = properties->chromosomes.GenerateInterKCForces();
    for (int ic = 0; ic < properties->chromosomes.nchromosomes_; ++ic) {
        for (int i = 0; i < ndim; ++i) {
            fprintf(f_forces, "%g ", kc_forces[ic][i]);
        }
    }

    fprintf(f_forces, "\n");
    fflush(f_forces);
}
