// Xlink KMC properties to make sure it works correctly
// For some periodic boundary conditions

#include "bob.h"
#include <sys/time.h>
#include "correlation_data.h"
#include "diffusion_properties.h"
#include "xlink_management.h"
#include "xlink_entry.h"
#ifndef NOGRAPH
#include "graphics.h"
#endif

#include <iostream>

// Main program
int main(int argc, char *argv[]) {
    system_parameters parameters;
    system_properties properties;
    system_potential potential;
    FILE *f_posit;
    char default_file[F_MAX];
    int n_dim, n_steps, n_posit, n_thermo, n_graph, graph_flag;
    int n_bonds;
    double **h, **r, **s, **ra, **ua;

    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " default_file\n";
        exit(1);
    }
    strcpy(default_file, argv[1]);

    // Read the default parameters
    parse_parameters(default_file, &parameters);

    // Run the init_diffusion free code to not turn on interactions
    init_xlink_analysis(&parameters, &properties, &potential);

    n_dim = parameters.n_dim;
    n_steps = parameters.n_steps;
    n_posit = parameters.n_posit;
    n_thermo = parameters.n_thermo;
    n_graph = parameters.n_graph;
    graph_flag = parameters.graph_flag;
    h = properties.unit_cell.h;
    r = properties.sites.r;
    s = properties.sites.s;
    ra = properties.anchors.r_anchor;
    ua = properties.anchors.u_anchor;
    n_bonds = properties.bonds.n_bonds;

    #ifndef NOGRAPH
    /* Activate graphics if graph_flag == 1. */
    Graphics graphics;
    if (graph_flag == 1) {
        graphics.Init(&parameters, parameters.n_dim,
                      properties.unit_cell.h, 0);
        graphics.ResizeWindow(800,800);
        graphics.SetBoundaryType("sphere");
        graphics.DrawLoop(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.crosslinks.n_types_,
                          properties.crosslinks.stage_0_xlinks_,
                          properties.crosslinks.stage_2_xlinks_);
    }
    #endif

    f_posit = gfopen("xlink_analysis.posit" ,"w");
    //f_thermo = gfopen("diffusion_analysis_free.thermo", "w");

    properties.write_header_func(&parameters, &properties, f_posit);

    // Correlation data information
    CorrelationData motor_dist_1d;
    CorrelationData motor_dist_2d;
    CorrelationData n_exp_dist;
    CorrelationData psi1_dist;
    double fudge = 0.0;
    if (n_bonds == 2) {
        {
            double bin_size[] = {parameters.bin_size};
            double bin_min[] = {-5};
            double bin_max[] = {5};
            motor_dist_1d.Init(1, bin_size, bin_min, bin_max);
        }
        {
            double bin_size[] = {parameters.bin_size, parameters.bin_size};
            double bin_min[] = {-fudge, -fudge};
            double bin_max[] = {properties.bonds.length[0] + fudge, properties.bonds.length[1] + fudge};
            motor_dist_2d.Init(2, bin_size, bin_min, bin_max);
        }
        {
            double bin_size[] = {parameters.bin_size};
            double bin_min[] = {0-fudge};
            double bin_max[] = {properties.bonds.length[0]+fudge};
            psi1_dist.Init(1, bin_size, bin_min, bin_max);
            n_exp_dist.Init(1, bin_size, bin_min, bin_max);
        }
    }

    // Run the simulation
    for (int istep = 0; istep < n_steps; ++istep) {
        properties.time += parameters.delta;
        properties.i_current_step = istep;

        // Run the xlink kmc
        properties.crosslinks.StepKMC(&parameters, &properties);

        // Position step without any other stuff should just run
        // without potentials
        position_step_spindle_bd_mp(&parameters, &properties, &potential);

        // Generate our distributions
        if (n_bonds == 2) {
            int itype = 0; int ibond = 0;
            for (xlink_list::iterator xlink = properties.crosslinks.stage_2_xlinks_[itype][ibond].begin();
                                      xlink < properties.crosslinks.stage_2_xlinks_[itype][ibond].end();
                                      ++xlink) {
                if (xlink->IsActive()) {
                    double f[3] = {0.0};
                    xlink->CalcForce(parameters.n_dim,
                                     0,
                                     properties.unit_cell.h,
                                     properties.bonds.r_bond,
                                     properties.bonds.s_bond,
                                     properties.bonds.u_bond,
                                     properties.bonds.length,
                                     f);

                    // Should still be active
                    if ((xlink->cross_position_[0] > 10) &&
                        (xlink->cross_position_[1] > 10)) {
                        double s = xlink->cross_position_[1] - xlink->cross_position_[0];
                        motor_dist_1d.Bin(1, &s, 1.0);
                    }
                    motor_dist_2d.Bin(2, xlink->cross_position_, 1.0);
                }
            }

            // stage 1 stuff
            if (properties.crosslinks.attachment_model_ == 2) {
                int itype = 0; ibond = 0;
                for (xlink_list::iterator xlink = properties.crosslinks.stage_1_xlinks_[itype][ibond].begin();
                                          xlink < properties.crosslinks.stage_1_xlinks_[itype][ibond].end();
                                          ++xlink) {
                    if (xlink->IsActive()) {
                        psi1_dist.Bin(1, &xlink->cross_position_[0], 1.0);
                        n_exp_dist.Bin(1, &xlink->cross_position_[0], xlink->n_exp_);
                    }
                }
            }
        }

        #ifndef NOGRAPH
        /* Display configuration. */
        if (graph_flag && istep % n_graph == 0) {
            graphics.Draw(properties.bonds.n_bonds,
                          properties.unit_cell.h,
                          properties.bonds.r_bond,
                          properties.bonds.u_bond,
                          properties.bonds.length,
                          properties.crosslinks.n_types_,
                          properties.crosslinks.stage_0_xlinks_,
                          properties.crosslinks.stage_1_xlinks_,
                          properties.crosslinks.stage_2_xlinks_);
        }
        #endif
        if (istep % n_posit == 0) {
            update_bond_site_positions_mp(n_dim, parameters.n_periodic, properties.bonds.n_bonds,
                                          properties.sites.n_sites, h, properties.unit_cell.h_inv,
                                          properties.bonds.bond_site_1, properties.bonds.bond_site_2,
                                          properties.bonds.r_bond, properties.bonds.u_bond, properties.bonds.
                                          length, r, properties.sites.s);
            write_positions_spb_dynamics(n_dim,
                                         properties.sites.n_sites,
                                         properties.bonds.n_bonds,
                                         properties.anchors.n_anchors, 
                                         h, r, ra, ua,
                                         properties.time, f_posit);
            properties.crosslinks.WriteState(&parameters, &properties);
        }
    }

    // Output this stuff
    //motor_dist_1d.NormalizeByConstant(1.0/parameters.n_steps);
    //motor_dist_1d.OutputBinary("motor_dist_1d");

    motor_dist_2d.NormalizeByConstant(1.0/parameters.n_steps);
    motor_dist_2d.OutputBinary("motor_dist_2d");

    //psi1_dist.NormalizeByConstant(1.0/parameters.n_steps);
    //psi1_dist.OutputBinary("psi1_dist");

    //n_exp_dist.NormalizeByConstant(1.0/parameters.n_steps);
    //n_exp_dist.OutputBinary("n_exp_dist");

    std::cout << "Successful run!\n";

    return 0;
}
