#include "bob.h"
#include "xlink_entry.h"
#include "xlink_management.h"
#include "minimum_distance.h"
#include "lookup_table.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

#include <iostream>

double dot_product(int, double*, double*);
void error_exit(const char *error_msg, ...);

XlinkEntry::XlinkEntry(XlinkManagement *parent) : parent_(parent) {
    this->active_ = 0;
    this->direction_state_[0] = 0;
    this->direction_state_[1] = 0;
}

XlinkEntry::XlinkEntry(XlinkManagement *parent,
                       int n_dim, int n_periodic,
                       double **h, double **r_bond,
                       double **s_bond, double **u_bond,
                       double *length,
                       int type, int bond_1, int bond_2,
                       double lambda, double mu,
                       int direction_state[2]) : parent_(parent) {

    this->Init(parent, n_dim, n_periodic, h,
               r_bond, s_bond, u_bond, length,
               type, bond_1, bond_2, lambda, mu,
               direction_state);
}


XlinkEntry::XlinkEntry(XlinkManagement *parent,
                       int n_dim, int n_periodic,
                       double **h, double **r_bond,
                       double **s_bond, double **u_bond,
                       double *length,
                       int type, int head_type, int bond_1, 
                       double lambda,
                       gsl_rng *r_rng,
                       nl_list *neighbs,
                       int direction_state[2]) : parent_(parent) {
    
    this->Init(parent, n_dim, n_periodic, h,
               r_bond, s_bond, u_bond, length, type, head_type,
               bond_1, lambda, r_rng, neighbs,
               direction_state);
}

XlinkEntry::XlinkEntry(XlinkManagement *parent,
                       int n_dim, int n_periodic,
                       double **h, double **r_bond,
                       double **s_bond, double **u_bond,
                       double *length,
                       int type, int bond_1, int bond_2,
                       gsl_rng *r_rng,
                       int direction_state[2]) : parent_(parent) {

    this->Init(parent, n_dim, n_periodic, h,
               r_bond, s_bond, u_bond, length,
               type, bond_1, bond_2, r_rng,
               direction_state);
}

bool XlinkEntry::Init(XlinkManagement *parent,
                      int n_dim, int n_periodic,
                      double **h, double **r_bond, double **s_bond,
                      double **u_bond, double *length,
                      int type, int bond_1, int bond_2,
                      double lambda, double mu,
                      int direction_state[2]) {
    this->parent_ = parent;
    this->SetActive();
    this->type_ = type;
    this->head_parent_[0] = bond_1;
    this->head_parent_[1] = bond_2;
    this->cross_position_[0] = lambda + 0.5 * length[bond_1];
    this->cross_position_[1] = mu + 0.5 * length[bond_2];
    this->stage_ = 2;
    this->direction_state_[0] = direction_state[0];
    this->direction_state_[1] = direction_state[1];

    // Calculate energy/force for new link
    double f[3];
    CalcForce(n_dim, n_periodic, h,
              r_bond, s_bond, u_bond, length, f);
    return this->active_;
}

bool XlinkEntry::Init(XlinkManagement *parent,
                      int n_dim, int n_periodic,
                      double **h, double **r_bond, double **s_bond,
                      double **u_bond, double *length,
                      int type, int bond_1, int bond_2,
                      gsl_rng *r_rng,
                      int direction_state[2]) {
    this->parent_ = parent;
    this->type_ = type;
    this->head_parent_[0] = bond_1;
    this->head_parent_[1] = bond_2;
    this->stage_ = 2;
    this->head_type_[0] = this->head_type_[1] = 0;
    this->direction_state_[0] = direction_state[0];
    this->direction_state_[1] = direction_state[1];

    this->Insert_0_2(n_dim, n_periodic, h, r_bond, s_bond, u_bond, length, r_rng);

    return this->active_;
}

bool XlinkEntry::Init(XlinkManagement *parent,
                      int n_dim, int n_periodic,
                      double **h, double **r_bond,
                      double **s_bond, double **u_bond,
                      double *length,
                      int type,
                      int head_type,
                      int bond_1, 
                      double lambda,
                      gsl_rng *r_rng,
                      nl_list *neighbs,
                      int direction_state[2]) {
    parent_ = parent;
    type_ = type;
    head_parent_[0] = bond_1;
    head_type_[0] = head_type;
    stage_ = 1;
    cross_position_[0] = lambda;// + length[bond_1];
    active_ = 1;
    direction_state_[0] = direction_state[0];
    direction_state_[1] = direction_state[1];
    
    Update_1_2(n_dim, u_bond, length, neighbs);

    return active_;
}
    
void XlinkEntry::Insert_0_2(int n_dim, int n_periodic,
                            double **h, double **r_bond, double **s_bond,
                            double **u_bond, double *length, gsl_rng *r_rng) {
    int bond_1 = this->head_parent_[0];
    int bond_2 = this->head_parent_[1];
    /* Calculate cosine and sine of angle between bonds and account for numerical error */
    double costheta = dot_product(n_dim, u_bond[bond_1], u_bond[bond_2]);
    double sintheta;
    if (fabs(costheta) > 1.0) {
        costheta = SIGN(1.0, costheta);
        sintheta = 0.0;
    }
    else
        sintheta = sqrt(1.0 - costheta*costheta);

    /* Calculate COM separation of spherocylinders */
    double ds[3];
    for (int i = 0; i < n_periodic; ++i) {  /* First handle periodic subspace. */
        ds[i] = s_bond[bond_2][i] - s_bond[bond_1][i];
        ds[i] -= NINT(ds[i]);
    }
    double dr[3] = {0.0, 0.0, 0.0};
    for (int i = 0; i < n_periodic; ++i) {
        for (int j = 0; j < n_periodic; ++j)
            dr[i] += h[i][j] * ds[j];
    }
    for (int i = n_periodic; i < n_dim; ++i)        /* Then handle free subspace. */
        dr[i] = r_bond[bond_2][i] - r_bond[bond_1][i];
    
    double k_eff = (1.0 - parent_->barrier_weight_[type_]) * parent_->k_stretch_[type_];
    double x, y;
    /* If sin(theta) > 0 then can use typical truncated bivariate normal insertion routine */
    if (sintheta > 0.0) {

        double r_min_mag2, r_min[3], lambda, mu;
        min_distance_carrier_lines(n_dim, n_periodic, h,
                                   r_bond[bond_1], s_bond[bond_1], u_bond[bond_1],
                                   r_bond[bond_2], s_bond[bond_2], u_bond[bond_2],
                                   r_min, &r_min_mag2,
                                   &lambda, &mu);

        double scale_factor = sqrt(1.0 / k_eff) / sintheta;
        double a[2] = { (-lambda - 0.5 * length[bond_1]) / scale_factor,
                        (-mu     - 0.5 * length[bond_2]) / scale_factor };
        double b[2] = { (-lambda + 0.5 * length[bond_1]) / scale_factor,
                        (-mu     + 0.5 * length[bond_2]) / scale_factor };

        double u[2];
        this->GenTruncBivariateNormal(a, b, costheta, u, r_rng);

        x = u[0]*scale_factor + lambda;
        y = u[1]*scale_factor + mu;
    }
    else { /* Use derived parallel insertion routine (see notes) */
        double mu0 = dot_product(n_dim, dr, u_bond[bond_1]);
        do {
            x = length[bond_1] * gsl_rng_uniform(r_rng) - 0.5 * length[bond_1];
            y = -mu0 + x + gsl_ran_gaussian_ziggurat(r_rng, 1.0) / sqrt(k_eff);
            
        } while (ABS(y) > 0.5 * length[bond_2]);
        /* Correct for assumption that bonds are parallel and not antiparallel */
        y *= SIGN(1.0, dot_product(n_dim, u_bond[bond_1], u_bond[bond_2]));
    }

    cross_position_[0] = x + 0.5 * length[bond_1];
    cross_position_[1] = y + 0.5 * length[bond_2];

    double f[3];
    CalcForce(n_dim, n_periodic, h,
              r_bond, s_bond, u_bond, length, f);

    SetActive();
}


void XlinkEntry::UpdateXlinkVector(int n_dim, int n_periodic, double **h,
                                   double **r, double **s, double **u,
                                   double *length) {
    double ds[3];
    int bond_1 = this->head_parent_[0];
    int bond_2 = this->head_parent_[1];
    /* Compute center-to-center separation of spherocylinders. */
    for (int i = 0; i < n_periodic; ++i) {      /* First handle periodic subspace. */
        ds[i] = s[bond_2][i] - s[bond_1][i];
        ds[i] -= NINT(ds[i]);
    }
    dr_[0] = dr_[1] = dr_[2] = 0.0;
    for (int i = 0; i < n_periodic; ++i) {
        for (int j = 0; j < n_periodic; ++j)
            dr_[i] += h[i][j] * ds[j];
    }
    for (int i = n_periodic; i < n_dim; ++i)    /* Then handle free subspace. */
        dr_[i] = r[bond_2][i] - r[bond_1][i];

    /* Calculate crosslink vector */
    double lambda = this->cross_position_[0] - 0.5 * length[bond_1];
    double mu = this->cross_position_[1] - 0.5 * length[bond_2];
    for (int i = 0; i < n_dim; ++i) {
        this->r_cross_[i] = dr_[i] - lambda * u[bond_1][i] + mu * u[bond_2][i];
    }

}

double XlinkEntry::CalcForce(int n_dim, int n_periodic, double **h,
                             double **r, double **s, double **u,
                             double *length, double *f) {
    this->UpdateXlinkVector(n_dim, n_periodic, h,
                            r, s, u, length);

    double k = parent_->k_stretch_[type_];
    double factor;
    if (parent_->r_equil_[type_] == 0.0) {
        factor = k;
    }
    else {
        factor = k * (1.0 - parent_->r_equil_[type_] / sqrt(dot_product(n_dim, r_cross_, r_cross_)));
    }

    u_ = 0.0;
    for (int i = 0; i < n_dim; ++i) {
        f[i] = factor * r_cross_[i];
        flink_[i] = factor * r_cross_[i];
        u_ += 0.5 * SQR(f[i]);
    }
    feff_ = factor * sqrt(dot_product(n_dim, r_cross_, r_cross_));
    u_ *= 1.0 / k;
    return u_;
}

//Unbound diffusion step 
bool XlinkEntry::StepUnbound(int n_dim, 
                      double delta, 
                      double **h, gsl_rng *r) {
    // TODO Label all these variables
   
    // Radius, and radius^2  of the system, 
    // TODO need to change this for boundaries
    double conf_rad = 0.5 * h[0][0];// - r_cutoff + 1.0;
    double conf_rad2 = SQR(conf_rad);

    //Expected diffusion length of the crosslink in the solution in delta 
    double sigma_d = sqrt(2.0 * delta * parent_->diffusion_free_[type_]);

    //Distance actually diffused
    double dr[3] = {0.0, 0.0, 0.0};

    //Previous and final location of the crosslink
    double r_prev[3];
    double r_final[3];

    //Update position of the crosslink
    for (int i = 0; i < n_dim; ++i) {
        dr[i] = gsl_ran_gaussian_ziggurat(r, sigma_d);
        r_prev[i] = r_cross_[i];
        r_final[i] = r_cross_[i] + dr[i];
    }

    
    // Reflection algorithm //
    
    //Distance from center of the system squared
    double r_mag2 = dot_product(n_dim, r_final, r_final);
    
    //Iteration number for reflection
    int i_iter = 0;

    while (r_mag2 > conf_rad2) { //Went through boundary, reflect
        i_iter++;
        
        //Direction of proposed move of crosslink
        double u[3]; 
        double factor = 1.0/sqrt(dot_product(n_dim, dr, dr));
        for (int i = 0; i < n_dim; ++i)
            u[i] = factor * dr[i];

        //Dot product of the direction of the proposed move 
        //and the previous position.
        double r_dot_u = dot_product(n_dim, r_prev, u);

        //Magnitude of previous postion squared
        double r_prev_mag2 = dot_product(n_dim, r_prev, r_prev);
        
        //Factor by which the crosslink will move inward towards the center
        double mu0 = -r_dot_u + sqrt(SQR(r_dot_u) - (r_prev_mag2-conf_rad2));

        //New position of the crosslink normalized to the systems radius
        double R_hat[3];
        for (int i = 0; i < n_dim; ++i){
            R_hat[i] = (r_prev[i] + mu0 * u[i])/conf_rad;
        }
        
        //Actual distance moved in the direction of the proposed move.
        double dr_par = dot_product(n_dim, dr, R_hat);

        //Component of the direction of the proposed move and the actual 
        //new normalized position.
        double u_dot_R = dot_product(n_dim, u, R_hat);
        
        // Update postion of the crosslink
        for (int i = 0; i < n_dim; ++i) {
            r_final[i] = r_final[i] - 2.0 * (dr_par - mu0*u_dot_R) * R_hat[i];
            r_prev[i] = R_hat[i] * conf_rad;
            dr[i] = r_final[i] - r_prev[i];
        }
        r_mag2 = dot_product(n_dim, r_final, r_final);

        // Check for final pathological behavior of the xlink. If r_final is NaN
        // then yank to the surface of the sphere at the location of the crosslink
        for (int i = 0; i < n_dim; ++i) {
            if (std::isnan(r_final[i])) {
                r_final[i] = r_cross_[i]/std::sqrt(r_prev_mag2) * (conf_rad - 1e-05);
            }
        }

        // If crosslink is not inside the boundary in 10 iteration move it 
        // to the edge of the boundary. 
        if (i_iter == 10) {
            double factor = conf_rad/sqrt(dot_product(n_dim, r_final, r_final));
            for (int i = 0; i < n_dim; ++i) {
                r_final[i] = factor * r_final[i];
            }
            break;
        }
    }


    // Find the total distance the crosslink has travelled during the sim
    for (int i = 0; i < n_dim; ++i) {
        dr_[i] = r_final[i] - r_cross_[i];
        r_cross_[i] = r_final[i];
        dr_tot_[i] += dr_[i];
    }

    //// DEBUG FIXME XXX XXX
    //// Check to see if we've gotten a bad r_final
    //for (int i = 0; i < n_dim; ++i) {
    //    if (std::isnan(r_cross_[i])) {
    //        std::cout << "Encountered NaN in StepStage0 Step: " << parent_->properties_->i_current_step << std::endl;
    //        exit(1);
    //    }
    //}
    
    return true;
}

// Update the velocity state and return the delta velocity
void XlinkEntry::UpdateVelocityStates(gsl_rng *r,
                                      double delta,
                                      uint8_t head_type,
                                      double fplus,
                                      double fminus) {
    // Update the direction state 
    double pplus = fplus * delta;
    double pminus = fminus * delta;
    double roll = gsl_rng_uniform_pos(r);

    // Minus end directed
    if (direction_state_[head_type] == 0) {
        if (roll < pplus) {
            direction_state_[head_type] = 1;
        }
    }
    // Plus end directed
    else if (direction_state_[head_type] == 1) {
        if (roll < pminus) {
            direction_state_[head_type] = 0;
        }
    }
}

//Bound diffusion step
bool XlinkEntry::StepSingly(int n_dim, double delta, double *length, gsl_rng *r) {
    double diffusion_bound = parent_->diffusion_bound_[type_][head_type_[0]];
    if (diffusion_bound > 0.0) {
        cross_position_[0] += sqrt(2.0 * diffusion_bound * delta) *
            gsl_ran_gaussian_ziggurat(r, 1.0);
    }
    if (parent_->bi_directional_[type_] == 0) {
        double velocity = parent_->velocity_[type_][head_type_[0]];
        cross_position_[0] += delta * velocity;
    } else {
        // Determine the state change (if there is one) to being plus or minus end directed
        UpdateVelocityStates(r,
                             delta,
                             head_type_[0],
                             parent_->sfreq_plus_[type_][head_type_[0]],
                             parent_->sfreq_minus_[type_][head_type_[0]]);
        if (direction_state_ == 0) { // FIXME: This might be a bug, might need direction_state_[head_type_[0]]
            cross_position_[0] -= parent_->svelocity_minus_[type_][head_type_[0]] * delta;
        } else {
            cross_position_[0] += parent_->svelocity_plus_[type_][head_type_[0]] * delta;
        }
    }

    // Always do the end check
    bool remove_0 = false; 
    /* Reposition at end of bond if it has stepped off */
    if (cross_position_[0] > length[head_parent_[0]]) {
        cross_position_[0] = length[head_parent_[0]];
        remove_0 = !parent_->end_pause_[type_][head_type_[0]];
    }
    else if (cross_position_[0] < 0.0) {
        cross_position_[0] = 0.0;
        remove_0 = !parent_->end_pause_[type_][head_type_[0]];
    }

    if (remove_0) {
        SetInactive();
    }

    return active_;
}

//Doubly bound diffusion
bool XlinkEntry::Step(int n_dim,
                      double delta, double **u_bond,
                      double *length, double *f, gsl_rng *r) {
    bool walking   = !((parent_->velocity_[type_][0] == 0.0) &&
                     (parent_->velocity_[type_][1] == 0.0));

    bool diffusing = !((parent_->diffusion_bound_2_[type_][0] == 0.0) &&
                       (parent_->diffusion_bound_2_[type_][1] == 0.0));
    //if (!walking && !diffusing)
        //return active_;
    
    int stall_type = parent_->stall_type_[type_];
    double f_mag_i, f_mag_j;
    int bond_1 = head_parent_[0]; int bond_2 = head_parent_[1];
    double ui_dot_f = dot_product(n_dim, u_bond[bond_1], f);
    double uj_dot_f = dot_product(n_dim, u_bond[bond_2], f);
    double ui_dot_uj = dot_product(n_dim, u_bond[bond_1], u_bond[bond_2]);
    if (stall_type) {
        f_mag_i = ui_dot_f;
        f_mag_j = -uj_dot_f;
    }
    else
        f_mag_i = f_mag_j = 0.0;

    // Original walking type of action
    if (parent_->bi_directional_[type_] == 0) {
        if (walking) {
            double costheta_thres = parent_->velocity_switch_costheta_[type_][head_type_[0]];
            /* If parallel force is along velocity vector,
               then it shouldn't slow down crosslink */
            double k_stretch =  parent_->k_stretch_[type_];
            double velocity_i = parent_->velocity_[type_][head_type_[0]] *
                ((ui_dot_uj > costheta_thres) ?
                 parent_->velocity_p_scale_[type_][head_type_[0]] :
                 parent_->velocity_ap_scale_[type_][head_type_[0]]);
            double elong = sqrt(dot_product(n_dim, r_cross_, r_cross_));
            if (f_mag_i*velocity_i > 0.0)
                f_mag_i = 0.0;
            else {
                if (stall_type == 1)
                    f_mag_i = ABS(f_mag_i);
                else if (stall_type == 2)
                    f_mag_i = k_stretch * elong;
                else if (stall_type == 3)
                    f_mag_i = -ABS(f_mag_i);
            }

            costheta_thres = parent_->velocity_switch_costheta_[type_][head_type_[1]];
            double velocity_j = parent_->velocity_[type_][head_type_[1]] *
                ((ui_dot_uj > costheta_thres) ?
                 parent_->velocity_p_scale_[type_][head_type_[1]] :
                 parent_->velocity_ap_scale_[type_][head_type_[1]]);
            if (f_mag_j*velocity_j > 0.0)
                f_mag_j = 0.0;
            else {
                if (stall_type == 1)
                    f_mag_j = ABS(f_mag_j);
                else if (stall_type == 2)
                    f_mag_j = k_stretch * elong;
                else if (stall_type == 3)
                    f_mag_j = -ABS(f_mag_j);
            }
            
            if (f_mag_i < parent_->f_stall_[type_][head_type_[0]])
                velocity_i *= 1.0 - f_mag_i/parent_->f_stall_[type_][head_type_[0]];
            else
                velocity_i = 0.0;
            if (f_mag_j < parent_->f_stall_[type_][head_type_[1]])
                velocity_j *= 1.0 - f_mag_j/parent_->f_stall_[type_][head_type_[1]];
            else
                velocity_j = 0.0;
            
            cross_position_[0] += delta * velocity_i;
            cross_position_[1] += delta * velocity_j;
        }
    }
    // New stochastic switching
    else {
        // Determine if we are parallel or antiparallel
        double costheta_thres = 0.0;
        /* If parallel force is along velocity vector,
           then it shouldn't slow down crosslink */
        double k_stretch =  parent_->k_stretch_[type_];
        bool parallel = ui_dot_uj > costheta_thres;
        double velocity_i = 0.0;
        double velocity_j = 0.0;
        // Update the states appropriately
        if (parallel) {
            UpdateVelocityStates(r,
                                 delta,
                                 head_type_[0],
                                 parent_->sfreq_plus_p_[type_][head_type_[0]],
                                 parent_->sfreq_minus_p_[type_][head_type_[0]]);
            UpdateVelocityStates(r,
                                 delta,
                                 head_type_[1],
                                 parent_->sfreq_plus_p_[type_][head_type_[1]],
                                 parent_->sfreq_minus_p_[type_][head_type_[1]]);
        } else {
            UpdateVelocityStates(r,
                                 delta,
                                 head_type_[0],
                                 parent_->sfreq_plus_ap_[type_][head_type_[0]],
                                 parent_->sfreq_minus_ap_[type_][head_type_[0]]);
            UpdateVelocityStates(r,
                                 delta,
                                 head_type_[1],
                                 parent_->sfreq_plus_ap_[type_][head_type_[1]],
                                 parent_->sfreq_minus_ap_[type_][head_type_[1]]);
        }
        // Velocities
        // Head0 in the case of parallel motion
        if (direction_state_[head_type_[0]] == 0) {
            velocity_i = -parent_->svelocity_minus_[type_][head_type_[0]];
        } else {
            velocity_i = parent_->svelocity_plus_[type_][head_type_[0]];
        }
        // Head1 in the case of parallel motion
        if (direction_state_[head_type_[1]] == 0) {
            velocity_j = -parent_->svelocity_minus_[type_][head_type_[1]];
        } else {
            velocity_j = parent_->svelocity_plus_[type_][head_type_[1]];
        }

        // Check the stall force stuff
        double elong = sqrt(dot_product(n_dim, r_cross_, r_cross_));
        if (f_mag_i*velocity_i > 0.0)
            f_mag_i = 0.0;
        else {
            if (stall_type == 1)
                f_mag_i = ABS(f_mag_i);
            else if (stall_type == 2)
                f_mag_i = k_stretch * elong;
            else if (stall_type == 3)
                f_mag_i = -ABS(f_mag_i);
        }
        if (f_mag_j*velocity_j > 0.0)
            f_mag_j = 0.0;
        else {
            if (stall_type == 1)
                f_mag_j = ABS(f_mag_j);
            else if (stall_type == 2)
                f_mag_j = k_stretch * elong;
            else if (stall_type == 3)
                f_mag_j = -ABS(f_mag_j);
        }

        if (f_mag_i < parent_->f_stall_[type_][head_type_[0]])
            velocity_i *= 1.0 - f_mag_i/parent_->f_stall_[type_][head_type_[0]];
        else
            velocity_i = 0.0;
        if (f_mag_j < parent_->f_stall_[type_][head_type_[1]])
            velocity_j *= 1.0 - f_mag_j/parent_->f_stall_[type_][head_type_[1]];
        else
            velocity_j = 0.0;

        //if (!parallel) {
        //    std::cout << "Found parallel: " << parallel << std::endl;
        //    std::cout << "u1("
        //        << u_bond[bond_1][0] << ", "
        //        << u_bond[bond_1][1] << ", "
        //        << u_bond[bond_1][2] << ")\n";
        //    std::cout << "u1("
        //        << u_bond[bond_2][0] << ", "
        //        << u_bond[bond_2][1] << ", "
        //        << u_bond[bond_2][2] << ")\n";
        //    std::cout << "Direction states: " << direction_state_[head_type_[0]] << ", " << direction_state_[head_type_[1]] << std::endl;
        //    std::cout << "Velocity_i: " << velocity_i << std::endl;
        //    std::cout << "Velocity_j: " << velocity_j << std::endl;
        //    exit(1);
        //}
        
        cross_position_[0] += delta * velocity_i;
        cross_position_[1] += delta * velocity_j;
    }


    if (parent_->diffusion_bound_2_[type_][head_type_[0]]) {
        cross_position_[0] +=
            sqrt(2.0 * delta * parent_->diffusion_bound_2_[type_][head_type_[0]]) *
            gsl_ran_gaussian_ziggurat(r, 1.0);
        cross_position_[0] +=
            ui_dot_f * parent_->diffusion_bound_2_[type_][head_type_[0]] * delta;
    }
    if (parent_->diffusion_bound_2_[type_][head_type_[1]]) {
        cross_position_[1] +=
            sqrt(2.0 * delta * parent_->diffusion_bound_2_[type_][head_type_[1]]) *
            gsl_ran_gaussian_ziggurat(r, 1.0);
        cross_position_[1] +=
            -uj_dot_f * parent_->diffusion_bound_2_[type_][head_type_[1]] * delta;
    }

    bool remove_0 = false; bool remove_1 = false;
    /* Reposition at end of bond if it's stepped off */
    if (cross_position_[0] > length[bond_1]) {
        cross_position_[0] = length[bond_1];
        remove_0 = !parent_->end_pause_[type_][head_type_[0]];
    }
    else if (cross_position_[0] < 0.0) {
        cross_position_[0] = 0.0;
        remove_0 = !parent_->end_pause_[type_][head_type_[0]];
    }
        
    if (cross_position_[1] > length[bond_2]) {
        cross_position_[1] = length[bond_2];
        remove_1 = !parent_->end_pause_[type_][head_type_[1]];
    }
    else if (cross_position_[1] < 0.0) {
        cross_position_[1] = 0.0;
        remove_1 = !parent_->end_pause_[type_][head_type_[1]];
    }
    
    if (remove_0 || remove_1) {
        #ifdef ENABLE_OPENMP
        // This was the old way of doing things, now use the shared, main lock
        //omp_lock_t lock;
        //omp_init_lock(&lock);
        //omp_set_lock(&lock);
        omp_set_lock(parent_->steplock_);
        #endif
        if (remove_0 == true && remove_1 == true) {
            if (parent_->attachment_model_ == 2) {
                //std::cout << "Tripped double end walking off!\n";
                int head_bond = head_parent_[0];
                for (int i = 0; i < n_dim; ++i) {
                    r_cross_[i] = 0.5 * r_cross_[i] +
                        parent_->properties_->bonds.r_bond[head_bond][i] +
                        parent_->properties_->bonds.u_bond[head_bond][i] *
                        (cross_position_[0] -
                         0.5 * length[head_bond]);
                }
                parent_->PushStage0(this);
            }
            parent_->DeleteStage2(this);
        }
        else {
            if (remove_0) {
                if (parent_->attachment_model_ == 2) {
                    this->Convert_2_1((uint8_t) 0);
                    parent_->PushStage1(this);
                }
                parent_->DeleteStage2(this);
            }
            else {
                if (parent_->attachment_model_ == 2) {
                    this->Convert_2_1((uint8_t) 1);
                    parent_->PushStage1(this);
                }
                parent_->DeleteStage2(this);
            }
        }
        #ifdef ENABLE_OPENMP
        //omp_unset_lock(&lock);
        //omp_destroy_lock(&lock);
        omp_unset_lock(parent_->steplock_);
        #endif

    }

    return active_;
}

void XlinkEntry::SetInactive() {
    this->active_ = false;
}

void XlinkEntry::SetActive() {
    this->active_ = true;
}

bool XlinkEntry::GenTruncBivariateNormal(double *a, double *b,
                                         double rho, double *u,
                                         gsl_rng *r) {
    bool success = false;
    int n_tries = 0;

    double x_minus = MIN(a[0], b[0]);
    double x_plus = MAX(a[0], b[0]);
    double y_minus = MIN(a[1], b[1]);
    double y_plus = MAX(a[1], b[1]);
    do {
        n_tries++;
        gsl_ran_bivariate_gaussian(r, 1.0, 1.0, rho, &(u[0]), &(u[1]));
        success = ( (u[0] > x_minus && u[0] < x_plus) && (u[1] > y_minus && u[1] < y_plus));
    } while (!success);
    
    return success;
}

double XlinkEntry::Stage_1_2_Probability(int n_dim, double *dr, 
                                         double *u_1, double length_1,
                                         double *u_line, double length_2) {
    if (parent_->force_dependent_) {
        return Stage_1_2_ProbabilityFdep(n_dim, dr, u_1, length_1, u_line, length_2);
    } else {
        return Stage_1_2_ProbabilityEdep(n_dim, dr, u_1, length_1, u_line, length_2);
    }
}


/*
 * Calculate probability for Xlink to go from stage 1 to stage 2 for a given neighbor pair of rod. Energy dependent version. 
 *    Input: 
 *        Energy dependent version
 *        n_dim = number of dimensions
 *        dr = Distance between neighbor rods
 *        u_1 = direction of parent rod
 *        length_1 = length of parent of rod
 *        u_line = direction of neighbor rod
 *        length_2 = length of neighbor rod
 *    Output:
 */
double XlinkEntry::Stage_1_2_ProbabilityEdep(int n_dim, double *dr, 
                                             double *u_1, double length_1,
                                             double *u_line, double length_2) {
    double polar_affinity = parent_->PolarAffinityOn(n_dim, type_, u_1, u_line);
    if (polar_affinity > 0.0) {
        double dr_cross_line[3]; //Vector from xlink attached location to center of neighbor rod center.
        for (int i = 0; i < n_dim; ++i) {
            dr_cross_line[i] = dr[i] - (cross_position_[0] - 0.5 * length_1) * u_1[i];
        }
        // Negative distance from closest point on neighbor rod from attatchment to center of neighbor rod.
        double mu0 = -dot_product(n_dim, dr_cross_line, u_line);

        double r_min_mag2 = 0.0;
        for (int i = 0; i < n_dim; ++i) {
            double dri = u_line[i] * mu0 + dr_cross_line[i];
            // Minimum distance squared between attachment site and neighbor rod.
            r_min_mag2 += SQR(dri);
        }
        // If minimum distance is too large no xlinks expected to bind
        if (r_min_mag2 > SQR(parent_->r_cutoff_1_2_[type_]))
            return 0.0;
        // Code for xlinks with no equilibrium length
        if (parent_->r_equil_[type_] == 0.0) {
            double kb = parent_->k_stretch_[type_] * (1.0 - parent_->barrier_weight_[type_]);
            double scale_factor = sqrt(0.5 * kb);
            double lim0 = scale_factor * (-mu0 - 0.5 * length_2);
            double term0 = erf(lim0);
            double lim1 = scale_factor * (-mu0 + 0.5 * length_2);
            double term1 = erf(lim1);
            return sqrt(M_PI_2 / kb) * exp(-0.5*kb*r_min_mag2) * (term1 - term0) *
                polar_affinity;
        }
        // Code for xlinks with some equilibrium length 
        else {
            // Bounds are set opposite ends of the neighbor rod
            double lim0 = -mu0 - 0.5 * length_2;
            double lim1 = -mu0 + 0.5 * length_2;
            double r_min_mag = sqrt(r_min_mag2);
            // Get value of integral from end to first bound site.
            //  Since lookup table function is odd you can simply negate the negative limits.
            double x[2] = {fabs(lim0), r_min_mag};
            double term0 = parent_->n_exp_lookup_[type_].Lookup(x) * ((lim0 < 0) ? -1.0 : 1.0);
            x[0] = fabs(lim1);
            double term1 = parent_->n_exp_lookup_[type_].Lookup(x) * ((lim1 < 0) ? -1.0 : 1.0);
            double n_exp = (term1 - term0) * polar_affinity;
            if (n_exp > 1E-3){
                return n_exp;
            }
            else
                return 0.0;
        }

    }
    else {
        return 0.0;
    }
}

// Force dependent version
double XlinkEntry::Stage_1_2_ProbabilityFdep(int n_dim, double *dr, 
                                             double *u_1, double length_1,
                                             double *u_line, double length_2) {
    double polar_affinity = parent_->PolarAffinityOn(n_dim, type_, u_1, u_line);
    if (polar_affinity > 0.0) {
        double dr_cross_line[3];
        for (int i = 0; i < n_dim; ++i) {
            dr_cross_line[i] = dr[i] - (cross_position_[0] - 0.5 * length_1) * u_1[i];
        }
        double mu0 = -dot_product(n_dim, dr_cross_line, u_line);

        double r_min_mag2 = 0.0;
        for (int i = 0; i < n_dim; ++i) {
            double dri = u_line[i] * mu0 + dr_cross_line[i];
            r_min_mag2 += SQR(dri);
        }
        if (r_min_mag2 > SQR(parent_->r_cutoff_1_2_[type_]))
            return 0.0;
        // if (r_min_mag2 > 9.210340371976182*2 / kb)
        //     return 0.0;

        if (parent_->r_equil_[type_] == 0.0) {
            std::cout << "ERROR:  Zero rest length not implemented for force dependent version\n";
            exit(1);
        }
        else {
            double lim0 = -mu0 - 0.5 * length_2;
            double lim1 = -mu0 + 0.5 * length_2;
            double r_min_mag = sqrt(r_min_mag2);
            double x[2] = {fabs(lim0), r_min_mag};
            double term0 = parent_->n_exp_lookup_[type_].Lookup(x) * ((lim0 < 0) ? -1.0 : 1.0);
            x[0] = fabs(lim1);
            double term1 = parent_->n_exp_lookup_[type_].Lookup(x) * ((lim1 < 0) ? -1.0 : 1.0);
            double n_exp = (term1 - term0) * polar_affinity;
            if (n_exp > 1E-3)
                return n_exp;
            else
                return 0.0;
        }

    }
    else {
        return 0.0;
    }
}

void XlinkEntry::UpdateNeighbs(int n_dim, int n_periodic, int n_bonds, double **h,
                               double **r_bond, double **s_bond, double **u_bond, double *length,
                               double r_cutoff2) {
    n_exp_ = 0.0;
    dr_tot_[0] = dr_tot_[1] = dr_tot_[2] = 0.0;

    if (!IsActive())
        return;
    
    neighbs_.clear();
    double binding_affinity_factor =
        parent_->eps_eff_1_[type_][0] * parent_->on_rate_1_[type_][0] +
        parent_->eps_eff_1_[type_][1] * parent_->on_rate_1_[type_][1];
    if (binding_affinity_factor > 0) {
        for (int i_bond = 0; i_bond < n_bonds; ++i_bond) {
            nl_entry neighb;
            double *s_ = NULL; //FIXME, explodes with PBC
            double mu, r_min[3];
            min_distance_sphere_sphero(n_dim, n_periodic, h,
                                       r_cross_, s_,
                                       r_bond[i_bond], s_bond[i_bond],
                                       u_bond[i_bond], length[i_bond],
                                       r_min, &neighb.r_min_mag2, &mu);
            if (neighb.r_min_mag2 < r_cutoff2) {
                nl_entry neighb;
                neighb.label = i_bond;
                neighb.value = 0.0;
                
                neighbs_.push_back(neighb);
            }
        }
    }
}

void XlinkEntry::Update_0_1(int n_dim, int n_periodic, int n_bonds, double **h,
                            double **r_bond, double **s_bond, double **u_bond, double *length,
                            double delta) {
    n_exp_ = 0.0;
    if (!IsActive())
        return;
    
    // 1.299 = alpha_cap: Scaling factor used to achieve the same total binding
    // rate as with implicit uniform distributions. Has slight dependence on MT
    // length due to the difference in the spatial distribution of motors in 
    // equilibrium with the MT, giving different effective "persite" binding 
    // rates for different lengths. Chose higher value so shorter MTs have 
    // slightly higher binding rate than would be expected in equilibrium. 
    // Spindle formation has shown no dependence in the past.
    double binding_affinity_factor =
        (parent_->eps_eff_1_[type_][0] * parent_->on_rate_1_[type_][0] +
         parent_->eps_eff_1_[type_][1] * parent_->on_rate_1_[type_][1]) * delta * 1.299;
;

    if (dot_product(n_dim, dr_tot_, dr_tot_) > 0.25 * SQR(8.0)) {
        double r_cutoff = 
            parent_->r_cutoff_0_1_[type_] + 8.0;
        double r_cutoff2 = SQR(r_cutoff);
        UpdateNeighbs(n_dim, n_periodic, n_bonds, h,
                      r_bond, s_bond, u_bond, length, r_cutoff2);
    }

    for (nl_list::iterator neighb = neighbs_.begin();
         neighb < neighbs_.end();
         neighb++) {
        int i_bond = neighb->label;

        neighb->value = binding_affinity_factor * 
            Stage_0_1_Probability(n_dim, n_periodic, h,
                                  r_bond[i_bond], s_bond[i_bond],
                                  u_bond[i_bond], length[i_bond]);

        n_exp_ += neighb->value;
    }    
}

void XlinkEntry::Update_1_2(int n_dim, double **u_bond, double *length, nl_list *neighbs) {
    n_exp_ = 0.0;
    if (!IsActive())
        return;

    int i_bond = head_parent_[0];
    neighbs_ = neighbs[i_bond]; // Make local duplicate of neighbor list
    double binding_affinity_factor =
        parent_->eps_eff_2_[type_][!head_type_[0]] * parent_->on_rate_2_[type_][!head_type_[0]];
    if (binding_affinity_factor > 0) {
        for (nl_list::iterator p = neighbs_.begin(); p < neighbs_.end(); p++) {
            p->value = binding_affinity_factor * 
                Stage_1_2_Probability(n_dim, p->dr, u_bond[i_bond], length[i_bond],
                                      u_bond[p->label], length[p->label]);
            n_exp_ += p->value;
        }
    }
}

/* Calculate probability (for one-stage attachment) to go from stage 0 to
   stage 2. This is a static member since an Xlink doesn't even exist until
   it's inserted in one-stage attachment.  */
double XlinkEntry::Stage_0_2_Probability(int n_dim, int n_periodic, double **h,
                                         double beta_k, double r_equil, double r_min_mag2,
                                         double *r_1, double *s_1, double *u_1, double length_1,
                                         double *r_2, double *s_2, double *u_2, double length_2) {
    if (r_equil != 0.0) {
        // error_exit("Crosslink partition function must currently use equilibrium "
        //            "spring length of 0");
    }

    double costheta = dot_product(n_dim, u_1, u_2);
    double sintheta = 0.0;
    /* Handle roundoff errors that give unphysical numbers */
    if ( ABS(costheta) >= 1.0 ) 
        sintheta = 0.0;
    else
        sintheta = sqrt(1.0 - SQR(costheta));

    double u = 0.0;
    /* Rods practically parallel, do the integral for parallel rods else you get a div_by_0 error */
    if (sintheta < 1e-7) {
        /* Calculate separation of 2 particles */
        double ds[3];
        for (int i = 0; i < n_periodic; ++i) {  /* First handle periodic subspace. */
            ds[i] = s_2[i] - s_1[i];
            ds[i] -= NINT(ds[i]);
        }
        double dr[3] = {0.0, 0.0, 0.0};
        for (int i = 0; i < n_periodic; ++i) {
            for (int j = 0; j < n_periodic; ++j)
                dr[i] += h[i][j] * ds[j];
        }
        for (int i = n_periodic; i < n_dim; ++i)        /* Then handle free subspace. */
            dr[i] = r_2[i] - r_1[i];

        /* See notes for what the hell is going on here */
        /* there are more efficient ways to do this, but this is such a rare 
           occasion that it's not worth the programming effort */
               
        double mu = dot_product(n_dim, dr, u_2);
        double scale_factor = sqrt(0.5 * beta_k);

        u = exp(-0.5 * beta_k * SQR(length_1 - mu)) +
            exp(-0.5 * beta_k * SQR(length_2 + mu)) -
            exp(-0.5 * beta_k * SQR(mu)) -
            exp(-0.5 * beta_k * SQR(length_2 - length_1 + mu));
        
        u += sqrt(M_PI) * scale_factor *
            (-mu * erf(scale_factor * (length_1 - mu)) +
             (length_2 + mu) * erf(scale_factor * (length_1 - length_2 - mu)) -
             mu * (erf(scale_factor * mu)) +
             (length_2 + mu) * erf(scale_factor * length_2 + mu) +
             length_1 * erf(scale_factor * (length_2 - length_1 + mu)) -
             length_1 * erf(scale_factor * (-length_1 + mu)));

        u *= 1.0 / beta_k * exp(-0.5 * beta_k * r_min_mag2);
    }
    else {
        /* Find the minimum distance between the two carrier lines */
        double mu, lambda, r_min[3], r_min_mag2;
        min_distance_carrier_lines(n_dim, n_periodic, h, r_1, s_1, u_1, 
                                   r_2, s_2, u_2, r_min, &r_min_mag2, &lambda, &mu);

        double scale_factor = sqrt(beta_k) * sintheta;
        double half_length_1 = 0.5 * length_1;
        double half_length_2 = 0.5 * length_2;
        double delta_a = (-lambda - half_length_1) * scale_factor;
        double delta_b = (-lambda + half_length_1) * scale_factor;
        double gamma_a = (-mu - half_length_2) * scale_factor;
        double gamma_b = (-mu + half_length_2) * scale_factor;
            
        u += BivariateNormalCDF(gamma_b, delta_b, costheta);
        u -= BivariateNormalCDF(gamma_a, delta_b, costheta);
        u -= BivariateNormalCDF(gamma_b, delta_a, costheta);
        u += BivariateNormalCDF(gamma_a, delta_a, costheta);
            
        u *= 2.0 * M_PI / sintheta / beta_k;
        if (r_min_mag2 > 1e-7) 
            u*= exp(-0.5 * beta_k * r_min_mag2);
    }

    if (u < 0.0)
        return 0.0;
    else
        return u;
}

double XlinkEntry::Stage_0_1_Probability(int n_dim, int n_periodic, double **h,
                                         double *r_bond, double *s_bond, double *u_bond,
                                         double length) {
    double *s_1 = NULL; double mu0; double r_min[3]; double r_min_mag2;
    min_distance_sphere_sphero(n_dim, n_periodic, h,
                               r_cross_, s_1,
                               r_bond, s_bond,
                               u_bond, length,
                               r_min, &r_min_mag2, &mu0);
    if (r_min_mag2 < SQR(parent_->r_cutoff_0_1_[type_])) {
        double a = sqrt(SQR(parent_->r_cutoff_0_1_[type_]) - r_min_mag2);
        double mueff = ABS(mu0);

        double max = mueff + a;
        if (max > 0.5 * length)
            max = 0.5 * length;
        else if (max < -0.5 * length)
            max = -0.5 * length;
        
        double min = mueff - a;
        if (min > 0.5 * length)
            min = 0.5 * length;
        else if (min < -0.5 * length)
            min = -0.5 * length;

        double return_val = 
            (max - min) / (4.0/3.0 * M_PI * CUBE(parent_->r_cutoff_0_1_[type_]));

        return return_val;
    }
    else
        return 0.0;
}


#define SMALLISH 1e-4

double XlinkEntry::PartBivariateNormalCDF(double x, double rho) {
    double sum, rho2, x2, diff, diff2, com_num, com_dom;
    int i, n_terms;
    double A[10], B[10];
    double inv_2pi = 0.15915494309;
    double inv_root_2pi = 0.3989422804;
    double abs_x = fabs(x);

    rho2 = SQR(rho);
    diff2 = 1.0 - rho2;
    diff = sqrt(diff2);
    x2 = SQR(x);
    B[0] = inv_2pi * diff * exp(-0.5 * x2 / diff2);
    A[0] = -inv_root_2pi * abs_x * XlinkEntry::NormCDF(-abs_x / diff) + B[0];
    sum = 0.0;

    i = 1;
    do {
        com_num = (2.0 * i - 1);
        com_dom = (2.0 * i + 1) * i;
        B[i] = 0.5 * SQR(com_num) / com_dom * diff2 * B[i - 1];
        A[i] = -0.5 * com_num / com_dom * x2 * A[i - 1] + B[i];
        i++;
        n_terms = i;
    } while(A[i-1] > SMALLISH && i < 10); 

    for (i = 0; i < n_terms; ++i) {
        sum += A[i];
    }
    if (rho > 0) {
        if (x < 0)
            sum = XlinkEntry::NormCDF(x) - sum;
        else
            sum = 0.5 - sum;
    }
    else {
        if (x > 0)
            sum = XlinkEntry::NormCDF(x) - 0.5 + sum;
    }

    return sum;
}

double XlinkEntry::BivariateNormalCDFHermite(double x, double y, double rho) {
    int i;
    double factorial_factor, rho_factor, exp_factor, term, sum;
    double herm_x[30], herm_y[30];
    
    factorial_factor = 1.0;
    rho_factor = rho;
    i = 0;

    sum = 0.0;
    herm_x[0] = herm_y[0] = 1.0;
    herm_x[1] = x;
    herm_y[1] = y;
    exp_factor =  0.5 / M_PI * exp(-0.5*(x*x + y*y));
    do {
        if (i > 0) {
            herm_x[i + 1] = x*herm_x[i] - i*herm_x[i-1];
            herm_y[i + 1] = y*herm_y[i] - i*herm_y[i-1];
        }
            
        term = exp_factor * herm_x[i] * herm_y[i] * rho_factor / factorial_factor;
        sum += term;
        factorial_factor *= (i + 2);
        rho_factor *= rho;
        i++;
        
        if (ABS(term) < SMALLISH)
            if (i%2 != 0 ) {
                break;
            }
    } while (i < 29);
        
    sum += XlinkEntry::NormCDF(x) * XlinkEntry::NormCDF(y);

    return sum;
}

double XlinkEntry::BivariateNormalCDF(double x, double y, double rho) {
    const double rt_half = 0.707106781;

    if (SQR(rho) <= 0.5) {
        return XlinkEntry::BivariateNormalCDFHermite(x, y, rho);
    }
    else {
        double common = 1.0 / sqrt(x*x + y*y -2.0 * rho * x * y);

        double rho_eff = SIGN(1.0, x) * (rho * x - y) * common;
        double rho_effabs = fabs(rho_eff);
        if (rho_effabs > 1.0)
            rho_eff = SIGN(1.0, rho_eff);

        double sum1;
        if (rho_effabs < rt_half) {
            sum1 = XlinkEntry::BivariateNormalCDFHermite(x, 0.0, rho_eff);
        }
        else {
            sum1 = XlinkEntry::PartBivariateNormalCDF(x, rho_eff);
        }

        rho_eff = SIGN(1.0, y) * (rho * y - x) * common;
        rho_effabs = fabs(rho_eff);
        double sum2;
        if (rho_effabs > 1.0)
            rho_eff = SIGN(1.0, rho_eff);
        if (rho_effabs < rt_half) {
            sum2 = XlinkEntry::BivariateNormalCDFHermite(y, 0.0, rho_eff);
        }
        else {
            sum2 = XlinkEntry::PartBivariateNormalCDF(y, rho_eff);
        }
        return sum1 + sum2 - 0.5 * ( (x*y) < 0.0 );
    }
}

#undef SMALLISH

void XlinkEntry::ClearNeighbors() {
    neighbs_.clear();
    n_exp_ = 0.0;
}

void XlinkEntry::Convert_1_2(system_parameters *parameters,
                             system_properties *properties,
                             int other_bond) {
    this->ClearNeighbors();
    this->stage_ = 2;
    
    int from_index, to_index;

    /* Initialize Crosslink */
    if (head_parent_[0] < other_bond) {
        head_parent_[1] = other_bond;
        head_type_[1] = !head_type_[0];
        from_index = 0;
        to_index = 1;
    }
    else {
        head_parent_[1] = head_parent_[0];
        head_parent_[0] = other_bond;
        head_type_[1] = head_type_[0];
        head_type_[0] = !head_type_[0];
        cross_position_[1] = cross_position_[0];
        from_index = 1;
        to_index = 0;
    }
    int from_bond = head_parent_[from_index];
    int to_bond = head_parent_[to_index];

    double r_1[3], s_1[3], dr[3];
    for (int i = 0; i < parameters->n_dim; ++i)
        r_1[i] = properties->bonds.r_bond[from_bond][i] +
            (cross_position_[from_index] - 0.5 * properties->bonds.length[from_bond])*
            properties->bonds.u_bond[from_bond][i];

    periodic_boundary_conditions_single(parameters->n_periodic,
                                        properties->unit_cell.h,
                                        properties->unit_cell.h_inv,
                                        r_1,
                                        s_1);
    double kb = parent_->k_stretch_[type_];
    if (!parent_->force_dependent_) {
        kb = kb * (1.0 - parent_->barrier_weight_[type_]);
    }
    cross_position_[to_index] =
        this->RandomPosition(parameters->n_dim, parameters->n_periodic,
                             properties->unit_cell.h, r_1, s_1,
                             properties->bonds.r_bond[to_bond],
                             properties->bonds.s_bond[to_bond],
                             properties->bonds.u_bond[to_bond],
                             properties->bonds.length[to_bond],
                             kb, 
                             dr,
                             properties->rng.r);
    if (!std::isnan(cross_position_[to_index])) {
        active_ = true;
        
        //Calculate force and more importantly energy for new xlink
        double f[3];
        CalcForce(parameters->n_dim, parameters->n_periodic,
                  properties->unit_cell.h,
                  properties->bonds.r_bond,
                  properties->bonds.s_bond,
                  properties->bonds.u_bond,
                  properties->bonds.length, 
                  f);
    }
    else
        active_ = false;
}


void XlinkEntry::Convert_0_1(system_parameters *parameters,
                             system_properties *properties,
                             int head_type, double neighb_pos) {
    stage_ = 1;
    head_type_[0] = head_type;
    SetActive();

    double pos = 0.0;
    int to_bond = -1;
    //find a bond to attach to from weighted probability distribution
    for (nl_list::iterator neighb = neighbs_.begin(); neighb < neighbs_.end(); ++neighb) {
        pos += neighb->value;
        if (pos > neighb_pos) {
            head_parent_[0] = to_bond = neighb->label;
            break;
        }
    }

    double *s_1 = NULL; // FIXME
    double mu;
    double dr[3]; 
    // Calculate coordinate along rod s.t. the line vector is perpendicular to
    // the separation vector (closest point along carrier line). In this frame,
    // the position of the crosslink should be gaussian distributed
    min_distance_point_carrier_line(parameters->n_dim, parameters->n_periodic,
                                    properties->unit_cell.h, r_cross_, s_1,
                                    properties->bonds.r_bond[to_bond],
                                    properties->bonds.s_bond[to_bond],
                                    properties->bonds.u_bond[to_bond],
                                    properties->bonds.length[to_bond],
                                    dr, &mu);
    double r_min[3];
    double r_min_mag2 = 0.0;
    for (int i = 0; i < parameters->n_dim; ++i) {
        r_min[i] = -mu * properties->bonds.u_bond[to_bond][i] - dr[i];
        r_min_mag2 += SQR(r_min[i]);
    }
    
    //    double sigma_d = sqrt(2.0 * parameters->delta * parent_->diffusion_free_[type_]);
    double a = sqrt(SQR(1.0)-r_min_mag2);
    if (std::isnan(a))
        a = 0.0;
    
    for (int i = 0; i < 100; ++i) { // Find position that lies along rod
        // cross_position_[0] = gsl_ran_gaussian_ziggurat(properties->rng.r, sigma_d) +
        //     mu + 0.5 * properties->bonds.length[to_bond];
        double u = gsl_rng_uniform(properties->rng.r);
        // double xvec[2] = {0.0, sqrt(r_min_mag2)};
        // cross_position_[0] = ((gsl_rng_uniform(properties->rng.r) < 0.5) ? -1.0 : 1.0) *
        //     parent_->n_exp_0_1_lookup_[type_].Invert(0, u, xvec) +
        //     mu + 0.5 * properties->bonds.length[to_bond];
        cross_position_[0] = (u - 0.5) * a + mu + 0.5 * properties->bonds.length[to_bond];
        
        if (cross_position_[0] >= 0 &&
            cross_position_[0] <= properties->bonds.length[to_bond])
            break;
        if (i == 99) {
            cross_position_[0] = -mu + 0.5 * properties->bonds.length[to_bond];
            if (cross_position_[0] < 0)
                cross_position_[0] = 0;
            else if (cross_position_[0] > properties->bonds.length[to_bond])
                cross_position_[0] = properties->bonds.length[to_bond];
        }
    } 
    n_exp_ = 0.0;
}

double XlinkEntry::RandomPosition(int n_dim, int n_periodic,
                                  double **h, double *r_1, double *s_1,
                                  double *r_line, double *s_line, double *u_line,
                                  double length, double kb, double *dr, gsl_rng *r) {

    double mu;
    min_distance_point_carrier_line(n_dim, n_periodic, h, r_1, s_1,
                                    r_line, s_line, u_line,
                                    length, dr, &mu);
    if (parent_->r_equil_[type_] == 0.0) {
        do {
            double pos = gsl_ran_gaussian_ziggurat(r, sqrt(1.0/kb)) + mu + 0.5 * length;
            if (pos >= 0 && pos <= length)
                return pos;
        } while (1);
    }
    else {
        double y02 = 0.0;
        for (int i = 0; i < n_dim; ++i) {
            double dri = u_line[i] * mu + dr[i];
            y02 += SQR(dri);
        }
        int i_trial = 0;
        do {
            i_trial++;
            double u = gsl_rng_uniform(r);
            double xvec[2] = {0.0, sqrt(y02)};
            double pos = ((gsl_rng_uniform(r) < 0.5) ? -1.0 : 1.0) *
                parent_->n_exp_lookup_[type_].Invert(0, u, xvec) + mu + 0.5 * length;
            if (pos >= 0 && pos <= length)
                return pos;
        } while(i_trial < 100);
        return nan("");
    }
}

void XlinkEntry::Convert_2_1(gsl_rng *r) {

    stage_ = 1;
    if (gsl_rng_uniform(r) > 0.5) {
        head_parent_[0] = head_parent_[1];
        head_type_[0] = head_type_[1];
        cross_position_[0] = cross_position_[1];
    }
}

void XlinkEntry::Convert_2_1(uint8_t head_to_detach) {
    stage_ = 1;
    if (head_type_[0] == head_to_detach) {
        head_parent_[0] = head_parent_[1];
        head_type_[0] = head_type_[1];
        cross_position_[0] = cross_position_[1];
    }
}

float* XlinkEntry::GetColor() {
    return parent_->color_[type_];
}

double XlinkEntry::GetDiameter() {
    return 1.0;
}

// Check against the boundary conditions for pathological behavior
void XlinkEntry::CheckBoundary(int ndim,
                               double **h) {
    double conf_rad = 0.5*h[0][0];
    double conf_rad2 = SQR(conf_rad);
    double rmag2 = dot_product(ndim, r_cross_, r_cross_);
    if (rmag2 > conf_rad2) {
        //std::cout << "tripped boundary check: " << rmag2 << std::endl;
        for (int i = 0; i < ndim; ++i) {
            r_cross_[i] = r_cross_[i]/sqrt(rmag2)*(conf_rad - 1e-05);
        }
    }
}

bool XlinkEntry::CheckEndPosition(double* lengths, int i_head){
    int i_bond = head_parent_[i_head];
    if (lengths[i_bond] < cross_position_[i_head])
        std::cout<<"-- Crosslink  is off plus-end of bond "<<i_bond<<
            " with head ("<<i_head<<").\n";
    else if (0.0 > cross_position_[i_head])
        std::cout<<"-- Crosslink is off minus-end of bond "<<i_bond<<
            " with head ("<<i_head<<").\n";
    else 
        return true;

    return false;
}


