#ifndef _XLINK_ENTRY_H
#define _XLINK_ENTRY_H
#include <stdint.h>
#include <vector>
#include <cmath>
#include <gsl/gsl_rng.h>
#include "neighbors.h"

class XlinkManagement;


#ifndef _PARAMETERS_H
typedef struct system_parameters system_parameters;
#endif
#ifndef _SYSTEM_PROPERTIES
typedef struct system_properties system_properties;
#endif

typedef struct nl_entry nl_entry;
typedef std::vector<nl_entry> nl_list;

class XlinkEntry {
 public:
    XlinkEntry(XlinkManagement *parent);
    XlinkEntry(XlinkManagement *parent,
               int n_dim, int n_periodic,
               double **h, double **r_bond,
               double **s_bond, double **u_bond,
               double *length,
               int type, int bond_1, int bond_2,
               double lambda, double mu,
               int direction_state[2]);

    XlinkEntry(XlinkManagement *parent,
               int n_dim, int n_periodic,
               double **h, double **r_bond,
               double **s_bond, double **u_bond,
               double *length,
               int type, int bond_1, int head_type,
               double lambda,
               gsl_rng *r_rng,
               nl_list *neighbs,
               int direction_state[2]);

    XlinkEntry(XlinkManagement *parent,
               int n_dim, int n_periodic,
               double **h, double **r_bond,
               double **s_bond, double **u_bond,
               double *length,
               int type, int bond_1, int bond_2,
               gsl_rng *r_rng,
               int direction_state[2]);

    bool Init(XlinkManagement *parent,
              int n_dim, int n_periodic,
              double **h, double **r_bond, double **s_bond,
              double **u_bond, double *length,
              int type, int bond_1, int bond_2,
              double lambda, double mu,
              int direction_state[2]);
    bool Init(XlinkManagement *parent,
              int n_dim, int n_periodic,
              double **h, double **r_bond, double **s_bond,
              double **u_bond, double *length,
              int type, int bond_1, int bond_2,
              gsl_rng *r_rng,
              int direction_state[2]);

    bool Init(XlinkManagement *parent,
              int n_dim, int n_periodic,
              double **h, double **r_bond,
              double **s_bond, double **u_bond,
              double *length,
              int type, int head_type,
              int bond_1,
              double lambda,
              gsl_rng *r_rng,
              nl_list *neighbs,
              int direction_state[2]);


 private:
    XlinkManagement *parent_; /* Management class has all of the parameters for
                                each type of xlink stored */
    bool active_; /* Is this link actually present? */

 public:
    uint8_t stage_; /* Stage of xlink. FIXME should only be here until I know
                      that this is unnecessary */

    uint8_t type_; /* Flavor of crosslink, akin to a specific protein */

    uint8_t head_type_[2]; /* Head flavor for specific type (0 or 1). Only first
                             index used in stage 1, both used in stage 2. By
                             convention, index 0 should be the one associated
                             with the "parent" bond, while 1 should be
                             associated with "cross_bond" */
 public:
    int head_parent_[2]; /* Parent bond indices. Could potentially just point to bonds
                           themselves eventually. */
    int direction_state_[2]; // 0 for minus end directed
                          // 1 for plus end directed
    double dr_[3]; /* Center to center separation of parent spherocylinders for stage 2 xlinks */

    double cross_position_[2]; /* Position relative to "minus" end of the
                                 crosslink. Index 0 is for theposition along the
                                 "parent" spherocylinder, while 1 is for the
                                 position along "cross_bond" */
    double r_cross_[3]; /* Head-Head lab space for stage 2 xlinks */
                        /* Lab space position vector for stage 0 xlinks */

    double u_; /* Energy of crosslink (1/2 k * r^2) */
    double flink_[3]; // 3d force on xlink
    double feff_; // Effective force for force dep kinetics

 public:
    double n_exp_; /* Total "raw" probability density (per t) for stage 1 xlinks
                     to attach to *any* nearby spherocylinder. */

    nl_list neighbs_; /* MT Neighbors and the probability to attach from stage 1 to
                        stage 2. */

    double dr_tot_[3];

 private:
    /* Calculate probability for Xlink to go from stage 1 to stage 2 for a given neighbor. */
    double Stage_1_2_Probability(int n_dim,
                                 double *dr, double *u_1, double length_1,
                                 double *u_line, double length_2, double kb, double polar_affinity);

    static inline double NormCDF(double xin) {
        double a[] = {0.254829592, -0.284496736, 1.421413741, -1.453152027, 1.061405429};
        double p  =  0.3275911; double inv_rt_2 = 0.707106781;

        double x = fabs(xin)*inv_rt_2;
        double t = 1.0/(1.0 + p*x);
        double y = 1.0 - (((((a[4]*t + a[3])*t) + a[2])*t + a[1])*t + a[0])*t*exp(-x*x);

        int sign = (0.0 < xin) - (xin < 0.0);
        return 0.5*(1.0 + sign*y);
    };

    static double BivariateNormalCDFHermite(double x, double y, double rho);

    static double BivariateNormalCDF(double x, double y, double rho);

    double RandomPosition(int n_dim, int n_periodic,
                                 double **h, double *r_1, double *s_1,
                                 double *r_line, double *s_line, double *u_line,
                                 double length, double kb, double *dr, gsl_rng *r);

    static double PartBivariateNormalCDF(double x, double rho);

    void CalcForce(double *f);

 public:
    /* Calculate probability (for one-stage attachment) to go from stage 0 to
       stage 2. This is a static member since an Xlink doesn't even exist until
       it's inserted in one-stage attachment.  */
    static double Stage_0_2_Probability(int n_dim, int n_periodic, double **h,
                                        double beta_k, double r_equil, double r_min_mag2,
                                        double *r_1, double *s_1, double *u_1, double length_1,
                                        double *r_2, double *s_2, double *u_2, double length_2);

    /* Calculate crosslink vector for force calculations and position updates */
    void UpdateXlinkVector(int n_dim, int n_periodic, double **h,
                           double **r, double **s, double **u,
                           double *length);

    /* Move xlink forward in time delta t */
    bool Step(int n_dim,
              double delta, double **u,
              double *length, double *f, gsl_rng *r);

    // Unbound xlinks diffuse and stuff
    bool StepUnbound(int n_dim, double delta, double **h, gsl_rng *r);

    /* Move single head forward in time delta t */
    bool StepSingly(int n_dim, double delta, double *length, gsl_rng *r);

    /* Update force and place in f vector */
    double CalcForce(int n_dim, int n_periodic, double **h,
                     double **r, double **s, double **u,
                     double *length, double *f);

    /* Set "active" flag on and potentially set some sanity values */
    void SetActive();

    /* Set "active" flag off and potentially set some sanity values */
    void SetInactive();

    inline bool IsActive() {return active_;};

    inline bool GetType() {return type_;};

    double GetDiameter();

    /* Purge neighbor list */
    void ClearNeighbors();

    void Insert_0_2(int n_dim, int n_periodic,
                    double **h, double **r_bond, double **s_bond,
                    double **u_bond, double *length, gsl_rng *r);

    bool GenTruncBivariateNormal(double *a, double *b,
                                 double rho, double *u,
                                 gsl_rng *r);

    void UpdateNeighbs(int n_dim, int n_periodic, int n_bonds, double **h,
                       double **r_bond, double **s_bond, double **u_bond, double *length,
                       double r_cutoff2);

    void Update_0_2(int n_dim, int n_periodic, int n_bonds,
                    double **r_bond, double **s_bond, double **u_bond, double *length);

    void Update_0_1(int n_dim, int n_periodic, int n_bonds, double **h,
                    double **r_bond, double **s_bond, double **u_bond, double *length,
                    double delta);

    void Update_1_2(int n_dim, double **u_bond, double *length, nl_list *neighbs);

    double Stage_0_1_Probability();

    double Stage_1_2_Probability(int n_dim, double *dr, double *u_1, double length_1,
                                 double *u_line, double length_2);
    double Stage_1_2_ProbabilityEdep(int n_dim, double *dr, double *u_1, double length_1,
                                 double *u_line, double length_2);
    double Stage_1_2_ProbabilityFdep(int n_dim, double *dr, double *u_1, double length_1,
                                 double *u_line, double length_2);

    /* Convert from stage 1 to stage 2 according to gaussian probability
       distribution */
    void Convert_1_2(system_parameters *parameters,
                     system_properties *properties,
                     int other_bond);

    void Convert_2_1(gsl_rng *r);

    void Convert_2_1(uint8_t head_to_detach);

    void Convert_0_1(system_parameters *parameters,
                     system_properties *properties,
                     int head_type, double neighb_pos);

    float* GetColor();

    double Stage_0_1_Probability(int n_dim, int n_periodic, double **h,
                                 double *r_bond, double *s_bond, double *u_bond,
                                 double length);

    // Check that we are inside the boundary condition for this particular setup
    void CheckBoundary(int ndim,
                       double **h);

    //Check to see if a head has fallen off the end of an MT
    bool CheckEndPosition(double* lengths, int i_head);

    // Update the current state of the crosslink based on stochastic switching
    // and do the velocity change (with delta)
    void UpdateVelocityStates(gsl_rng *r,
                              double delta,
                              uint8_t head_type,
                              double fplus,
                              double fminus);
};

typedef std::vector<XlinkEntry> xlink_list;

#endif
